// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2014 Christoph Hellwig.
 */
#include <linux blkdev.h="">
#include <linux kmod.h="">
#include <linux file.h="">
#include <linux jhash.h="">
#include <linux sched.h="">
#include <linux addr.h="" sunrpc="">

#include "pnfs.h"
#include "netns.h"
#include "trace.h"

#define NFSDDBG_FACILITY                NFSDDBG_PNFS

struct nfs4_layout {
	struct list_head		lo_perstate;
	struct nfs4_layout_stateid	*lo_state;
	struct nfsd4_layout_seg		lo_seg;
};

static struct kmem_cache *nfs4_layout_cache;
static struct kmem_cache *nfs4_layout_stateid_cache;

static const struct nfsd4_callback_ops nfsd4_cb_layout_ops;
static const struct lock_manager_operations nfsd4_layouts_lm_ops;

const struct nfsd4_layout_ops *nfsd4_layout_ops[LAYOUT_TYPE_MAX] =  {
#ifdef CONFIG_NFSD_FLEXFILELAYOUT
	[LAYOUT_FLEX_FILES]	= &amp;ff_layout_ops,
#endif
#ifdef CONFIG_NFSD_BLOCKLAYOUT
	[LAYOUT_BLOCK_VOLUME]	= &amp;bl_layout_ops,
#endif
#ifdef CONFIG_NFSD_SCSILAYOUT
	[LAYOUT_SCSI]		= &amp;scsi_layout_ops,
#endif
};

/* pNFS device ID to export fsid mapping */
#define DEVID_HASH_BITS	8
#define DEVID_HASH_SIZE	(1 &lt;&lt; DEVID_HASH_BITS)
#define DEVID_HASH_MASK	(DEVID_HASH_SIZE - 1)
static u64 nfsd_devid_seq = 1;
static struct list_head nfsd_devid_hash[DEVID_HASH_SIZE];
static DEFINE_SPINLOCK(nfsd_devid_lock);

static inline u32 devid_hashfn(u64 idx)
{
	return jhash_2words(idx, idx &gt;&gt; 32, 0) &amp; DEVID_HASH_MASK;
}

static void
nfsd4_alloc_devid_map(const struct svc_fh *fhp)
{
	const struct knfsd_fh *fh = &amp;fhp-&gt;fh_handle;
	size_t fsid_len = key_len(fh-&gt;fh_fsid_type);
	struct nfsd4_deviceid_map *map, *old;
	int i;

	map = kzalloc(sizeof(*map) + fsid_len, GFP_KERNEL);
	if (!map)
		return;

	map-&gt;fsid_type = fh-&gt;fh_fsid_type;
	memcpy(&amp;map-&gt;fsid, fh-&gt;fh_fsid, fsid_len);

	spin_lock(&amp;nfsd_devid_lock);
	if (fhp-&gt;fh_export-&gt;ex_devid_map)
		goto out_unlock;

	for (i = 0; i &lt; DEVID_HASH_SIZE; i++) {
		list_for_each_entry(old, &amp;nfsd_devid_hash[i], hash) {
			if (old-&gt;fsid_type != fh-&gt;fh_fsid_type)
				continue;
			if (memcmp(old-&gt;fsid, fh-&gt;fh_fsid,
					key_len(old-&gt;fsid_type)))
				continue;

			fhp-&gt;fh_export-&gt;ex_devid_map = old;
			goto out_unlock;
		}
	}

	map-&gt;idx = nfsd_devid_seq++;
	list_add_tail_rcu(&amp;map-&gt;hash, &amp;nfsd_devid_hash[devid_hashfn(map-&gt;idx)]);
	fhp-&gt;fh_export-&gt;ex_devid_map = map;
	map = NULL;

out_unlock:
	spin_unlock(&amp;nfsd_devid_lock);
	kfree(map);
}

struct nfsd4_deviceid_map *
nfsd4_find_devid_map(int idx)
{
	struct nfsd4_deviceid_map *map, *ret = NULL;

	rcu_read_lock();
	list_for_each_entry_rcu(map, &amp;nfsd_devid_hash[devid_hashfn(idx)], hash)
		if (map-&gt;idx == idx)
			ret = map;
	rcu_read_unlock();

	return ret;
}

int
nfsd4_set_deviceid(struct nfsd4_deviceid *id, const struct svc_fh *fhp,
		u32 device_generation)
{
	if (!fhp-&gt;fh_export-&gt;ex_devid_map) {
		nfsd4_alloc_devid_map(fhp);
		if (!fhp-&gt;fh_export-&gt;ex_devid_map)
			return -ENOMEM;
	}

	id-&gt;fsid_idx = fhp-&gt;fh_export-&gt;ex_devid_map-&gt;idx;
	id-&gt;generation = device_generation;
	id-&gt;pad = 0;
	return 0;
}

void nfsd4_setup_layout_type(struct svc_export *exp)
{
#if defined(CONFIG_NFSD_BLOCKLAYOUT) || defined(CONFIG_NFSD_SCSILAYOUT)
	struct super_block *sb = exp-&gt;ex_path.mnt-&gt;mnt_sb;
#endif

	if (!(exp-&gt;ex_flags &amp; NFSEXP_PNFS))
		return;

#ifdef CONFIG_NFSD_FLEXFILELAYOUT
	exp-&gt;ex_layout_types |= 1 &lt;&lt; LAYOUT_FLEX_FILES;
#endif
#ifdef CONFIG_NFSD_BLOCKLAYOUT
	if (sb-&gt;s_export_op-&gt;get_uuid &amp;&amp;
	    sb-&gt;s_export_op-&gt;map_blocks &amp;&amp;
	    sb-&gt;s_export_op-&gt;commit_blocks)
		exp-&gt;ex_layout_types |= 1 &lt;&lt; LAYOUT_BLOCK_VOLUME;
#endif
#ifdef CONFIG_NFSD_SCSILAYOUT
	if (sb-&gt;s_export_op-&gt;map_blocks &amp;&amp;
	    sb-&gt;s_export_op-&gt;commit_blocks &amp;&amp;
	    sb-&gt;s_bdev &amp;&amp;
	    sb-&gt;s_bdev-&gt;bd_disk-&gt;fops-&gt;pr_ops &amp;&amp;
	    sb-&gt;s_bdev-&gt;bd_disk-&gt;fops-&gt;get_unique_id)
		exp-&gt;ex_layout_types |= 1 &lt;&lt; LAYOUT_SCSI;
#endif
}

static void
nfsd4_free_layout_stateid(struct nfs4_stid *stid)
{
	struct nfs4_layout_stateid *ls = layoutstateid(stid);
	struct nfs4_client *clp = ls-&gt;ls_stid.sc_client;
	struct nfs4_file *fp = ls-&gt;ls_stid.sc_file;

	trace_nfsd_layoutstate_free(&amp;ls-&gt;ls_stid.sc_stateid);

	spin_lock(&amp;clp-&gt;cl_lock);
	list_del_init(&amp;ls-&gt;ls_perclnt);
	spin_unlock(&amp;clp-&gt;cl_lock);

	spin_lock(&amp;fp-&gt;fi_lock);
	list_del_init(&amp;ls-&gt;ls_perfile);
	spin_unlock(&amp;fp-&gt;fi_lock);

	if (!nfsd4_layout_ops[ls-&gt;ls_layout_type]-&gt;disable_recalls)
		vfs_setlease(ls-&gt;ls_file-&gt;nf_file, F_UNLCK, NULL, (void **)&amp;ls);
	nfsd_file_put(ls-&gt;ls_file);

	if (ls-&gt;ls_recalled)
		atomic_dec(&amp;ls-&gt;ls_stid.sc_file-&gt;fi_lo_recalls);

	kmem_cache_free(nfs4_layout_stateid_cache, ls);
}

static int
nfsd4_layout_setlease(struct nfs4_layout_stateid *ls)
{
	struct file_lock *fl;
	int status;

	if (nfsd4_layout_ops[ls-&gt;ls_layout_type]-&gt;disable_recalls)
		return 0;

	fl = locks_alloc_lock();
	if (!fl)
		return -ENOMEM;
	locks_init_lock(fl);
	fl-&gt;fl_lmops = &amp;nfsd4_layouts_lm_ops;
	fl-&gt;fl_flags = FL_LAYOUT;
	fl-&gt;fl_type = F_RDLCK;
	fl-&gt;fl_end = OFFSET_MAX;
	fl-&gt;fl_owner = ls;
	fl-&gt;fl_pid = current-&gt;tgid;
	fl-&gt;fl_file = ls-&gt;ls_file-&gt;nf_file;

	status = vfs_setlease(fl-&gt;fl_file, fl-&gt;fl_type, &amp;fl, NULL);
	if (status) {
		locks_free_lock(fl);
		return status;
	}
	BUG_ON(fl != NULL);
	return 0;
}

static struct nfs4_layout_stateid *
nfsd4_alloc_layout_stateid(struct nfsd4_compound_state *cstate,
		struct nfs4_stid *parent, u32 layout_type)
{
	struct nfs4_client *clp = cstate-&gt;clp;
	struct nfs4_file *fp = parent-&gt;sc_file;
	struct nfs4_layout_stateid *ls;
	struct nfs4_stid *stp;

	stp = nfs4_alloc_stid(cstate-&gt;clp, nfs4_layout_stateid_cache,
					nfsd4_free_layout_stateid);
	if (!stp)
		return NULL;

	get_nfs4_file(fp);
	stp-&gt;sc_file = fp;

	ls = layoutstateid(stp);
	INIT_LIST_HEAD(&amp;ls-&gt;ls_perclnt);
	INIT_LIST_HEAD(&amp;ls-&gt;ls_perfile);
	spin_lock_init(&amp;ls-&gt;ls_lock);
	INIT_LIST_HEAD(&amp;ls-&gt;ls_layouts);
	mutex_init(&amp;ls-&gt;ls_mutex);
	ls-&gt;ls_layout_type = layout_type;
	nfsd4_init_cb(&amp;ls-&gt;ls_recall, clp, &amp;nfsd4_cb_layout_ops,
			NFSPROC4_CLNT_CB_LAYOUT);

	if (parent-&gt;sc_type == NFS4_DELEG_STID)
		ls-&gt;ls_file = nfsd_file_get(fp-&gt;fi_deleg_file);
	else
		ls-&gt;ls_file = find_any_file(fp);
	BUG_ON(!ls-&gt;ls_file);

	if (nfsd4_layout_setlease(ls)) {
		nfsd_file_put(ls-&gt;ls_file);
		put_nfs4_file(fp);
		kmem_cache_free(nfs4_layout_stateid_cache, ls);
		return NULL;
	}

	spin_lock(&amp;clp-&gt;cl_lock);
	stp-&gt;sc_type = NFS4_LAYOUT_STID;
	list_add(&amp;ls-&gt;ls_perclnt, &amp;clp-&gt;cl_lo_states);
	spin_unlock(&amp;clp-&gt;cl_lock);

	spin_lock(&amp;fp-&gt;fi_lock);
	list_add(&amp;ls-&gt;ls_perfile, &amp;fp-&gt;fi_lo_states);
	spin_unlock(&amp;fp-&gt;fi_lock);

	trace_nfsd_layoutstate_alloc(&amp;ls-&gt;ls_stid.sc_stateid);
	return ls;
}

__be32
nfsd4_preprocess_layout_stateid(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate, stateid_t *stateid,
		bool create, u32 layout_type, struct nfs4_layout_stateid **lsp)
{
	struct nfs4_layout_stateid *ls;
	struct nfs4_stid *stid;
	unsigned char typemask = NFS4_LAYOUT_STID;
	__be32 status;

	if (create)
		typemask |= (NFS4_OPEN_STID | NFS4_LOCK_STID | NFS4_DELEG_STID);

	status = nfsd4_lookup_stateid(cstate, stateid, typemask, &amp;stid,
			net_generic(SVC_NET(rqstp), nfsd_net_id));
	if (status)
		goto out;

	if (!fh_match(&amp;cstate-&gt;current_fh.fh_handle,
		      &amp;stid-&gt;sc_file-&gt;fi_fhandle)) {
		status = nfserr_bad_stateid;
		goto out_put_stid;
	}

	if (stid-&gt;sc_type != NFS4_LAYOUT_STID) {
		ls = nfsd4_alloc_layout_stateid(cstate, stid, layout_type);
		nfs4_put_stid(stid);

		status = nfserr_jukebox;
		if (!ls)
			goto out;
		mutex_lock(&amp;ls-&gt;ls_mutex);
	} else {
		ls = container_of(stid, struct nfs4_layout_stateid, ls_stid);

		status = nfserr_bad_stateid;
		mutex_lock(&amp;ls-&gt;ls_mutex);
		if (nfsd4_stateid_generation_after(stateid, &amp;stid-&gt;sc_stateid))
			goto out_unlock_stid;
		if (layout_type != ls-&gt;ls_layout_type)
			goto out_unlock_stid;
	}

	*lsp = ls;
	return 0;

out_unlock_stid:
	mutex_unlock(&amp;ls-&gt;ls_mutex);
out_put_stid:
	nfs4_put_stid(stid);
out:
	return status;
}

static void
nfsd4_recall_file_layout(struct nfs4_layout_stateid *ls)
{
	spin_lock(&amp;ls-&gt;ls_lock);
	if (ls-&gt;ls_recalled)
		goto out_unlock;

	ls-&gt;ls_recalled = true;
	atomic_inc(&amp;ls-&gt;ls_stid.sc_file-&gt;fi_lo_recalls);
	if (list_empty(&amp;ls-&gt;ls_layouts))
		goto out_unlock;

	trace_nfsd_layout_recall(&amp;ls-&gt;ls_stid.sc_stateid);

	refcount_inc(&amp;ls-&gt;ls_stid.sc_count);
	nfsd4_run_cb(&amp;ls-&gt;ls_recall);

out_unlock:
	spin_unlock(&amp;ls-&gt;ls_lock);
}

static inline u64
layout_end(struct nfsd4_layout_seg *seg)
{
	u64 end = seg-&gt;offset + seg-&gt;length;
	return end &gt;= seg-&gt;offset ? end : NFS4_MAX_UINT64;
}

static void
layout_update_len(struct nfsd4_layout_seg *lo, u64 end)
{
	if (end == NFS4_MAX_UINT64)
		lo-&gt;length = NFS4_MAX_UINT64;
	else
		lo-&gt;length = end - lo-&gt;offset;
}

static bool
layouts_overlapping(struct nfs4_layout *lo, struct nfsd4_layout_seg *s)
{
	if (s-&gt;iomode != IOMODE_ANY &amp;&amp; s-&gt;iomode != lo-&gt;lo_seg.iomode)
		return false;
	if (layout_end(&amp;lo-&gt;lo_seg) &lt;= s-&gt;offset)
		return false;
	if (layout_end(s) &lt;= lo-&gt;lo_seg.offset)
		return false;
	return true;
}

static bool
layouts_try_merge(struct nfsd4_layout_seg *lo, struct nfsd4_layout_seg *new)
{
	if (lo-&gt;iomode != new-&gt;iomode)
		return false;
	if (layout_end(new) &lt; lo-&gt;offset)
		return false;
	if (layout_end(lo) &lt; new-&gt;offset)
		return false;

	lo-&gt;offset = min(lo-&gt;offset, new-&gt;offset);
	layout_update_len(lo, max(layout_end(lo), layout_end(new)));
	return true;
}

static __be32
nfsd4_recall_conflict(struct nfs4_layout_stateid *ls)
{
	struct nfs4_file *fp = ls-&gt;ls_stid.sc_file;
	struct nfs4_layout_stateid *l, *n;
	__be32 nfserr = nfs_ok;

	assert_spin_locked(&amp;fp-&gt;fi_lock);

	list_for_each_entry_safe(l, n, &amp;fp-&gt;fi_lo_states, ls_perfile) {
		if (l != ls) {
			nfsd4_recall_file_layout(l);
			nfserr = nfserr_recallconflict;
		}
	}

	return nfserr;
}

__be32
nfsd4_insert_layout(struct nfsd4_layoutget *lgp, struct nfs4_layout_stateid *ls)
{
	struct nfsd4_layout_seg *seg = &amp;lgp-&gt;lg_seg;
	struct nfs4_file *fp = ls-&gt;ls_stid.sc_file;
	struct nfs4_layout *lp, *new = NULL;
	__be32 nfserr;

	spin_lock(&amp;fp-&gt;fi_lock);
	nfserr = nfsd4_recall_conflict(ls);
	if (nfserr)
		goto out;
	spin_lock(&amp;ls-&gt;ls_lock);
	list_for_each_entry(lp, &amp;ls-&gt;ls_layouts, lo_perstate) {
		if (layouts_try_merge(&amp;lp-&gt;lo_seg, seg))
			goto done;
	}
	spin_unlock(&amp;ls-&gt;ls_lock);
	spin_unlock(&amp;fp-&gt;fi_lock);

	new = kmem_cache_alloc(nfs4_layout_cache, GFP_KERNEL);
	if (!new)
		return nfserr_jukebox;
	memcpy(&amp;new-&gt;lo_seg, seg, sizeof(new-&gt;lo_seg));
	new-&gt;lo_state = ls;

	spin_lock(&amp;fp-&gt;fi_lock);
	nfserr = nfsd4_recall_conflict(ls);
	if (nfserr)
		goto out;
	spin_lock(&amp;ls-&gt;ls_lock);
	list_for_each_entry(lp, &amp;ls-&gt;ls_layouts, lo_perstate) {
		if (layouts_try_merge(&amp;lp-&gt;lo_seg, seg))
			goto done;
	}

	refcount_inc(&amp;ls-&gt;ls_stid.sc_count);
	list_add_tail(&amp;new-&gt;lo_perstate, &amp;ls-&gt;ls_layouts);
	new = NULL;
done:
	nfs4_inc_and_copy_stateid(&amp;lgp-&gt;lg_sid, &amp;ls-&gt;ls_stid);
	spin_unlock(&amp;ls-&gt;ls_lock);
out:
	spin_unlock(&amp;fp-&gt;fi_lock);
	if (new)
		kmem_cache_free(nfs4_layout_cache, new);
	return nfserr;
}

static void
nfsd4_free_layouts(struct list_head *reaplist)
{
	while (!list_empty(reaplist)) {
		struct nfs4_layout *lp = list_first_entry(reaplist,
				struct nfs4_layout, lo_perstate);

		list_del(&amp;lp-&gt;lo_perstate);
		nfs4_put_stid(&amp;lp-&gt;lo_state-&gt;ls_stid);
		kmem_cache_free(nfs4_layout_cache, lp);
	}
}

static void
nfsd4_return_file_layout(struct nfs4_layout *lp, struct nfsd4_layout_seg *seg,
		struct list_head *reaplist)
{
	struct nfsd4_layout_seg *lo = &amp;lp-&gt;lo_seg;
	u64 end = layout_end(lo);

	if (seg-&gt;offset &lt;= lo-&gt;offset) {
		if (layout_end(seg) &gt;= end) {
			list_move_tail(&amp;lp-&gt;lo_perstate, reaplist);
			return;
		}
		lo-&gt;offset = layout_end(seg);
	} else {
		/* retain the whole layout segment on a split. */
		if (layout_end(seg) &lt; end) {
			dprintk("%s: split not supported\n", __func__);
			return;
		}
		end = seg-&gt;offset;
	}

	layout_update_len(lo, end);
}

__be32
nfsd4_return_file_layouts(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate,
		struct nfsd4_layoutreturn *lrp)
{
	struct nfs4_layout_stateid *ls;
	struct nfs4_layout *lp, *n;
	LIST_HEAD(reaplist);
	__be32 nfserr;
	int found = 0;

	nfserr = nfsd4_preprocess_layout_stateid(rqstp, cstate, &amp;lrp-&gt;lr_sid,
						false, lrp-&gt;lr_layout_type,
						&amp;ls);
	if (nfserr) {
		trace_nfsd_layout_return_lookup_fail(&amp;lrp-&gt;lr_sid);
		return nfserr;
	}

	spin_lock(&amp;ls-&gt;ls_lock);
	list_for_each_entry_safe(lp, n, &amp;ls-&gt;ls_layouts, lo_perstate) {
		if (layouts_overlapping(lp, &amp;lrp-&gt;lr_seg)) {
			nfsd4_return_file_layout(lp, &amp;lrp-&gt;lr_seg, &amp;reaplist);
			found++;
		}
	}
	if (!list_empty(&amp;ls-&gt;ls_layouts)) {
		if (found)
			nfs4_inc_and_copy_stateid(&amp;lrp-&gt;lr_sid, &amp;ls-&gt;ls_stid);
		lrp-&gt;lrs_present = 1;
	} else {
		trace_nfsd_layoutstate_unhash(&amp;ls-&gt;ls_stid.sc_stateid);
		nfs4_unhash_stid(&amp;ls-&gt;ls_stid);
		lrp-&gt;lrs_present = 0;
	}
	spin_unlock(&amp;ls-&gt;ls_lock);

	mutex_unlock(&amp;ls-&gt;ls_mutex);
	nfs4_put_stid(&amp;ls-&gt;ls_stid);
	nfsd4_free_layouts(&amp;reaplist);
	return nfs_ok;
}

__be32
nfsd4_return_client_layouts(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate,
		struct nfsd4_layoutreturn *lrp)
{
	struct nfs4_layout_stateid *ls, *n;
	struct nfs4_client *clp = cstate-&gt;clp;
	struct nfs4_layout *lp, *t;
	LIST_HEAD(reaplist);

	lrp-&gt;lrs_present = 0;

	spin_lock(&amp;clp-&gt;cl_lock);
	list_for_each_entry_safe(ls, n, &amp;clp-&gt;cl_lo_states, ls_perclnt) {
		if (ls-&gt;ls_layout_type != lrp-&gt;lr_layout_type)
			continue;

		if (lrp-&gt;lr_return_type == RETURN_FSID &amp;&amp;
		    !fh_fsid_match(&amp;ls-&gt;ls_stid.sc_file-&gt;fi_fhandle,
				   &amp;cstate-&gt;current_fh.fh_handle))
			continue;

		spin_lock(&amp;ls-&gt;ls_lock);
		list_for_each_entry_safe(lp, t, &amp;ls-&gt;ls_layouts, lo_perstate) {
			if (lrp-&gt;lr_seg.iomode == IOMODE_ANY ||
			    lrp-&gt;lr_seg.iomode == lp-&gt;lo_seg.iomode)
				list_move_tail(&amp;lp-&gt;lo_perstate, &amp;reaplist);
		}
		spin_unlock(&amp;ls-&gt;ls_lock);
	}
	spin_unlock(&amp;clp-&gt;cl_lock);

	nfsd4_free_layouts(&amp;reaplist);
	return 0;
}

static void
nfsd4_return_all_layouts(struct nfs4_layout_stateid *ls,
		struct list_head *reaplist)
{
	spin_lock(&amp;ls-&gt;ls_lock);
	list_splice_init(&amp;ls-&gt;ls_layouts, reaplist);
	spin_unlock(&amp;ls-&gt;ls_lock);
}

void
nfsd4_return_all_client_layouts(struct nfs4_client *clp)
{
	struct nfs4_layout_stateid *ls, *n;
	LIST_HEAD(reaplist);

	spin_lock(&amp;clp-&gt;cl_lock);
	list_for_each_entry_safe(ls, n, &amp;clp-&gt;cl_lo_states, ls_perclnt)
		nfsd4_return_all_layouts(ls, &amp;reaplist);
	spin_unlock(&amp;clp-&gt;cl_lock);

	nfsd4_free_layouts(&amp;reaplist);
}

void
nfsd4_return_all_file_layouts(struct nfs4_client *clp, struct nfs4_file *fp)
{
	struct nfs4_layout_stateid *ls, *n;
	LIST_HEAD(reaplist);

	spin_lock(&amp;fp-&gt;fi_lock);
	list_for_each_entry_safe(ls, n, &amp;fp-&gt;fi_lo_states, ls_perfile) {
		if (ls-&gt;ls_stid.sc_client == clp)
			nfsd4_return_all_layouts(ls, &amp;reaplist);
	}
	spin_unlock(&amp;fp-&gt;fi_lock);

	nfsd4_free_layouts(&amp;reaplist);
}

static void
nfsd4_cb_layout_fail(struct nfs4_layout_stateid *ls)
{
	struct nfs4_client *clp = ls-&gt;ls_stid.sc_client;
	char addr_str[INET6_ADDRSTRLEN];
	static char const nfsd_recall_failed[] = "/sbin/nfsd-recall-failed";
	static char *envp[] = {
		"HOME=/",
		"TERM=linux",
		"PATH=/sbin:/usr/sbin:/bin:/usr/bin",
		NULL
	};
	char *argv[8];
	int error;

	rpc_ntop((struct sockaddr *)&amp;clp-&gt;cl_addr, addr_str, sizeof(addr_str));

	printk(KERN_WARNING
		"nfsd: client %s failed to respond to layout recall. "
		"  Fencing..\n", addr_str);

	argv[0] = (char *)nfsd_recall_failed;
	argv[1] = addr_str;
	argv[2] = ls-&gt;ls_file-&gt;nf_file-&gt;f_path.mnt-&gt;mnt_sb-&gt;s_id;
	argv[3] = NULL;

	error = call_usermodehelper(nfsd_recall_failed, argv, envp,
				    UMH_WAIT_PROC);
	if (error) {
		printk(KERN_ERR "nfsd: fence failed for client %s: %d!\n",
			addr_str, error);
	}
}

static void
nfsd4_cb_layout_prepare(struct nfsd4_callback *cb)
{
	struct nfs4_layout_stateid *ls =
		container_of(cb, struct nfs4_layout_stateid, ls_recall);

	mutex_lock(&amp;ls-&gt;ls_mutex);
	nfs4_inc_and_copy_stateid(&amp;ls-&gt;ls_recall_sid, &amp;ls-&gt;ls_stid);
	mutex_unlock(&amp;ls-&gt;ls_mutex);
}

static int
nfsd4_cb_layout_done(struct nfsd4_callback *cb, struct rpc_task *task)
{
	struct nfs4_layout_stateid *ls =
		container_of(cb, struct nfs4_layout_stateid, ls_recall);
	struct nfsd_net *nn;
	ktime_t now, cutoff;
	const struct nfsd4_layout_ops *ops;

	trace_nfsd_cb_layout_done(&amp;ls-&gt;ls_stid.sc_stateid, task);
	switch (task-&gt;tk_status) {
	case 0:
	case -NFS4ERR_DELAY:
		/*
		 * Anything left? If not, then call it done. Note that we don't
		 * take the spinlock since this is an optimization and nothing
		 * should get added until the cb counter goes to zero.
		 */
		if (list_empty(&amp;ls-&gt;ls_layouts))
			return 1;

		/* Poll the client until it's done with the layout */
		now = ktime_get();
		nn = net_generic(ls-&gt;ls_stid.sc_client-&gt;net, nfsd_net_id);

		/* Client gets 2 lease periods to return it */
		cutoff = ktime_add_ns(task-&gt;tk_start,
					 (u64)nn-&gt;nfsd4_lease * NSEC_PER_SEC * 2);

		if (ktime_before(now, cutoff)) {
			rpc_delay(task, HZ/100); /* 10 mili-seconds */
			return 0;
		}
		fallthrough;
	default:
		/*
		 * Unknown error or non-responding client, we'll need to fence.
		 */
		trace_nfsd_layout_recall_fail(&amp;ls-&gt;ls_stid.sc_stateid);

		ops = nfsd4_layout_ops[ls-&gt;ls_layout_type];
		if (ops-&gt;fence_client)
			ops-&gt;fence_client(ls);
		else
			nfsd4_cb_layout_fail(ls);
		return 1;
	case -NFS4ERR_NOMATCHING_LAYOUT:
		trace_nfsd_layout_recall_done(&amp;ls-&gt;ls_stid.sc_stateid);
		task-&gt;tk_status = 0;
		return 1;
	}
}

static void
nfsd4_cb_layout_release(struct nfsd4_callback *cb)
{
	struct nfs4_layout_stateid *ls =
		container_of(cb, struct nfs4_layout_stateid, ls_recall);
	LIST_HEAD(reaplist);

	trace_nfsd_layout_recall_release(&amp;ls-&gt;ls_stid.sc_stateid);

	nfsd4_return_all_layouts(ls, &amp;reaplist);
	nfsd4_free_layouts(&amp;reaplist);
	nfs4_put_stid(&amp;ls-&gt;ls_stid);
}

static const struct nfsd4_callback_ops nfsd4_cb_layout_ops = {
	.prepare	= nfsd4_cb_layout_prepare,
	.done		= nfsd4_cb_layout_done,
	.release	= nfsd4_cb_layout_release,
};

static bool
nfsd4_layout_lm_break(struct file_lock *fl)
{
	/*
	 * We don't want the locks code to timeout the lease for us;
	 * we'll remove it ourself if a layout isn't returned
	 * in time:
	 */
	fl-&gt;fl_break_time = 0;
	nfsd4_recall_file_layout(fl-&gt;fl_owner);
	return false;
}

static int
nfsd4_layout_lm_change(struct file_lock *onlist, int arg,
		struct list_head *dispose)
{
	BUG_ON(!(arg &amp; F_UNLCK));
	return lease_modify(onlist, arg, dispose);
}

static const struct lock_manager_operations nfsd4_layouts_lm_ops = {
	.lm_break	= nfsd4_layout_lm_break,
	.lm_change	= nfsd4_layout_lm_change,
};

int
nfsd4_init_pnfs(void)
{
	int i;

	for (i = 0; i &lt; DEVID_HASH_SIZE; i++)
		INIT_LIST_HEAD(&amp;nfsd_devid_hash[i]);

	nfs4_layout_cache = kmem_cache_create("nfs4_layout",
			sizeof(struct nfs4_layout), 0, 0, NULL);
	if (!nfs4_layout_cache)
		return -ENOMEM;

	nfs4_layout_stateid_cache = kmem_cache_create("nfs4_layout_stateid",
			sizeof(struct nfs4_layout_stateid), 0, 0, NULL);
	if (!nfs4_layout_stateid_cache) {
		kmem_cache_destroy(nfs4_layout_cache);
		return -ENOMEM;
	}
	return 0;
}

void
nfsd4_exit_pnfs(void)
{
	int i;

	kmem_cache_destroy(nfs4_layout_cache);
	kmem_cache_destroy(nfs4_layout_stateid_cache);

	for (i = 0; i &lt; DEVID_HASH_SIZE; i++) {
		struct nfsd4_deviceid_map *map, *n;

		list_for_each_entry_safe(map, n, &amp;nfsd_devid_hash[i], hash)
			kfree(map);
	}
}
</linux></linux></linux></linux></linux></linux>