import anthropic
import time
from concurrent.futures import ThreadPoolExecutor
from tqdm import tqdm
import threading
import datetime
import os
import json
from openai import OpenAI
from mistralai.client import MistralClient



def load_jsonl(file_path):
    data = []
    with open(file_path, 'r') as f:
        for line in f:
            data.append(json.loads(line))
    return data


class Timer(object):
    def __init__(self):
        self.__start = time.time()

    def start(self):
        self.__start = time.time()

    def get_time(self, restart=True, format=False):
        end = time.time()
        span = end - self.__start
        if restart:
            self.__start = end
        if format:
            return self.format(span)
        else:
            return span

    def format(self, seconds):
        return datetime.timedelta(seconds=int(seconds))

    def print(self, name):
        print(name, self.get_time())


class AutoAPIModel:
    def __init__(self, params, api_key=None):
        self.api_key = api_key
        if 'engine' not in params:
            params['engine'] = 'gpt-3.5-turbo'
        if 'temperature' not in params:
            params['temperature'] = 0.5
        if 'max_tokens' not in params:
            params['max_tokens'] = 512
        if 'base_url' not in params:
            params['base_url'] = None
        # if 'top_logprobs' not in params:
        #     params['top_logprobs'] = 5
        if 'attempt_num' not in params:
            params['attempt_num'] = 20      
        if 'do_sample' not in params:
            params['do_sample'] = False
        if 'chat_system_instruction' not in params:
            params['chat_system_instruction'] = None

        self.params = params
        if api_key:
            if 'gpt' in params['engine']:
                self.client = OpenAI(base_url=params['base_url'], api_key=api_key)
            elif 'deepseek' in params['engine']:
                self.client = OpenAI(base_url=params['base_url'], api_key=api_key)
            elif 'claude' in params['engine']:
                self.client = anthropic.Anthropic(api_key=api_key)
            elif 'mistral' in params['engine']:
                self.client = MistralClient(api_key=api_key)
            else:
                raise ValueError("Invalid Model Engine. Please provide a valid model engine.")
        else:
            if 'gpt' in params['engine']:
                self.client = OpenAI(base_url=params['base_url'], api_key=os.environ.get("OPENAI_API_KEY"))
            elif 'deepseek' in params['engine']:
                self.client = OpenAI(base_url=params['base_url'], api_key=os.environ.get("DEEPSEEK_API_KEY"))
            elif 'claude' in params['engine']:
                self.client = anthropic.Anthropic(api_key=os.environ.get("ANTHROPIC_API_KEY"))
            elif 'mistral' in params['engine']:
                self.client = MistralClient(api_key=os.environ.get("MISTRAL_API_KEY"))
            else:
                raise ValueError("Invalid Model Engine. Please provide a valid model engine.")


    def multi_threading_chat_completion(self, prompts, max_workers=4):
        # Select the function handler based on the model engine
        if 'gpt' in self.params['engine']:
            self.chat_completion_handler = self.openai_chat_completion
        elif 'deepseek' in self.params['engine']:
            self.chat_completion_handler = self.openai_chat_completion
        elif 'claude' in self.params['engine']:
            self.chat_completion_handler = self.claude_chat_completion
        elif 'mistral' in self.params['engine']:
            self.chat_completion_handler = self.mistral_chat_completion
        else:
            raise ValueError("Invalid Model Engine. Please provide a valid model engine.")

        # Create a buffer file to store the responses
        if 'buffer_path' not in self.params:
            self.params['buffer_path'] = './{}_temp_buffer.jsonl'.format(self.params['engine'])
        with open(self.params['buffer_path'], 'w') as f:
            pass
        self.outbuf = open(self.params['buffer_path'], 'a')
        
        # Create a lock to write to the buffer file
        self.lock = threading.Lock()

        data = [{'prompt':prompt} for prompt in prompts]
        timer = Timer()
        print(f"using model_{self.params['engine']}")
        print('Processing queires')
        with ThreadPoolExecutor(max_workers=max_workers) as executor:
            futures = list(
                tqdm(
                    # executor.map(self.single_thread_api_call, enumerate(prompts)), 
                    # executor.map(lambda x: score_error_reason_helper(x, client, subject, model_name), unscored_data_list),
                    executor.map(lambda x: self.single_thread_api_call(x), data),
                    total=len(prompts)
                )
            )
        print("Average time after {0} samples: {1}".format(len(prompts), timer.get_time(restart=False) / len(prompts)))
        print('Processed queries')
        
        # # load buffered response back
        # result_list = load_jsonl(self.params['buffer_path'])
        # result_list = sorted(result_list, key=lambda x: x['id'])
        # result_list = [result['result'] for result in result_list]
        return [dp['response'] for dp in data]



    def openai_chat_completion(self, prompt):
        if self.params['chat_system_instruction']:
            msg = [
                {'role': 'system', 'content': self.params['chat_system_instruction']},
                {'role': 'user', 'content': prompt},
            ]
        else:
            msg = [{'role': 'user', 'content': prompt}]

        attempt = 0
        while True:
            try:
                # chat model
                response = self.client.chat.completions.create(
                    model=self.params['engine'],
                    messages=msg,
                    temperature=self.params['temperature'],
                    max_tokens=self.params['max_tokens'],
                )
                return response.choices[0].message.content.strip()

            except Exception as e:
                print(e)
                # print(response)
                # print(response.choices[0].message.content.strip())
                attempt += 1
                if attempt >= self.params['attempt_num']:
                    return None
                wait_sec = 1
                time.sleep(wait_sec)


    def claude_chat_completion(self, prompt):
        attempt = 0
        while True:
            try:
                response = self.client.messages.create(
                    model=self.params['engine'],
                    max_tokens=self.params['max_tokens'],
                    temperature=self.params['temperature'],
                    # system="Respond only in Yoda-speak.",
                    messages=[
                        {"role": "user", "content": prompt}
                    ]
                )
                return response.content[0].text

            except Exception as e:
                print(e)
                attempt += 1
                if attempt >= self.params['attempt_num']:
                    return None
                wait_sec = 1
                time.sleep(wait_sec)


    def mistral_chat_completion(self, prompt):
        attempt = 0
        while True:
            try:
                response = self.client.chat(
                    model=self.params['engine'],
                    messages=[{'role': 'user', 'content': prompt}],
                    max_tokens=self.params['max_tokens'],
                    temperature=self.params['temperature'],
                )
                return response.choices[0].message.content

            except Exception as e:
                print(e)
                print(response)
                print(response.choices[0].message.content)
                attempt += 1
                if attempt >= self.params['attempt_num']:
                    return None
                wait_sec = 5
                time.sleep(wait_sec)


    def write_result(self, result):
        self.lock.acquire()
        self.outbuf.write(json.dumps(result, ensure_ascii=False) + '\n')
        self.outbuf.flush()
        self.lock.release()


    # def single_thread_api_call(self, id_prompt):
    #     id, prompt = id_prompt
    #     response = self.chat_completion_handler(prompt)
    #     result = {'id': id, 'result': response}
    #     self.write_result(result)
        

    def single_thread_api_call(self, datapoint):
        response = self.chat_completion_handler(datapoint['prompt'])
        datapoint['response'] = response


if __name__ == '__main__':

    example_prompt = '''\
Evaluate and compare the coherence of the two following summary candidates for the given input source text.

Input source text: Paul Merson has restarted his row with Andros Townsend after the Tottenham midfielder was brought on with only seven minutes remaining in his team's 0-0 draw with Burnley on Sunday. 'Just been watching the game, did you miss the coach? #RubberDub #7minutes,' Merson put on Twitter. Merson initially angered Townsend for writing in his Sky Sports column that 'if Andros Townsend can get in (the England team) then it opens it up to anybody.' Paul Merson had another dig at Andros Townsend after his appearance for Tottenham against Burnley Townsend was brought on in the 83rd minute for Tottenham as they drew 0-0 against Burnley Andros Townsend scores England's equaliser in their 1-1 friendly draw with Italy in Turin on Tuesday night The former Arsenal man was proven wrong when Townsend hit a stunning equaliser for England against Italy and he duly admitted his mistake. 'It's not as though I was watching hoping he wouldn't score for England, I'm genuinely pleased for him and fair play to him – it was a great goal,' Merson said. 'It's just a matter of opinion, and my opinion was that he got pulled off after half an hour at Manchester United in front of Roy Hodgson, so he shouldn't have been in the squad. 'When I'm wrong, I hold my hands up. I don't have a problem with doing that - I'll always be the first to admit when I'm wrong.' Townsend hit back at Merson on Twitter after scoring for England against Italy Sky Sports pundit  Merson (centre) criticised Townsend's call-up to the England squad last week Townsend hit back at Merson after netting for England in Turin on Wednesday, saying 'Not bad for a player that should be 'nowhere near the squad' ay @PaulMerse?' Any bad feeling between the pair seemed to have passed but Merson was unable to resist having another dig at Townsend after Tottenham drew at Turf Moor.

Compare the following outputs:

Summary candidate A: paul merson was brought on with only seven minutes remaining in his team 's 0-0 draw with burnley . andros townsend scored the tottenham midfielder in the 89th minute . paul merson had another dig at andros townsend after his appearance . the midfielder had been brought on to the england squad last week . click here for all the latest arsenal news news .

Summary candidate B: paul merson has restarted his row with andros townsend . the tottenham midfielder was brought on with only seven minutes remaining in his team 's 0-0 draw with burnley . andros townsend scores england 's equaliser in their 1-1 friendly draw with italy in turin .

Question: Which summary candidate has better coherence? If the candidate A is better, please return 'A'. If the candidate B is better, please return 'B'. You must return the choice only.
Answer: \
'''
    reverse_example_prompt = '''\
Evaluate and compare the coherence of the two following summary candidates for the given input source text.

Input source text: Paul Merson has restarted his row with Andros Townsend after the Tottenham midfielder was brought on with only seven minutes remaining in his team's 0-0 draw with Burnley on Sunday. 'Just been watching the game, did you miss the coach? #RubberDub #7minutes,' Merson put on Twitter. Merson initially angered Townsend for writing in his Sky Sports column that 'if Andros Townsend can get in (the England team) then it opens it up to anybody.' Paul Merson had another dig at Andros Townsend after his appearance for Tottenham against Burnley Townsend was brought on in the 83rd minute for Tottenham as they drew 0-0 against Burnley Andros Townsend scores England's equaliser in their 1-1 friendly draw with Italy in Turin on Tuesday night The former Arsenal man was proven wrong when Townsend hit a stunning equaliser for England against Italy and he duly admitted his mistake. 'It's not as though I was watching hoping he wouldn't score for England, I'm genuinely pleased for him and fair play to him – it was a great goal,' Merson said. 'It's just a matter of opinion, and my opinion was that he got pulled off after half an hour at Manchester United in front of Roy Hodgson, so he shouldn't have been in the squad. 'When I'm wrong, I hold my hands up. I don't have a problem with doing that - I'll always be the first to admit when I'm wrong.' Townsend hit back at Merson on Twitter after scoring for England against Italy Sky Sports pundit  Merson (centre) criticised Townsend's call-up to the England squad last week Townsend hit back at Merson after netting for England in Turin on Wednesday, saying 'Not bad for a player that should be 'nowhere near the squad' ay @PaulMerse?' Any bad feeling between the pair seemed to have passed but Merson was unable to resist having another dig at Townsend after Tottenham drew at Turf Moor.

Compare the following outputs:

Summary candidate A: paul merson has restarted his row with andros townsend . the tottenham midfielder was brought on with only seven minutes remaining in his team 's 0-0 draw with burnley . andros townsend scores england 's equaliser in their 1-1 friendly draw with italy in turin .
Summary candidate B: paul merson was brought on with only seven minutes remaining in his team 's 0-0 draw with burnley . andros townsend scored the tottenham midfielder in the 89th minute . paul merson had another dig at andros townsend after his appearance . the midfielder had been brought on to the england squad last week . click here for all the latest arsenal news news .

Question: Which summary candidate has better coherence? If the candidate A is better, please return 'A'. If the candidate B is better, please return 'B'. You must return the choice only.
Answer: \
'''
    prompts = [example_prompt] * 3 + [reverse_example_prompt] * 2
    model = AutoAPIModel({'engine':'gpt-3.5-turbo'}, api_key="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")
    result = model.multi_threading_chat_completion(prompts)
    print(result)
