from collections import OrderedDict

import numpy as np
import torch
import torch.optim as optim
from torch import nn as nn

import rlkit.torch.pytorch_util as ptu
from rlkit.core.eval_util import create_stats_ordered_dict
from rlkit.torch.torch_rl_algorithm import TorchTrainer


class BetaStepTrainer(TorchTrainer):
    """
    Trainer for Behavior Cloning
    Policy is trained by maximizing log likelihood of actions in a given dataset.
    Q function is trained by Q^pi
    """
    def __init__(
            self,
            env,
            behavior_policy,
            qf,
            target_qf,

            discount=0.99,
            reward_scale=1.0,

            policy_lr=1e-4,
            qf_lr=1e-4,
            optimizer_class=optim.Adam,

            soft_target_tau=5e-3,
            target_update_period=2,

            std_scale = 1.0,

    ):
        super().__init__()
        self.env = env
        self.beta_prime_policy = behavior_policy
        self.qf = qf
        self.target_qf = target_qf
        self.soft_target_tau = soft_target_tau
        self.target_update_period = target_update_period

        self.qf_criterion = nn.MSELoss()

        self.policy_optimizer = optimizer_class(
            self.beta_prime_policy.parameters(),
            lr=policy_lr,
        )
        self.qf_optimizer = optimizer_class(
            self.qf.parameters(),
            lr=qf_lr,
        )

        self.discount = discount
        self.reward_scale = reward_scale
        self.eval_statistics = OrderedDict()
        self._n_train_steps_total = 0
        self._need_to_update_eval_statistics = True

        self.discrete = False

        self.std_scale = std_scale

    def train_from_torch(self, batch):
        rewards = batch['rewards']
        terminals = batch['terminals']
        obs = batch['observations']
        actions = batch['actions']
        next_obs = batch['next_observations']

        # if self._n_train_steps_total < 20000:
        """
        Policy and Alpha Loss
        """
        if self._n_train_steps_total < 10000:
            log_likelihood = self.beta_prime_policy.log_prob(obs, actions)
            policy_loss = -log_likelihood.mean()

            self.policy_optimizer.zero_grad()
            policy_loss.backward()
            self.policy_optimizer.step()
        else:

            """
            QF Loss
            """
            q_pred = self.qf(obs, actions)

            next_actions, _, _, _, *_ = self.beta_prime_policy(
                next_obs, reparameterize=False, return_log_prob=True,
            )

            target_q_values = self.target_qf(next_obs, next_actions)
            q_target = self.reward_scale * rewards + (1. - terminals) * self.discount * target_q_values
            qf_loss = self.qf_criterion(q_pred, q_target.detach())

            self.qf_optimizer.zero_grad()
            qf_loss.backward()
            self.qf_optimizer.step()

            """
            Soft Updates
            """
            if self._n_train_steps_total % self.target_update_period == 0:
                ptu.soft_update_from_to(
                    self.qf, self.target_qf, self.soft_target_tau
                )

        self._n_train_steps_total += 1

    def get_diagnostics(self):
        return self.eval_statistics

    def end_epoch(self, epoch):
        self._need_to_update_eval_statistics = True

    @property
    def networks(self):
        return [
            self.beta_prime_policy,
            self.qf,
            self.target_qf,
        ]

    def get_snapshot(self):
        return dict(
            policy=self.beta_prime_policy,
            qf=self.qf,
            target_qf=self.target_qf,
        )

    def set_snapshot(self, snapshot):
        self.beta_prime_policy = snapshot['policy']
        self.qf = snapshot['qf']
        self.target_qf = snapshot['target_qf']