################################################################################
# experilog/measure.py
#
# 
# 
# 
# 2024
#
# Container for measurements.

from numpy        import ndarray
from numpy.typing import NDArray
from typing       import Optional, Union
from warnings     import warn

UNIT_WARN_THRESHOLD = 2

Quantity = Union[int, float, NDArray]

class Measurement():
  """
  Defines a measurement - a type containing a quantity and a unit.
  """

  def __init__(self,
      # Arguments:
      quantity: Quantity,
      # Keyword Arguments:
      unit:            Optional[str] = None,
      silence_warning: bool          = False
    ) -> None:
    """
    Initializes ``Measurement``.

    Args:
      quantity (Quantity):
        The quantity measured.
      unit (str, optional):
        The full plural name of the unit being measured.
        Defaults to ``None``.
      silence_warning (bool, optional):
        Whether to silence the unit name warning.
        Defaults to ``False``.
    """
    if unit is not None and not silence_warning:
      if len(unit) <= UNIT_WARN_THRESHOLD:
        warn(
          f"Unit '{unit}' is quite short. Are you sure you used the full" + \
          "plural unit name?"
        )
    self._quantity = quantity
    self._unit     = unit
  
  def __repr__(self):
    return f"{self._quantity} {self._unit}"
  
  def get_json_dict(self):
    """
    Get the quantity in JSON dict format.

    Returns:
      Dict[str, Any]:
        The JSON dict.
    """
    quantity = self._quantity.tolist() if isinstance(self._quantity, ndarray) \
                                       else self._quantity
    return {"quantity": quantity, "unit": self._unit}