import numpy as np
from PIL import Image
import random
import os
import os.path as osp
import glob

from typing import Tuple

def random_crop(dir_img: str, crop_size: int) -> Tuple[np.ndarray, Tuple[int, int]]:
    """Randomly crops a part of the image of specified dimensions.

    Args:
    - dir_img (str): Path to the image to be cropped.
    - crop_size (int): Desired size of the cropped image.
    
    Returns:
    - np.ndarray: The cropped image.
    - Tuple[int, int]: Top-left coordinates of the cropped region.
    """
    img = Image.open(dir_img)
    if img.mode == "RGBA":
        img = img.convert("L")

    # Random intensity normalization
    #min_val = np.random.randint(7000, 15000)
    #max_val = np.random.randint(20000, 25000)
    #img_normalized = np.clip((np.array(img, np.float32) - min_val) / (max_val - min_val), 0, 1)
    
    # Now, you may want to convert the float normalized image back to uint8 format
    # to get values between 0 and 255 for further processing
    ## change
    #img = np.array(img * 255, dtype=np.uint32)
    #print(img.shape)
    #width, height = img.shape[:2]
#
    ## Coordinate for random crop, top-left corner
    #x = random.randrange(0, height - crop_size)
    #y = random.randrange(0, width - crop_size)
    
    # Convert the PIL Image to a NumPy array
    img_array = np.array(img)

    # Note: the .size attribute of an Image object returns (width, height) 
    # instead of (height, width) which is the order of dimensions in a NumPy array.
    width, height = img.size

    # Coordinate for random crop, top-left corner
    x = random.randrange(0, height - crop_size)
    y = random.randrange(0, width - crop_size)

    return img_array[x : x + crop_size, y : y + crop_size], (x, y)


def get_data(dir_dataset: str, dataset_size: int, crop_size: int) -> np.ndarray:
    """Randomly crops multiple parts of an image and saves them.

    Args:
    - dir_dataset (str): Directory where the image is located.
    - dataset_size (int): Number of crops desired.
    - crop_size (int): Size of the crops.
    - save_dir (str): Directory to save the cropped images.
    
    Returns:
    - np.ndarray: Array of all cropped images.
    """
    dir_cases = glob.glob(osp.join(dir_dataset, "*.bmp"))
    num_cases = len(dir_cases)
    dataset = np.zeros((dataset_size, crop_size, crop_size, 3), dtype=np.uint8)
    samples_per_case = dataset_size // num_cases
    remaining = dataset_size % num_cases

    db_idx = 0

    # Equally sample from cases in directory
    for case in dir_cases:
        for sample_idx in range(samples_per_case):
            sampled_case, _ = random_crop(case, crop_size)
            dataset[db_idx] = np.copy(sampled_case)

            # Save the crop to the specified directory
            img_to_save = Image.fromarray(sampled_case)
            #img_to_save.save(osp.join(save_dir, f"crop_{db_idx}.tiff"))

            db_idx += 1

    # Sample remainders
    for idx in range(remaining):
        sampled_case, _ = random_crop(dir_cases[idx], crop_size)
        dataset[db_idx] = np.copy(sampled_case)
        
        # Save the crop to the specified directory
        img_to_save = Image.fromarray(sampled_case)
        #img_to_save.save(osp.join(save_dir, f"crop_{db_idx}.tiff"))

        db_idx += 1

    return dataset

# Example usage
#get_data("path_to_your_images", 100, 200, "path_to_save_directory")
