"""Re-definition with different names of the current types in the OpenAI API.
We re-define them to avoid relying on the API, and to allow for more flexibility
in the agent implementations.

The license for the original code is here:
https://github.com/openai/openai-python/blob/bba23438a63121102fe066982c91771fcca19c80/LICENSE
"""

from typing import Literal

from pydantic import BaseModel
from typing_extensions import Required, TypedDict

from agentgym.functions_engine.functions_engine import FunctionCall


class ChatUserMessage(TypedDict, total=False):
    role: Required[Literal["user"]]
    content: Required[str]


class ChatAssistantMessage(TypedDict, total=False):
    role: Required[Literal["assistant"]]
    content: Required[str | None]
    tool_calls: Required[list[FunctionCall] | None]


class ChatToolResultMessage(TypedDict, total=False):
    tool_call: Required[FunctionCall]
    content: Required[str]
    role: Required[Literal["tool"]]
    tool_call_id: Required[str | None]
    error: Required[str | None]


class ChatSystemMessage(TypedDict, total=False):
    role: Required[Literal["system"]]
    content: Required[str]


ChatMessage = ChatUserMessage | ChatAssistantMessage | ChatToolResultMessage | ChatSystemMessage


class TaskEnvironment(BaseModel):
    responses: list[dict[str, str]] = []

    def assistant_messages(self) -> list[dict[str, str]]:
        return [m for m in self.responses if m["role"] == "assistant"]

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, TaskEnvironment):
            return False
        self_dict = self.dict(exclude={"responses"})
        other_dict = other.dict(exclude={"responses"})
        return self_dict == other_dict


class DummyEnv(TaskEnvironment): ...
