'''
from convert import (
    lidar_to_pano_with_intensities,
    lidar_to_pano_with_intensities_with_bbox_mask,
)

from convert import (
    lidar_to_pano_with_intensities,
    lidar_to_pano_with_intensities_with_bbox_mask,
)
'''
import numpy as np
import os
from pathlib import Path
from tqdm import tqdm
import shutil
from nus_loader_swps import nus_loader
from filter_ground import point_removal
import argparse
import configargparse

def lidar_to_pano_with_intensities(
    local_points_with_intensities: np.ndarray,
    lidar_H: int,
    lidar_W: int,
    lidar_K: int,
    max_depth=80,
    min_depth=1.8
):
    """
    Convert lidar frame to pano frame with intensities.
    Lidar points are in local coordinates.

    Args:
        local_points: (N, 4), float32, in lidar frame, with intensities.
        lidar_H: pano height.
        lidar_W: pano width.
        lidar_K: lidar intrinsics.
        max_depth: max depth in meters.

    Return:
        pano: (H, W), float32.
        intensities: (H, W), float32.
    """
    #Actually we don't do this.Just skip.
    point_cloud=point_removal(local_points_with_intensities,filter_range=True)
    local_points_with_intensities_without_ground=point_cloud
    #no ground
    local_points_without_ground = local_points_with_intensities_without_ground[:, :3]
    local_point_intensities_without_ground = local_points_with_intensities_without_ground[:, 3]
    local_point_intensities_without_ground=local_point_intensities_without_ground/255
    #
    fov_up, fov = lidar_K
    fov_down = fov - fov_up
    # Compute dists to lidar center.
    dists = np.linalg.norm(local_points_without_ground, axis=1)
    # Fill pano and intensities.
    pano_without_ground = np.zeros((lidar_H, lidar_W))
    intensities_without_ground = np.zeros((lidar_H, lidar_W))
    for local_point, dist, local_point_intensity in zip(
        local_points_without_ground,
        dists,
        local_point_intensities_without_ground,
    ):
        # Check max depth and min depth
        if dist >= max_depth or dist<=min_depth:
            continue

        x, y, z = local_point
        beta = np.pi - np.arctan2(y, x)
        alpha = np.arctan2(z, np.sqrt(x**2 + y**2)) + fov_down / 180 * np.pi
        c = int(round(beta / (2 * np.pi / lidar_W)))
        r = int(round(lidar_H - alpha / (fov / 180 * np.pi / lidar_H)))

        # Check out-of-bounds.
        if r >= lidar_H or r < 0 or c >= lidar_W or c < 0:
            continue

        # Set to min dist if not set.
        if pano_without_ground[r, c] == 0.0:
            pano_without_ground[r, c] = dist
            intensities_without_ground[r, c] = local_point_intensity
        elif pano_without_ground[r, c] > dist:
            pano_without_ground[r, c] = dist
            intensities_without_ground[r, c] = local_point_intensity

    #with ground
    local_points = local_points_with_intensities[:, :3]
    local_point_intensities = local_points_with_intensities[:, 3]
    local_point_intensities=local_point_intensities/255
    
    
    fov_up, fov = lidar_K
    fov_down = fov - fov_up
    # Compute dists to lidar center.
    dists = np.linalg.norm(local_points, axis=1)
    # Fill pano and intensities.
    pano = np.zeros((lidar_H, lidar_W))
    intensities = np.zeros((lidar_H, lidar_W))
    for local_points, dist, local_point_intensity in zip(
        local_points,
        dists,
        local_point_intensities,
    ):
        # Check max depth and min depth
        if dist >= max_depth or dist<=min_depth:
            continue

        x, y, z = local_points
        beta = np.pi - np.arctan2(y, x)
        alpha = np.arctan2(z, np.sqrt(x**2 + y**2)) + fov_down / 180 * np.pi
        c = int(round(beta / (2 * np.pi / lidar_W)))
        r = int(round(lidar_H - alpha / (fov / 180 * np.pi / lidar_H)))

        # Check out-of-bounds.
        if r >= lidar_H or r < 0 or c >= lidar_W or c < 0:
            continue

        # Set to min dist if not set.
        if pano[r, c] == 0.0:
            pano[r, c] = dist
            intensities[r, c] = local_point_intensity
        elif pano[r, c] > dist:
            pano[r, c] = dist
            intensities[r, c] = local_point_intensity

    return pano, intensities, pano_without_ground, intensities_without_ground


def get_arg_parser():
    parser = configargparse.ArgumentParser()
    parser.add_argument("--samples",action="store_true")

    parser.add_argument(
        "--start",
        type=int, 
        default=0,
        help="choose start",
    )

    return parser


def LiDAR_2_Pano_Nus(
    local_points_with_intensities, lidar_H, lidar_W, intrinsics, max_depth=80.0
):
    pano, intensities,pano_without_ground,intensities_without_ground = lidar_to_pano_with_intensities(
        local_points_with_intensities=local_points_with_intensities,
        lidar_H=lidar_H,
        lidar_W=lidar_W,
        lidar_K=intrinsics,
        max_depth=max_depth,
    )
    range_view = np.zeros((lidar_H, lidar_W, 3))
    range_view[:, :, 1] = intensities
    range_view[:, :, 2] = pano

    range_view_without_ground = np.zeros((lidar_H, lidar_W, 3))
    range_view_without_ground[:, :, 1] = intensities_without_ground
    range_view_without_ground[:, :, 2] = pano_without_ground
    return range_view,range_view_without_ground


def generate_train_data(
    H,
    W,
    intrinsics,
    lidar_paths,
    out_dir,
    points_dim,
):
    """
    Args:
        H: Heights of the range view.
        W: Width of the range view.
        intrinsics: (fov_up, fov) of the range view.
        out_dir: Output directory.
    """

    out_dir = Path(out_dir)
    out_dir.mkdir(parents=True, exist_ok=True)

    for lidar_path in tqdm(lidar_paths):
        point_cloud = np.fromfile(lidar_path, dtype=np.float32)
        point_cloud = point_cloud.reshape((-1, points_dim))
        pano,pano_without_ground = LiDAR_2_Pano_Nus(point_cloud, H, W, intrinsics)
        frame_name = lidar_path.split("/")[-1] 
        suffix = frame_name.split(".")[-1]
        frame_name = frame_name.replace(suffix, "npy")
        frame_name_without_ground=frame_name.split(".")[0]+".pcd.withoutground.npy"
        np.save(out_dir / frame_name, pano)
        np.save(out_dir/ frame_name_without_ground,pano_without_ground)


def create_nus_rangeview(frames_path_select):
    out_dir='../../data/nuscenes/train_swps'
    H = 32
    W = 1080
    intrinsics = (10.0, 40.0)  # fov_up, fov
    
    generate_train_data(
    H=H,
    W=W,
    intrinsics=intrinsics,
    lidar_paths=frames_path_select,
    out_dir=out_dir,
    points_dim=5,
    )

def main():
    parser = get_arg_parser()
    opt = parser.parse_args()
    root= '../../data/nuscenes/nuscenes-mini'
    if opt.samples:
        frames_root='../../data/nuscenes/nuscenes-mini/samples/LIDAR_TOP'
    else:
        frames_root='../../data/nuscenes/nuscenes-mini/sweeps/LIDAR_TOP'
    T,F=nus_loader(root,frames_root)

    if opt.samples:
        F=F[opt.start:opt.start+36]
    else:
        F=F[opt.start:opt.start+180:5]
        
    frames_path_select=[os.path.join(frames_root,frame_name) for frame_name in F]
    create_nus_rangeview(frames_path_select)



if __name__ == "__main__":
    main()

