
using SpeedyWeather
using ArgParse

function parse_commandline()
    s = ArgParseSettings()

    @add_arg_table s begin
        # General arguments
        "--simulation_days"
        help = "number of days to simulate"
        arg_type = Int
        default = 30  # Set the default value here

        "--output.id"
        help = "simulation id"

        # Output related arguments
        "--output.path"
        help = "output path of the simulation"
        arg_type = String
        default = "DATA"
        required = true

        # Planet related arguments
        "--planet.rotation"
        help = "rotation rate of the planet"
        arg_type = Float64
        default = 7.292e-5

        "--planet.gravity"
        help = "gravity of the planet"
        arg_type = Float64
        default = 9.81

        "--planet.axial_tilt"
        help = "axial tilt of the planet"
        arg_type = Float64
        default = 23.4

        "--planet.solar_constant"
        help = "solar constant"
        arg_type = Float64
        default = 1365.0

        # planet atmosphere related arguments
        "--atmosphere.mol_mass_dry_air"
        help = "molar mass of dry air"
        arg_type = Float64
        default = 28.97

        "--atmosphere.mol_mass_vapour"
        help = "molar mass of vapour"
        arg_type = Float64
        default = 18.02

        "--atmosphere.heat_capacity"
        help = "heat capacity"
        arg_type = Float64
        default = 1004.0

        "--atmosphere.R_gas"
        help = "gas constant"
        arg_type = Float64
        default = 8.3145

        "--atmosphere.R_dry"
        help = "dry gas constant"
        arg_type = Float64
        default = 287.054

        "--atmosphere.R_vapour"
        help = "vapour gas constant"
        arg_type = Float64
        default = 461.5

        "--atmosphere.mol_ratio"
        help = "molar ratio"
        arg_type = Float64
        default = 0.622

        "--atmosphere.mu_virt_temp"
        help = "virtual temperature"
        arg_type = Float64
        default = 0.608

        "--atmosphere.kappa"
        help = "kappa"
        arg_type = Float64
        default = 0.2859

        "--atmosphere.latent_heat_condensation"
        help = "latent heat of condensation"
        arg_type = Float64
        default = 2.501e6

        "--atmosphere.latent_heat_sublimation"
        help = "latent heat of sublimation"
        arg_type = Float64
        default = 2.801e6

        "--atmosphere.stefan_boltzmann"
        help = "stefan boltzmann constant"
        arg_type = Float64
        default = 5.67e-8

        "--atmosphere.pres_ref"
        help = "reference pressure"
        arg_type = Float64
        default = 100000.0

        "--atmosphere.temp_ref"
        help = "reference temperature"
        arg_type = Float64
        default = 288.0

        "--atmosphere.moist_lapse_rate"
        help = "moist lapse rate"
        arg_type = Float64
        default = 3.5e-3

        "--atmosphere.dry_lapse_rate"
        help = "dry lapse rate"
        arg_type = Float64
        default = 9.8e-3

        "--atmosphere.layer_thickness"
        help = "layer thickness"
        arg_type = Float64
        default = 8500.0
    end

    return parse_args(s)
end

function main()
    parsed_args = parse_commandline()
    spectral_grid = SpectralGrid(trunc=31, nlev=5)
    ocean = AquaPlanet(spectral_grid, temp_equator=302, temp_poles=273)
    land_sea_mask = AquaPlanetMask(spectral_grid)
    orography = NoOrography(spectral_grid)
    model = PrimitiveWetModel(; spectral_grid, ocean, land_sea_mask, orography)
    for (key, value) in parsed_args
        split_key = split(key, ".")
        if length(split_key) == 2
            sub_model, attribute = split_key
            if hasproperty(model, Symbol(sub_model))
                if hasproperty(getproperty(model, Symbol(sub_model)), Symbol(attribute))
                    setproperty!(getproperty(model, Symbol(sub_model)), Symbol(attribute), value)
                end
            end
        end
    end
    simulation = initialize!(model)
    run!(simulation, period=Day(parsed_args["simulation_days"]), output=true)
end

main()




