from __future__ import annotations

import asyncio
import hashlib
import os
import tempfile
from typing import Optional, TypeGuard, Union

import attrs
from cattrs.preconf.json import make_converter

import sandbagging.utils.optional as op
from sandbagging.utils.redis_cache_wrapper import RedisWrapper, cache_key

import ray
import torch

json_converter = make_converter()


@attrs.frozen
class CompilationFailure:
    out: CommandOut


def get_default_num_workers():
    MIN_SET_ASIDE_CORES = 2
    FRAC_CORES = 0.9

    total_cores = os.cpu_count()
    assert total_cores is not None
    worker_cores = min(FRAC_CORES * total_cores, total_cores - MIN_SET_ASIDE_CORES)

    max_workers = max(1, int(worker_cores))
    return max_workers


run_semaphore = asyncio.Semaphore(get_default_num_workers())


@attrs.frozen
class CommandReturn:
    stdout: str
    stderr: str
    return_code: int


@attrs.frozen
class CommandTimeout:
    ...


CommandOut = Union[CommandReturn, CommandTimeout]


def command_out_is_ok(x: CommandOut) -> TypeGuard[CommandReturn]:
    return isinstance(x, CommandReturn) and x.return_code == 0


async def simple_command_run(
    command: str, timeout_seconds: float = 10.0, memory_limit_mb: float = 1024.0, inp: Optional[str] = None
) -> CommandOut:
    # Convert memory_limit from MB to KB
    memory_limit_kb = int(memory_limit_mb * 1024)

    async with run_semaphore:
        try:
            process = await asyncio.wait_for(
                asyncio.create_subprocess_exec(
                    "bash",
                    "-c",
                    f"ulimit -v {memory_limit_kb}; {command}",
                    stdin=asyncio.subprocess.PIPE,
                    stdout=asyncio.subprocess.PIPE,
                    stderr=asyncio.subprocess.PIPE,
                    universal_newlines=False,
                ),
                timeout=timeout_seconds,
            )
            stdout_bytes, stderr_bytes = await asyncio.wait_for(
                process.communicate(input=inp.encode("utf-8") if inp is not None else None), timeout=timeout_seconds
            )
            stdout = stdout_bytes.decode("utf-8", errors="replace")
            stderr = stderr_bytes.decode("utf-8", errors="replace")
            return CommandReturn(stdout=stdout, stderr=stderr, return_code=op.unwrap(process.returncode))
        except asyncio.TimeoutError:
            return CommandTimeout()
        finally:
            try:
                process.kill()
            except Exception:
                ...


async def read_output(stream, output_list):
    async for chunk in stream:
        output_list.append(chunk.decode())


compile_dir = os.path.expanduser("~/.cache/monitoring_cpp_compile/")
os.makedirs(compile_dir, exist_ok=True)


@attrs.frozen
class CompiledProgram:
    _code_hash: str = attrs.field(alias="_code_hash")

    @property
    def code_hash(self) -> str:
        return self._code_hash

    @property
    def binary_path(self) -> str:
        return f"/{compile_dir}/{self.code_hash}.out"

    @property
    def error_path(self) -> str:
        return f"/{compile_dir}/{self.code_hash}_err.json"

    @classmethod
    async def compile_cpp_code(cls, code: str) -> Union[CompiledProgram, CompilationFailure]:
        salt = "cpp_compile_no_c"
        m = hashlib.sha256()
        m.update((code + salt).encode("utf-8"))
        code_hash = m.hexdigest()

        out = cls(_code_hash=code_hash)

        try:
            with open(out.error_path, "r") as f:
                return json_converter.loads(f.read(), CompilationFailure)
        except FileNotFoundError:
            ...

        if os.path.exists(out.binary_path):
            return out

        with tempfile.TemporaryDirectory(prefix=compile_dir) as temp:
            cpp_path = f"{temp}/file.cpp"
            temp_binary_path = f"{temp}/file.out"
            with open(cpp_path, "w") as f:
                f.write(code)

            compile_command = ["g++", "-std=c++20", "-O3", cpp_path, "-o", temp_binary_path]

            compile_out = await simple_command_run(" ".join(compile_command))

            if not command_out_is_ok(compile_out):
                fail = CompilationFailure(compile_out)
                err_s = json_converter.dumps(fail)
                with open(out.error_path, "w") as f:
                    f.write(err_s)
                return fail

            # mv approach to avoid collide
            os.rename(temp_binary_path, out.binary_path)

        return out

    async def run_input(
        self,
        inp: Optional[str],
        timeout_seconds: float = 0.5,
        memory_limit_mb: float = 1_024.0,
        override_cache: bool = False,
    ) -> CommandOut:
        key = cache_key(
            dict(code_hash=self.code_hash, inp=inp, timeout_seconds=timeout_seconds, memory_limit_mb=memory_limit_mb),
            "code_exec_no_c_v2",
        )

        if not override_cache:
            cache_result = await RedisWrapper.singleton().read(key)
            if cache_result is not None:
                return json_converter.loads(cache_result, CommandOut)

        out = await simple_command_run(
            self.binary_path, timeout_seconds=timeout_seconds, memory_limit_mb=memory_limit_mb, inp=inp
        )
        await RedisWrapper.singleton().write(key, json_converter.dumps(out, unstructure_as=CommandOut))
        return out

    async def run_inputs(
        self,
        inps: list[str],
        timeout_seconds: float = 0.5,
        memory_limit_mb: float = 1_024.0,
        override_cache: bool = False,
    ) -> list[CommandOut]:
        key = cache_key(
            dict(code_hash=self.code_hash, inps=inps, timeout_seconds=timeout_seconds, memory_limit_mb=memory_limit_mb),
            "code_exec_many_v2",
        )

        if not override_cache:
            cache_result = await RedisWrapper.singleton().read(key)
            if cache_result is not None:
                return json_converter.loads(cache_result, list[CommandOut])

        out = await asyncio.gather(
            *(
                self.run_input(
                    inp=inp,
                    timeout_seconds=timeout_seconds,
                    memory_limit_mb=memory_limit_mb,
                    override_cache=override_cache,
                )
                for inp in inps
            )
        )
        await RedisWrapper.singleton().write(key, json_converter.dumps(out, unstructure_as=list[CommandOut]))
        return out

    async def run_inputs_combined(
        self,
        inps: list[str],
        timeout_seconds: float = 0.5,
        memory_limit_mb: float = 1_024.0,
        override_cache: bool = False,
    ) -> list[CommandOut]:
        key = cache_key(
            dict(code_hash=self.code_hash, inps=inps, timeout_seconds=timeout_seconds, memory_limit_mb=memory_limit_mb),
            "code_exec_many",
        )

        if not override_cache:
            cache_result = await RedisWrapper.singleton().read(key)
            if cache_result is not None:
                return json_converter.loads(cache_result, list[CommandOut])

        out = await asyncio.gather(
            *(
                self.run_input(
                    inp=inp,
                    timeout_seconds=timeout_seconds,
                    memory_limit_mb=memory_limit_mb,
                    override_cache=override_cache,
                )
                for inp in inps
            )
        )
        await RedisWrapper.singleton().write(key, json_converter.dumps(out, unstructure_as=list[CommandOut]))
        return out


async def run_python_uncached(
    python_code: str,
    inp: Optional[str],
    timeout_seconds: float = 2.0,
    memory_limit_mb: float = 1_024.0,
    python_command: str = "python3",
):
    with tempfile.TemporaryDirectory() as temp:
        python_file = f"{temp}/file.py"
        with open(python_file, "w") as f:
            f.write(python_code)
        return await simple_command_run(
            f"{python_command} {python_file}", timeout_seconds=timeout_seconds, memory_limit_mb=memory_limit_mb, inp=inp
        )


async def run_python(
    python_code: str,
    inp: Optional[str],
    timeout_seconds: float = 2.0,
    memory_limit_mb: float = 1_024.0,
    python_command: str = "python3",
    override_cache: bool = False,
):
    key = cache_key(
        dict(
            python_code=python_code,
            inp=inp,
            timeout_seconds=timeout_seconds,
            memory_limit_mb=memory_limit_mb,
            python_command=python_command,
        ),
        "python_code_exec_rerun",
    )
    if not override_cache:
        cache_result = await RedisWrapper.singleton().read(key)
        if cache_result is not None:
            return json_converter.loads(cache_result, CommandOut)

    out = await run_python_uncached(
        python_code=python_code,
        inp=inp,
        timeout_seconds=timeout_seconds,
        memory_limit_mb=memory_limit_mb,
        python_command=python_command,
    )

    await RedisWrapper.singleton().write(key, json_converter.dumps(out, unstructure_as=CommandOut))
    return out


async def run_python_many_inp(
    python_code: str,
    inps: list[str],
    timeout_seconds: float = 2.0,
    memory_limit_mb: float = 1_024.0,
    python_command: str = "python3",
    override_cache: bool = False,
) -> list[CommandOut]:
    key = cache_key(
        dict(
            python_code=python_code,
            inps=inps,
            timeout_seconds=timeout_seconds,
            memory_limit_mb=memory_limit_mb,
            python_command=python_command,
        ),
        "python_code_exec_many",
    )
    if not override_cache:
        cache_result = await RedisWrapper.singleton().read(key)
        if cache_result is not None:
            return json_converter.loads(cache_result, list[CommandOut])

    out = await asyncio.gather(
        *(
            run_python(
                python_code=python_code,
                inp=inp,
                timeout_seconds=timeout_seconds,
                memory_limit_mb=memory_limit_mb,
                python_command=python_command,
                override_cache=override_cache,
            )
            for inp in inps
        )
    )

    await RedisWrapper.singleton().write(key, json_converter.dumps(out, unstructure_as=list[CommandOut]))
    return out
