# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------
import math
import sys
from typing import Iterable

import torch

import util.misc as misc
import util.lr_sched as lr_sched
import os
import psutil
import numpy as np
import time

import torch.distributed as dist

def print_mem_usage(cur_epoch, data_iter_step, is_informed_mask = False, num_shared_dict = 0):
    if cur_epoch < 110:
        if data_iter_step % 200 == 0:
            if is_informed_mask:
                print(f'Num shared_dict : {num_shared_dict}')
            print('Memory usage is {} MiB'.format(psutil.Process(os.getpid()).memory_info().rss / 1024 ** 2))

def print_gpu_mem(cur_epoch, iter, msg, is_informed_mask=False):
    if is_informed_mask and cur_epoch < 45 and iter % 200 == 0:
        allocated_memory = torch.cuda.memory_allocated()
        cached_memory = torch.cuda.memory_cached()
        print(msg)
        print(f"Allocated GPU Memory: {allocated_memory / 1024**3:.2f} GB")
        print(f"Cached GPU Memory: {cached_memory / 1024**3:.2f} GB")
            
def train_one_epoch(model: torch.nn.Module,
                    data_loader: Iterable, optimizer: torch.optim.Optimizer,
                    device: torch.device, epoch: int, loss_scaler, cur_epoch,
                    log_writer=None, args=None, started_informed_masking = False):

    model.train(True)
    metric_logger = misc.MetricLogger(delimiter="  ")
    metric_logger.add_meter('lr', misc.SmoothedValue(window_size=1, fmt='{value:.6f}'))
    header = 'Epoch: [{}]'.format(epoch)
    print_freq = 20

    accum_iter = args.accum_iter

    optimizer.zero_grad()

    if log_writer is not None:
        print('log_dir: {}'.format(log_writer.log_dir))

    no_mask_cnt = 0
    L  = 14*14
    new_hint_ratio, new_hint_portion = misc.schedule_hint(args.hint_ratio, args.hint_portion, 
                    args.hint_schedule, cur_epoch, args.epochs, args.min_portion, 
                    args.min_ratio, args.schedule_exp, args.full_schedule)
    for data_iter_step, (samples, _, mask_in) in enumerate(metric_logger.log_every(data_loader, print_freq, header)):
        # if data_iter_step > 0:
        #     print(f'Loader: {time.time() - start}')
        print_gpu_mem(cur_epoch, data_iter_step, 'start one iter', args.informed_mask)
        if data_iter_step % accum_iter == 0:
            lr_sched.adjust_learning_rate(optimizer, data_iter_step / len(data_loader) + epoch, args)
            
        if args.informed_mask:
            if data_iter_step == 0:
                global_rank = misc.get_rank()
                if global_rank == 0:
                    # print(index)
                    # print(torch.stack([index, first_p, second_p]))
                    # print(mask_index)
                    samples_store = samples.detach().cpu().numpy()
                    mask_in_store = mask_in.detach().cpu().numpy()
                    store_path = f'/data2/projects/jeongwoo/jeongwoo/mae/analysis/convergence/mask_samples_samples_{args.ref_cluster}_{int(args.hint_portion * 100)}'
                    save = np.save(store_path, samples_store)
                    store_path = f'/data2/projects/jeongwoo/jeongwoo/mae/analysis/convergence/mask_samples_masks_{args.ref_cluster}_{int(args.hint_portion * 100)}'
                    save = np.save(store_path, mask_in_store)
                    # raise IndexError
            
        print_gpu_mem(cur_epoch, data_iter_step, 'before counting', args.informed_mask)

        if isinstance(samples, list):
            for tensor in samples:
                tensor = tensor.to(device, non_blocking=True)
        else:
            samples = samples.to(device, non_blocking=True)
        mask_in = mask_in.to(device, non_blocking=True)
            
        print_gpu_mem(cur_epoch, data_iter_step, 'after batch to gpu', args.informed_mask)
        
        with torch.cuda.amp.autocast():      
            loss, _, _ = model(samples, mask_ratio=args.mask_ratio, informed_mask = args.informed_mask,
                                        mask_in = mask_in, is_informed_mask_started = started_informed_masking,
                                        hint_ratio = new_hint_ratio, hint_portion = new_hint_portion, 
                                        hint_prob=args.hint_prob, uniform_prob=args.uniform_prob, prob_mask=args.prob_mask)
        ####################
        print_mem_usage(cur_epoch, data_iter_step, args.informed_mask)
        #####################
        loss_value = loss.item()
        print_gpu_mem(cur_epoch, data_iter_step, 'after model', args.informed_mask)

        if not math.isfinite(loss_value):
            print("Loss is {}, stopping training".format(loss_value))
            sys.exit(1)

        loss /= accum_iter
        loss_scaler(loss, optimizer, parameters=model.parameters(),
                    update_grad=(data_iter_step + 1) % accum_iter == 0)
        if (data_iter_step + 1) % accum_iter == 0:
            optimizer.zero_grad()

        torch.cuda.synchronize()

        metric_logger.update(loss=loss_value)

        lr = optimizer.param_groups[0]["lr"]
        metric_logger.update(lr=lr)

        loss_value_reduce = misc.all_reduce_mean(loss_value)
        if log_writer is not None and (data_iter_step + 1) % accum_iter == 0:
            """ We use epoch_1000x as the x-axis in tensorboard.
            This calibrates different curves when batch size changes.
            """
            epoch_1000x = int((data_iter_step / len(data_loader) + epoch) * 1000)
            log_writer.add_scalar('train_loss', loss_value_reduce, epoch_1000x)
            log_writer.add_scalar('lr', lr, epoch_1000x)

        # if cur_epoch > 0:
        #     del ids_shuffle_and_l1val
        # del samples
        
        print_gpu_mem(cur_epoch, data_iter_step, 'before mask generation', args.informed_mask)
        # start = time.time()
    
    print_gpu_mem(cur_epoch, data_iter_step, 'after writing', args.informed_mask)
    
    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    print("Averaged stats:", metric_logger)
    print_gpu_mem(cur_epoch, data_iter_step, 'end one epoch', args.informed_mask)
    return {k: meter.global_avg for k, meter in metric_logger.meters.items()}, None