import numpy as np

# We provide the code for the three approximation methods discussed in the paper.
# X is the input matrix that contains all the pairwise scores.
# All methods returns the index of the selected method.

def solve_pmbr(X, ratio=1 / 16, lam=1e-1, steps=10, rank=3):
  r = rank
  n, m = X.shape
  N = int(np.trunc(n * m * ratio))
  Omega = np.random.choice(n * m, size=int(np.trunc(N)), replace=False)
  Omega = np.unravel_index(Omega, X.shape)
  y = X[Omega]

  # Use random initialization for matrices A,B
  A = np.random.normal(size=(n, r))
  B = np.random.normal(size=(r, m))

  def linsolve_regular(A, b, lam=lam):
    """Solve linear problem A@x = b with Tikhonov regularization / ridge

    regression
    """
    return np.linalg.solve(A.T @ A + lam * np.eye(A.shape[1]), A.T @ b)

  for _ in range(steps):
    # Update B
    for j in range(n):
      B[:, j] = linsolve_regular(A[Omega[0][Omega[1] == j]], y[Omega[1] == j])

    # Update A
    for i in range(n):
      A[i, :] = linsolve_regular(
          B[:, Omega[1][Omega[0] == i]].T, y[Omega[0] == i]
      )
  # Final estimate
  R = A @ B
  predicted_ord = np.argsort(np.mean(R, axis=1))[::-1]
  return predicted_ord[0]


def solve_nxk(X, ratio=1 / 16):
  n, _ = X.shape
  s = np.random.choice(n, size=int(np.trunc(n * ratio)), replace=False)
  M = X[:, s]
  # Replace cases where i==j is random by another entrie because it is always
  # extremely high.
  for j, i in enumerate(s):
    M[i, j] = X[i, j + 1]
  predicted_ord = np.argsort(np.mean(M, axis=1))[::-1]
  return predicted_ord[0]


def solve_sxs(X, ratio=1 / 16):
  n, _ = X.shape
  s = np.random.choice(n, size=int(np.sqrt(n * n * ratio)), replace=False)
  M = X[:, s]
  M = M[s, :]
  # We need to remap the order based on shuffle s.
  predicted_ord = s[np.argsort(np.mean(M, axis=1))[::-1]]
  return predicted_ord[0]
