import os
import json
import numpy as np
import glob
from joblib import Parallel, delayed

def extract_entropy_from_json(file_path):
    try:
        with open(file_path, 'r') as file:
            data = json.load(file)
            return data['logits_entropy']
    except (IOError, KeyError, json.JSONDecodeError):
        print(f"Error reading {file_path}. Skipping.")
        return None

def process_obj_id(obj_path):
    entropies = []
    if os.path.isdir(obj_path):
        json_files = glob.glob(os.path.join(obj_path, '*metrics*.json'))
        for file_path in json_files:
            entropy = extract_entropy_from_json(file_path)
            if entropy is not None:
                entropies.append(entropy)
    return entropies

def process_directory(base_path):
    results = {}
    entropy_sums = {f'obj_id{i}': [] for i in range(1, 6)}

    tasks = []
    for image_dir in os.listdir(base_path):
        image_path = os.path.join(base_path, image_dir)
        if os.path.isdir(image_path):
            for obj_id in entropy_sums.keys():
                obj_path = os.path.join(image_path, obj_id, '0')
                tasks.append(delayed(process_obj_id)(obj_path))

    all_entropies = Parallel(n_jobs=-1)(tasks)

    for i, entropies in enumerate(all_entropies):
        obj_id = f'obj_id{(i % 5) + 1}'
        entropy_sums[obj_id].extend(entropies)

    averages = {}
    for obj_id, entropies in entropy_sums.items():
        if entropies:
            averages[obj_id] = np.mean(entropies)
        else:
            averages[obj_id] = float('nan') 
    
    return averages

if __name__ == '__main__':
    base_path = ''
    averages = process_directory(base_path)
    print("Average logits_entropy values for each obj_id:")
    for obj_id, avg in averages.items():
        print(f"{obj_id}: {avg:.4f}")
