# -*- coding: utf-8 -*-

import numpy as np
from scipy.integrate import solve_ivp
import matplotlib.pyplot as plt
from scipy.stats import beta

# initialize the beliefs about the strategy H
# the default initial probability distributions are Beta distributions:
# Beta(alpha, beta). 
alpha = [20, 10*5, 6, 16*5, 10]  
beta = [10, 20*5, 4, 4*5, 5]


num_agents = 1000
time_steps = 500
num_sim = 1
sigma = 10
tau = 10

def cal_uH(belief_pre, belief_next):
    uH = 2 * (belief_next - belief_pre)
    return uH

def cal_pol(belief_pre, belief_next):
    uH = cal_uH(belief_pre, belief_next)
    uT = 0 - uH
    policy = 1 / (1 + np.exp(tau * (uT - uH)))
    return policy

def beta_Stats(alpha, beta):
    mean = alpha / (alpha + beta)
    var = alpha * beta / ((alpha + beta)**2 * (alpha + beta + 1))
    return [mean, var]


def agent_based_simulations():
    beliefs_D1 = np.zeros((num_agents, time_steps)) 
    beliefs_P1 = np.zeros((num_agents, time_steps)) 
    beliefs_P2 = np.zeros((num_agents, time_steps))
    beliefs_P3 = np.zeros((num_agents, time_steps))
    beliefs_D2 = np.zeros((num_agents, time_steps))
    mean_P1 = np.zeros((1, time_steps))
    mean_P2 = np.zeros((1, time_steps))
    mean_P3 = np.zeros((1, time_steps))
    var_P1 = np.zeros((1, time_steps))
    var_P2 = np.zeros((1, time_steps))
    var_P3 = np.zeros((1, time_steps))
    beliefs_D1[:, 0] = init_beliefs_D1
    beliefs_P1[:, 0] = init_beliefs_P1
    beliefs_P2[:, 0] = init_beliefs_P2
    beliefs_P3[:, 0] = init_beliefs_P3
    beliefs_D2[:, 0] = init_beliefs_D2
    
    for t in range(0, time_steps-1):
        print(t)
        policy_P1 = cal_pol(beliefs_D1[:, t], beliefs_P2[:,t])
        policy_P2 = cal_pol(beliefs_P1[:, t], beliefs_P3[:,t])
        policy_P3 = cal_pol(beliefs_P2[:, t], beliefs_D2[:,t])
        mean_pol_P1 = np.mean(policy_P1)
        mean_pol_P2 = np.mean(policy_P2)
        mean_pol_P3 = np.mean(policy_P3)
        
        beliefs_D1[:, t + 1] = (beliefs_D1[:, t] * (sigma + t) + 1) / (sigma + t + 1)
        beliefs_P1[:, t + 1] = (beliefs_P1[:, t] * (sigma + t) + mean_pol_P1) / (sigma + t + 1)
        beliefs_P2[:, t + 1] = (beliefs_P2[:, t] * (sigma + t) + mean_pol_P2) / (sigma + t + 1)
        beliefs_P3[:, t + 1] = (beliefs_P3[:, t] * (sigma + t) + mean_pol_P3) / (sigma + t + 1)
        beliefs_D2[:, t + 1] = (beliefs_D2[:, t] * (sigma + t) + 0) / (sigma + t + 1)
        mean_P1[0, t] = mean_pol_P1
        mean_P2[0, t] = mean_pol_P2
        mean_P3[0, t] = mean_pol_P3
        var_P1[0, t] = np.var(policy_P1)
        var_P2[0, t] = np.var(policy_P2)
        var_P3[0, t] = np.var(policy_P3)
        
    mean_belief_D1 = np.mean(beliefs_D1, axis = 0)
    mean_belief_Pl = np.mean(beliefs_P1, axis = 0)
    mean_belief_P2 = np.mean(beliefs_P2, axis = 0)
    mean_belief_P3 = np.mean(beliefs_P3, axis = 0)
    mean_belief_D2 = np.mean(beliefs_D2, axis = 0)
    print("agent-based simulations end")
    
    return mean_P1, mean_P2, mean_P3, mean_belief_D1, mean_belief_Pl, mean_belief_P2, mean_belief_P3, mean_belief_D2, var_P1, var_P2, var_P3


#initital settings
init_beliefs_D1 = np.random.beta(alpha[0], beta[0], num_agents)
init_beliefs_P1 = np.random.beta(alpha[1], beta[1], num_agents)
init_beliefs_P2 = np.random.beta(alpha[2], beta[2], num_agents)
init_beliefs_P3 = np.random.beta(alpha[3], beta[3], num_agents)
init_beliefs_D2 = np.random.beta(alpha[4], beta[4], num_agents)


#run agent-based simulations
sim_mean_pol_P1  = np.zeros((num_sim, time_steps))
sim_mean_pol_P2  = np.zeros((num_sim, time_steps))
sim_mean_pol_P3  = np.zeros((num_sim, time_steps))
sim_var_pol_P1 = np.zeros((num_sim, time_steps))
sim_var_pol_P2 = np.zeros((num_sim, time_steps))
sim_var_pol_P3 = np.zeros((num_sim, time_steps))

for i in range(0, num_sim):
    print(i)
    sim_results = agent_based_simulations()    
    sim_mean_pol_P1[i, :] = sim_results[0]
    sim_mean_pol_P2[i, :] = sim_results[1]
    sim_mean_pol_P3[i, :] = sim_results[2]
    sim_var_pol_P1[i, :] = sim_results[8]
    sim_var_pol_P2[i, :] = sim_results[9]
    sim_var_pol_P3[i, :] = sim_results[10]
#plot mean dynamics figure
plt.figure()
t_space=np.linspace(0, time_steps, time_steps)
plt.plot(t_space, sim_results[0].transpose(), linewidth = 1, label = 'P1')
plt.plot(t_space, sim_results[1].transpose(), linewidth = 1, label = 'P2')
plt.plot(t_space, sim_results[2].transpose(), linewidth = 1, label = 'P3')
plt.title('Simulation Results')
plt.xlabel('Time t')
plt.legend()
plt.show()





# save  results

file = open("mean_policy.txt", "w")
for i in range(0, num_sim):
    for policy in sim_mean_pol_P1[i]:    
        file.write(str(policy) + ', ')
    file.write('\n')
    for policy in sim_mean_pol_P2[i]:    
        file.write(str(policy) + ', ')
    file.write('\n')
    for policy in sim_mean_pol_P3[i]:    
        file.write(str(policy) + ', ')
    file.write('\n')
file.write('\n')
file.close()

file = open("variance_policy.txt", "w")
for i in range(0, num_sim):
    for policy in sim_var_pol_P1[i]:    
        file.write(str(policy) + ', ')
    file.write('\n')
    for policy in sim_var_pol_P2[i]:    
        file.write(str(policy) + ', ')
    file.write('\n')
    for policy in sim_var_pol_P3[i]:    
        file.write(str(policy) + ', ')
    file.write('\n')
file.write('\n')
file.close()




