% README 
% To see the effects of initial belief distributions, change the values of
% alpha_row, beta_row, alpha_col, beta_col. To reproduce the results in
% Figure 1, set alpha_row = 280, beta_row = 120 for the case of small
% initial variance, and set alpha_row = 14, beta_row = 6 for the case of 
% large initial variance. 

% parameters for this numerical method 
dmu = 0.005;
dt = 0.01;
T = 50; % the number of time steps

% parameters for smooth fictitious play
beta = 10;  
lambda = 10; 

% parameters for 2-player subgames, the default values are for stag hunt
% games
A = -3;
B = 2;

% initialization of the beliefs about the opponent strategy H
% the default initial probability distributions are Beta distributions:
% Beta(alpha_row, beta_row) for the first population, and 
% Beta(alpha_col, beta_col) for the second population.
alpha_row = 140*2; %0.1 2
beta_row = 60*2;
alpha_col = alpha_row;
beta_col = beta_row;

mu = 0:dmu:1;
nmu = 1/dmu+1;
nC1s = int32(1/dt);
nC = int32(T/dt);
t = 0;
count = t*nC1s;
dirName = sprintf('twopopulation_staghunt',1,1);
if ~exist(dirName, 'dir')
    mkdir(dirName);
end
pMuT_row = zeros(T+1, nmu);
pMuT_col = zeros(T+1, nmu);
EmuT = zeros(T+1, 2);
ExT = zeros(T+1, 2);
if count==0
    pmu_row = betapdf(mu, alpha_row, beta_row); 
    pmu_col = betapdf(mu, alpha_col, beta_col); 
    Emu_row = int1d(mu .* pmu_row, dmu);
    Emu_col = int1d(mu .* pmu_col, dmu);
    pMuT_row(1, :) = pmu_row;
    pMuT_col(1, :) = pmu_col;
    int1d(pMuT_col, dmu)
else
    fileName = sprintf('%s/p%03d.mat',dirName,count/nC1s);
    load(fileName);
end

%caluclate expected payoffs
expmu = exp(beta * A * mu + beta * B);
x = 1 ./ (1 + expmu);

while 1
    if count>nC
        break
    end
    Emu = zeros(2,1);
    Emu(1) = int1d(mu .* pmu_row, dmu); 
    Emu(2) = int1d(mu .* pmu_col, dmu);  
    Ex = zeros(2,1);
    Ex(1) = int1d(x .* pmu_row, dmu);
    Ex(2) = int1d(x .* pmu_col, dmu);

    if mod(count, nC1s)==0
        disp(count / nC1s);
        EmuT(count / nC1s + 1, :) = Emu';
        ExT(count / nC1s + 1, :) = Ex';
        pMuT_row(count / nC1s + 1, :) = pmu_row;
        pMuT_col(count / nC1s + 1, :) = pmu_col;
    end

    % update the probability distribution for the first population using
    % the finite difference method
    dmu_pmu_row = zeros(1, nmu);
    dmu_pmu_row(:, 2:end-1) = (pmu_row(:, 3:end) - pmu_row(:, 1:end-2)) / (2 * dmu);
    y_col = Ex(2);
    f_row = (y_col - mu) / (lambda + t + 1);
    dmu_df_row = 0 - 1 / (lambda + t + 1);
    dt_pmu_row = -( dmu_pmu_row .* f_row + pmu_row .* dmu_df_row) ;
    pmu_row = pmu_row + dt_pmu_row * dt;
    pmu_row = normalizePDF(pmu_row, dmu);

    % update the probability distribution for the second population using
    % the finite difference method
    dmu_pmu_col = zeros(1, nmu);
    dmu_pmu_col(:, 2:end-1) = (pmu_col(:, 3:end) - pmu_col(:, 1:end-2)) / (2 * dmu);
    y_row = Ex(1);
    f_col = (y_row - mu) / (lambda + t + 1);
    dmu_df_col = 0 - 1 / (lambda + t + 1);
    dt_pmu_col = -( dmu_pmu_col .* f_col + pmu_col .* dmu_df_col);
    pmu_col = pmu_col + dt_pmu_col * dt;
    pmu_col(1, 1) = pmu_col(1, 2);
    pmu_col(1, end) = pmu_col(1, end - 1);
    pmu_col = max(pmu_col, 0);
    pmu_col = normalizePDF(pmu_col, dmu);
    
    t = t + dt;
    count = count + 1;
end

figure;
plot(1-EmuT(1:T,1), 'LineWidth',1.5,'Color','b','LineStyle','-.');
hold on
plot(1-ExT(1:T,1), 'LineWidth',1.5,'Color','r','LineStyle','-');
legend({'Mean Belief $\bar{\mu}_{2S}$', 'Mean Probability $\bar{x}_{1S}$' }, ...
    Interpreter="latex");
xlim([1 T]);
ylim([0 1]);
xlabel('Time $t$',Interpreter='latex');
set(gca, 'FontSize', 14);
axis tight
hold off




function A = int2d(f, dxi)
Nx1 = size(f,1);
Nx2 = size(f,2);
w = 4*ones(Nx1, Nx2);
w(1,:)=2;
w(end,:)=2;
w(:,1)=2;
w(:,end)=2;
w(1,1)=1;
w(1,end)=1;
w(end,1)=1;
w(end,end)=1;
A = sum(sum(w.*f))*dxi^2/4;
end


function A = int1d(f, dxi)
A=sum(f(1,1:end-1)+f(1,2:end))*dxi/2;
end

function A = normalizePDF(x, dx)
x(1, 1) = x(1, 2);
x(1, end) = x(1, end - 1);
x = max(x, 0);
A = x ./ int1d(x, dx);
end


