import os
import pickle
from collections import OrderedDict
import json
from dassl.data.datasets import DATASET_REGISTRY, Datum, DatasetBase
from dassl.utils import listdir_nohidden, mkdir_if_missing
import torch
from .oxford_pets import OxfordPets
from tqdm import tqdm


@DATASET_REGISTRY.register()
class ImageNet21K(DatasetBase):
    dataset_dir = "imagenet21k_resized"

    def __init__(self, cfg):
        root = os.path.abspath(os.path.expanduser(cfg.DATASET.ROOT))
        self.dataset_dir = os.path.join(root, self.dataset_dir)
        self.image_dir = self.dataset_dir
        self.preprocessed = os.path.join(self.dataset_dir, "preprocessed.pkl")
        self.split_fewshot_dir = os.path.join(self.dataset_dir, "split_fewshot")
        mkdir_if_missing(self.split_fewshot_dir)

        # train_text_file = os.path.join(self.dataset_dir, "train_classnames.txt")
        # self.train_classnames = self.read_classnames(train_text_file)
        # test_text_file = os.path.join(self.dataset_dir, "test_classnames.txt")
        # self.test_classnames = self.read_classnames(test_text_file)

        self.semantic_tree = torch.load(os.path.join(self.dataset_dir, "imagenet21k_miil_tree.pth"))
        class_tree_list = self.semantic_tree['class_tree_list']
        tree_depth = max([len(tree) for tree in class_tree_list])
        class_tree_list = [tree + ([-1] * (tree_depth - len(tree))) for tree in
                           class_tree_list]  # pad each tree to depth
        self.class_tree_list = torch.tensor(class_tree_list)

        self.train_classnames = self.semantic_tree['class_description']
        self.test_classnames = self.semantic_tree['class_description']

        if os.path.exists(self.preprocessed):
            with open(self.preprocessed, "rb") as f:
                preprocessed = pickle.load(f)
                train = preprocessed["train"]
                test = preprocessed["test"]
        else:
            train = self.read_data(self.train_classnames, "train")
            # Follow standard practice to perform evaluation on the val set
            # Also used as the val set (so evaluate the last-step model)
            test = self.read_data(self.test_classnames, "val")

            preprocessed = {"train": train, "test": test}
            with open(self.preprocessed, "wb") as f:
                pickle.dump(preprocessed, f, protocol=pickle.HIGHEST_PROTOCOL)

        num_shots = cfg.DATASET.NUM_SHOTS
        if num_shots >= 1:
            seed = cfg.SEED
            preprocessed = os.path.join(self.split_fewshot_dir, f"shot_{num_shots}-seed_{seed}.pkl")

            if os.path.exists(preprocessed):
                print(f"Loading preprocessed few-shot data from {preprocessed}")
                with open(preprocessed, "rb") as file:
                    data = pickle.load(file)
                    train = data["train"]
            else:
                train = self.generate_fewshot_dataset(train, num_shots=num_shots)
                data = {"train": train}
                print(f"Saving preprocessed few-shot data to {preprocessed}")
                with open(preprocessed, "wb") as file:
                    pickle.dump(data, file, protocol=pickle.HIGHEST_PROTOCOL)

        subsample = cfg.DATASET.SUBSAMPLE_CLASSES
        train, test = OxfordPets.subsample_classes(train, test, subsample=subsample)

        super().__init__(train_x=train, val=test, test=test)

        with open(os.path.join(self.dataset_dir, "img_num_per_folder.json"), 'r') as f:
            self.img_num_per_folder = json.load(f)
        self.img_num_per_cname = {self.train_classnames[folder]: num for folder, num in self.img_num_per_folder.items()
                                  if folder in self.train_classnames}
        self.img_num_per_label = []
        for label in range(len(self.lab2cname)):
            cname = self.lab2cname[label]
            self.img_num_per_label.append(self.img_num_per_cname[cname])
        self.img_freq_per_label = [n / sum(self.img_num_per_label) for n in self.img_num_per_label]

    @staticmethod
    def read_classnames(text_file):
        """Return a dictionary containing
        key-value pairs of <folder name>: <class name>.
        """
        classnames = OrderedDict()
        with open(text_file, "r") as f:
            lines = f.readlines()
            for line in lines:
                line = line.strip().split(" ")
                folder = line[0]
                classname = " ".join(line[1:])
                classnames[folder] = classname
        return classnames

    def read_data(self, classnames, split_dir):
        split_dir = os.path.join(self.image_dir, "imagenet21k_" + split_dir)
        folders = sorted(f.name for f in os.scandir(split_dir) if f.is_dir())
        items = []

        for label, folder in tqdm(enumerate(folders)):
            imnames = listdir_nohidden(os.path.join(split_dir, folder))
            classname = classnames[folder]
            for imname in imnames:
                impath = os.path.join(split_dir, folder, imname)
                item = Datum(impath=impath, label=label, classname=classname)
                items.append(item)

        return items
