# coding=utf-8
# Copyright 2023 The Soar Neurips2023 Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Generates plots."""
import os

from absl import app
from absl import flags
import h5py
import matplotlib.pyplot as plt
import numpy as np
import scipy.ndimage
from scipy.stats import pearsonr
import utils

_HDF5 = flags.DEFINE_string("hdf5", None, "Path to hdf5 of dataset.")
_NPY_DIR = flags.DEFINE_string("npy_dir", None, "Path to Numpy directory.")
_NUM_CENTERS = flags.DEFINE_integer("num_centers", 2000, "# k-means centers.")
_ETA = flags.DEFINE_float("eta", 2.5, "AVQ eta.")


def smooth(signal, amt):
  if amt == 0:
    return signal
  return scipy.ndimage.uniform_filter1d(signal, amt, mode="nearest")


def save_plot(name, vectorize=True):
  dpi = 300 if vectorize else 1200
  plt.savefig(name + ".png", dpi=dpi, bbox_inches="tight", pad_inches=0.02)
  if vectorize:
    plt.savefig(name + ".pdf", bbox_inches="tight", pad_inches=0.02)


def get_data():
  """Computes things needed for plots."""
  hdf5 = h5py.File(_HDF5.value, "r")
  ds = utils.normalize(hdf5["train"][()])
  qs = utils.normalize(hdf5["test"][()])
  gt = hdf5["neighbors"][()]
  print("Dataset shape:", ds.shape)
  print("Query shape:", qs.shape)
  print("Ground truth shape:", gt.shape, flush=True)

  eta = _ETA.value
  num_centers = _NUM_CENTERS.value

  # Initialize each as 3 empty lists, because we have 3 tokenizations.
  ranks, diffs, norms = [[], [], []], [[], [], []], [[], [], []]
  for outer_idx in range(100):
    print(f"K-means try {outer_idx+1}", flush=True)
    selects = np.random.choice(range(len(ds)), len(ds) // 3, replace=False)
    orig_centers, toke = utils.train_kmeans(ds[selects], num_centers)
    print("Performing spilled assignment...", flush=True)
    toke, toke2 = utils.redo_assignment(orig_centers, ds)
    print("# empty partitions:", num_centers - len(set(toke)))
    print("Updating centers...", flush=True)
    centers = utils.compute_avq_centers(ds, orig_centers, toke, eta)
    print("SOAR...", flush=True)
    toke3 = utils.soar_assign(ds, centers, toke, 1, False)

    for i, t in enumerate([toke, toke2, toke3]):
      ranks[i].append(utils.get_centroid_ranks(centers, t, qs, gt))
      diffs[i].append(utils.compute_score_diffs(ds, centers, t, qs, gt))
      norms[i].append(np.linalg.norm(ds - centers[t], axis=1)[gt])

  for cur, name in zip([ranks, diffs, norms], ["ranks", "diffs", "norms"]):
    for i, ll in enumerate(cur):
      fname = f"/tmp/npys/{name}{i+1}.npy"
      np.save(fname, np.concatenate([x.flatten() for x in ll]))


def group_data(npy_dir):
  """Performs grouping operations needed for plots."""
  print("Loading data...", flush=True)
  r1 = np.load(os.path.join(npy_dir, "ranks1.npy"))
  r2 = np.load(os.path.join(npy_dir, "ranks2.npy"))
  r3 = np.load(os.path.join(npy_dir, "ranks3.npy"))
  d1 = np.load(os.path.join(npy_dir, "diffs1.npy"))
  d2 = np.load(os.path.join(npy_dir, "diffs2.npy"))
  d3 = np.load(os.path.join(npy_dir, "diffs3.npy"))
  n1 = np.load(os.path.join(npy_dir, "norms1.npy"))
  n2 = np.load(os.path.join(npy_dir, "norms2.npy"))
  n3 = np.load(os.path.join(npy_dir, "norms3.npy"))

  print("Mean grouping...", flush=True)
  _, mean1 = utils.group_rank_data(r1, r2, np.mean)
  _, mean2 = utils.group_rank_data(r1, r3, np.mean)
  print("Median grouping...", flush=True)
  _, med1 = utils.group_rank_data(r1, r2, np.median)
  _, med2 = utils.group_rank_data(r1, r3, np.median)
  np.save(os.path.join(npy_dir, "grouped_means1.npy"), mean1)
  np.save(os.path.join(npy_dir, "grouped_means2.npy"), mean2)
  np.save(os.path.join(npy_dir, "grouped_medians1.npy"), med1)
  np.save(os.path.join(npy_dir, "grouped_medians2.npy"), med2)

  for i, (r, d, n) in enumerate([(r1, d1, n1), (r2, d2, n2), (r3, d3, n3)]):
    print(f"Grouping {i+1}...", flush=True)
    _, r2score = utils.group_rank_data(r, d, np.mean)
    _, r2norm = utils.group_rank_data(r, n, np.mean)
    _, r2angle = utils.group_rank_data(r, d / n, np.mean)
    np.save(os.path.join(npy_dir, f"rank-score{i+1}.npy"), r2score)
    np.save(os.path.join(npy_dir, f"rank-norm{i+1}.npy"), r2norm)
    np.save(os.path.join(npy_dir, f"rank-angle{i+1}.npy"), r2angle)


def plot_data(npy_dir):
  """Generates plots."""
  num_centers = _NUM_CENTERS.value

  print("Loading data...", flush=True)
  r1 = np.load(os.path.join(npy_dir, "ranks1.npy"))
  r2 = np.load(os.path.join(npy_dir, "ranks2.npy"))
  r3 = np.load(os.path.join(npy_dir, "ranks3.npy"))
  d1 = np.load(os.path.join(npy_dir, "diffs1.npy"))
  d2 = np.load(os.path.join(npy_dir, "diffs2.npy"))
  d3 = np.load(os.path.join(npy_dir, "diffs3.npy"))
  n1 = np.load(os.path.join(npy_dir, "norms1.npy"))
  n2 = np.load(os.path.join(npy_dir, "norms2.npy"))
  n3 = np.load(os.path.join(npy_dir, "norms3.npy"))

  mean1 = np.load(os.path.join(npy_dir, "grouped_means1.npy"))
  mean2 = np.load(os.path.join(npy_dir, "grouped_means2.npy"))
  med1 = np.load(os.path.join(npy_dir, "grouped_medians1.npy"))
  med2 = np.load(os.path.join(npy_dir, "grouped_medians2.npy"))
  r1xs = sorted(list(set(r1)))
  # Ensure stride is odd to get more variety in samples
  stride = 1 if len(r1) < 10**6 else 2 * (len(r1) // 10**6) // 2 + 1
  smooth_amt = 7

  for i, r in enumerate([r1, r2, r3]):
    print(f"Number of empty ranks{i+1}:", num_centers - len(set(r)), flush=True)

  print("Plotting score-norm and score-angle", flush=True)
  plt.figure(figsize=(4, 3))
  plt.gca().set_xlabel("Partitioning Residual Norm $\\Vert r\\Vert$")
  plt.gca().set_ylabel("Quantized Score Error $\\langle q,r\\rangle$")
  plt.scatter(n1[::stride], d1[::stride], s=0.1, linewidths=0)
  plt.text(0.53, 0.58, f"Corr={pearsonr(n1,d1).statistic:.3f}")
  save_plot("out/score-norm", False)
  plt.close()

  plt.figure(figsize=(4, 3))
  plt.gca().set_xlabel("Query-Residual Angle $\\cos\\theta$")
  plt.gca().set_ylabel("Quantized Score Error $\\langle q,r\\rangle$")
  plt.scatter(d1[::stride] / n1[::stride], d1[::stride], s=0.1, linewidths=0)
  plt.text(0.02, 0.58, f"Corr={pearsonr(d1/n1,d1).statistic:.3f}")
  save_plot("out/score-angle", False)
  plt.close()

  print("Plotting rank-rank...", flush=True)
  plt.figure(figsize=(4, 3))
  plt.gca().set_xlabel("Rank$(q,C_{\\pi(x)},C)$")
  plt.gca().set_ylabel("Mean Rank$(q,C_{\\pi'(x)},C)$")
  plt.plot(r1xs, smooth(mean1, smooth_amt), label="No SOAR")
  plt.plot(r1xs, smooth(mean2, smooth_amt), label="SOAR")
  plt.legend()
  save_plot("out/grouped_means")
  plt.close()

  plt.figure(figsize=(4, 3))
  plt.gca().set_xlabel("Rank$(q,C_{\\pi(x)},C)$")
  plt.gca().set_ylabel("Median Rank$(q,C_{\\pi'(x)},C)$")
  plt.plot(r1xs, smooth(med1, smooth_amt), label="No SOAR")
  plt.plot(r1xs, smooth(med2, smooth_amt), label="SOAR")
  plt.legend()
  save_plot("out/grouped_medians")
  plt.close()

  print("Plotting rank-rank scatter...", flush=True)
  for s1, s2, name in [(r1, r2, "r1-r2"), (r1, r3, "r1-r3")]:
    plt.xlim([0, num_centers])
    plt.ylim([0, num_centers])
    plt.scatter(s1[::23], s2[::23], s=0.1, linewidths=0)
    save_plot(f"out/{name}-scatter", False)
    plt.close()

  print("Plotting angle-angle scatter...", flush=True)
  for a, name in [(d2 / n2, "a1-a2"), (d3 / n3, "a1-a3")]:
    plt.figure(figsize=(4, 2.25) if name == "a1-a2" else (4, 3))
    plt.gca().set_xlabel("$\\cos\\theta$")
    plt.gca().set_ylabel("$\\cos\\theta'$")
    plt.scatter(d1[::stride] / n1[::stride], a[::stride], s=0.1, linewidths=0)
    plt.text(0.4, -0.3, f"Corr={pearsonr(d1/n1, a).statistic:.3f}")
    save_plot(f"out/{name}-scatter", False)
    plt.close()

  plt.figure(figsize=(4, 2.25))
  plt.gca().set_xlabel("$\\cos\\theta_1$")
  plt.gca().set_ylabel("$\\cos\\theta_2$")
  run1_cos = (d1 / n1)[: 10**6]
  run2_cos = (d1 / n1)[10**6 : 2 * 10**6]
  plt.scatter(run1_cos, run2_cos, s=0.1, linewidths=0)
  plt.text(0.4, -0.32, f"Corr={pearsonr(run1_cos, run2_cos).statistic:.3f}")
  save_plot("out/angle-scatter", False)
  plt.close()

  print("Plotting rank against score diff, norm, and angle...", flush=True)
  for i, (r, d, n) in enumerate([(r1, d1, n1), (r2, d2, n2), (r3, d3, n3)]):
    cur_x = sorted(list(set(r)))
    for name in [f"rank-score{i+1}", f"rank-norm{i+1}", f"rank-angle{i+1}"]:
      plt.figure(figsize=(3, 2.25))
      plt.gca().set_xlabel("Rank$(q,C_{\\pi(x)},C)$")
      plt.gca().set_ylabel("Mean $\\langle q,r\\rangle$")
      plt.plot(cur_x, np.load(os.path.join(npy_dir, name + ".npy")))
      save_plot(f"out/{name}")
      plt.close()

    print(f"({i+1}) Score-Norm R:", pearsonr(d, n), flush=True)
    print(f"({i+1}) Score-Angle R:", pearsonr(d, d / n))

  print("1-2 Rank-rank R:", pearsonr(r1, r2), flush=True)
  print("1-2 Score-score R:", pearsonr(d1, d2))
  print("1-2 Norm-norm R:", pearsonr(n1, n2))
  print("1-2 Angle-angle R:", pearsonr(d1 / n1, d2 / n2))
  print("1-3 Rank-rank R:", pearsonr(r1, r3), flush=True)
  print("1-3 Score-score R:", pearsonr(d1, d3))
  print("1-3 Norm-norm R:", pearsonr(n1, n3))
  print("1-3 Angle-angle R:", pearsonr(d1 / n1, d3 / n3))


def main(argv):
  del argv  # Unused.
  # get_data()
  # group_data(_NPY_DIR.value)
  plot_data(_NPY_DIR.value)


if __name__ == "__main__":
  app.run(main)
