;; gorilla-repl.fileformat = 1

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [[ a] [ 0 ]
          [ a]
          (let [
                a (sample (poisson 51.4))
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 23.259999999999998)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 232.6)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 45)
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 42.08)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 420.8)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 98)
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 30.080000000000002)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 300.8)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 73)
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 8.559999999999999)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 85.6)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 38)
               ]
            [ a ]
          )
         ]
    a
    )
  )

)


(def model_name "population_modified2000")
(def outfile "output/population_modified2000_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:rmh [:alpha 0.8 :sigma 2]]
    [:smc []]
    [:smc [:number-of-particles 1000]]
    [:pgibbs []]
    [:pgibbs [:number-of-particles 1000]]
    [:ipmcmc []]
    [:ipmcmc [:number-of-particles 1000 :number-of-nodes 32]]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with %s samples and options %s" method num_samples options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)




;; @@
;; ->
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 245.469086 ms
;;; Empirical mean: 193.37585182217978
;;; Empirical variance: 168.85813653129676
;;; Empirical std: 12.994542567220162
;;; Empirical skewness: -0.44856173151808054
;;; Empirical kurtosis: 5.154962826890492
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 208.943206 ms
;;; Empirical mean: 194.0952249398111
;;; Empirical variance: 163.90277801887024
;;; Empirical std: 12.80245203150046
;;; Empirical skewness: -0.21559404993264392
;;; Empirical kurtosis: 4.31277235399187
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 226.06181 ms
;;; Empirical mean: 194.02872635419544
;;; Empirical variance: 155.26117562044627
;;; Empirical std: 12.460384248507197
;;; Empirical skewness: -0.27693697015281366
;;; Empirical kurtosis: 4.765800800080494
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 220.003526 ms
;;; Empirical mean: 193.75671576940772
;;; Empirical variance: 156.55898736831304
;;; Empirical std: 12.512353390482266
;;; Empirical skewness: -0.2757340270033187
;;; Empirical kurtosis: 4.9163286583192045
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 227.300954 ms
;;; Empirical mean: 193.89932694469732
;;; Empirical variance: 165.04905551545957
;;; Empirical std: 12.847141920110463
;;; Empirical skewness: -0.1553851474048621
;;; Empirical kurtosis: 4.343239714774752
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 217.995947 ms
;;; Empirical mean: 194.28599547137264
;;; Empirical variance: 160.29736678106912
;;; Empirical std: 12.660859638313234
;;; Empirical skewness: -0.19471073784980208
;;; Empirical kurtosis: 4.41582629479481
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 208.880021 ms
;;; Empirical mean: 193.75129728056683
;;; Empirical variance: 166.35524518870898
;;; Empirical std: 12.8978775458875
;;; Empirical skewness: -0.2758171581737092
;;; Empirical kurtosis: 4.410250181105026
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 209.086768 ms
;;; Empirical mean: 194.08207070126147
;;; Empirical variance: 154.7144000238251
;;; Empirical std: 12.438424338469286
;;; Empirical skewness: -0.3865861534886763
;;; Empirical kurtosis: 4.510183012888779
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 210.398605 ms
;;; Empirical mean: 193.87231538481652
;;; Empirical variance: 171.04425150670784
;;; Empirical std: 13.078388719819726
;;; Empirical skewness: -0.22896293430726858
;;; Empirical kurtosis: 4.09733832921914
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 202.827121 ms
;;; Empirical mean: 194.13638317677575
;;; Empirical variance: 161.30898847458474
;;; Empirical std: 12.700747555737998
;;; Empirical skewness: -0.22812817971358695
;;; Empirical kurtosis: 4.146925364796579
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 215.974572 ms
;;; Empirical mean: 194.53248796541658
;;; Empirical variance: 160.98498373771847
;;; Empirical std: 12.687985803023208
;;; Empirical skewness: -0.21514514230260368
;;; Empirical kurtosis: 4.102011054430247
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 209.528269 ms
;;; Empirical mean: 192.72129613989324
;;; Empirical variance: 181.1706343897743
;;; Empirical std: 13.459964130330151
;;; Empirical skewness: 0.01406595617288372
;;; Empirical kurtosis: 3.823111103239725
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 214.151437 ms
;;; Empirical mean: 194.45715144619035
;;; Empirical variance: 161.21275561006632
;;; Empirical std: 12.696958518088744
;;; Empirical skewness: -0.21875193517831623
;;; Empirical kurtosis: 4.60380851951602
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 220.877034 ms
;;; Empirical mean: 193.74615515170427
;;; Empirical variance: 144.81765867348517
;;; Empirical std: 12.034020885534693
;;; Empirical skewness: -0.3284365911798337
;;; Empirical kurtosis: 4.884191102194251
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 225.637445 ms
;;; Empirical mean: 194.4461196477225
;;; Empirical variance: 159.99770358411675
;;; Empirical std: 12.64901986653973
;;; Empirical skewness: -0.2603983364705755
;;; Empirical kurtosis: 4.591576503963969
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 213.042166 ms
;;; Empirical mean: 193.5188435652561
;;; Empirical variance: 161.02272401209592
;;; Empirical std: 12.689472960375301
;;; Empirical skewness: 0.007817192198793226
;;; Empirical kurtosis: 4.345641700750577
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 208.292208 ms
;;; Empirical mean: 194.82804465540528
;;; Empirical variance: 176.3263883574589
;;; Empirical std: 13.278794687676246
;;; Empirical skewness: -0.29558265026406655
;;; Empirical kurtosis: 4.200094314554897
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 203.270102 ms
;;; Empirical mean: 193.98682423467812
;;; Empirical variance: 165.8312714335879
;;; Empirical std: 12.87754912371092
;;; Empirical skewness: -0.25119931812312285
;;; Empirical kurtosis: 4.466755839067096
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 206.410139 ms
;;; Empirical mean: 192.64632952778825
;;; Empirical variance: 183.15743898383442
;;; Empirical std: 13.533567119715128
;;; Empirical skewness: -0.015217797286221734
;;; Empirical kurtosis: 3.8702248839537607
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 214.526736 ms
;;; Empirical mean: 194.34089296384138
;;; Empirical variance: 154.73824834287404
;;; Empirical std: 12.439382956677314
;;; Empirical skewness: -0.23928289715366538
;;; Empirical kurtosis: 5.002701891555325
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 178.727297 ms
;;; Empirical mean: 196.697
;;; Empirical variance: 105.49919100000004
;;; Empirical std: 10.271279910507747
;;; Empirical skewness: 0.4068464507759915
;;; Empirical kurtosis: 2.845742399816354
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 176.782451 ms
;;; Empirical mean: 193.84400000000008
;;; Empirical variance: 120.69166399999999
;;; Empirical std: 10.985975787339056
;;; Empirical skewness: 0.22394447514110336
;;; Empirical kurtosis: 2.962836346611267
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 176.38401 ms
;;; Empirical mean: 196.82099999999997
;;; Empirical variance: 131.51295899999997
;;; Empirical std: 11.467909966510897
;;; Empirical skewness: 0.09036541074576858
;;; Empirical kurtosis: 2.9044805144825836
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 179.690351 ms
;;; Empirical mean: 190.566
;;; Empirical variance: 181.57164400000008
;;; Empirical std: 13.474852281194035
;;; Empirical skewness: -0.2348369072267294
;;; Empirical kurtosis: 2.917032541507047
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 190.129125 ms
;;; Empirical mean: 189.27199999999996
;;; Empirical variance: 319.264016
;;; Empirical std: 17.86796059991179
;;; Empirical skewness: -1.7388828353929382
;;; Empirical kurtosis: 7.268034317666335
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 181.133524 ms
;;; Empirical mean: 196.98899999999992
;;; Empirical variance: 179.86687899999993
;;; Empirical std: 13.41144582064141
;;; Empirical skewness: 0.744480907368941
;;; Empirical kurtosis: 3.7371875554318588
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 173.499456 ms
;;; Empirical mean: 168.35700000000006
;;; Empirical variance: 804.7755510000006
;;; Empirical std: 28.368566248578738
;;; Empirical skewness: 0.05585740405351611
;;; Empirical kurtosis: 1.6097893166216697
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 188.295795 ms
;;; Empirical mean: 193.86
;;; Empirical variance: 181.93239999999997
;;; Empirical std: 13.488231907852118
;;; Empirical skewness: -0.07953623102000698
;;; Empirical kurtosis: 2.4764699485566024
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 180.920086 ms
;;; Empirical mean: 168.29900000000004
;;; Empirical variance: 937.5375990000005
;;; Empirical std: 30.619235767732683
;;; Empirical skewness: 0.3430271770002267
;;; Empirical kurtosis: 1.5462724855558039
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 178.502486 ms
;;; Empirical mean: 194.36400000000003
;;; Empirical variance: 148.48350399999998
;;; Empirical std: 12.18538074907797
;;; Empirical skewness: 0.04935964266738846
;;; Empirical kurtosis: 2.9518226296379333
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 188.331898 ms
;;; Empirical mean: 187.58600000000007
;;; Empirical variance: 334.0386040000002
;;; Empirical std: 18.276723010430512
;;; Empirical skewness: -1.3949658148124082
;;; Empirical kurtosis: 5.063896760324056
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 187.903842 ms
;;; Empirical mean: 192.072
;;; Empirical variance: 172.86281599999998
;;; Empirical std: 13.147730450537841
;;; Empirical skewness: -0.7480092252216329
;;; Empirical kurtosis: 4.294440865040725
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 206.053659 ms
;;; Empirical mean: 193.88
;;; Empirical variance: 228.7956
;;; Empirical std: 15.125990876633505
;;; Empirical skewness: -0.8171002111453539
;;; Empirical kurtosis: 4.4120680379918635
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 209.921192 ms
;;; Empirical mean: 197.56100000000006
;;; Empirical variance: 153.182279
;;; Empirical std: 12.376682875471925
;;; Empirical skewness: 0.835572701074803
;;; Empirical kurtosis: 4.229060569943258
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 226.014911 ms
;;; Empirical mean: 145.141
;;; Empirical variance: 80.043119
;;; Empirical std: 8.946682010667418
;;; Empirical skewness: -0.27838344118617286
;;; Empirical kurtosis: 3.6154365040941627
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 207.404764 ms
;;; Empirical mean: 191.89399999999995
;;; Empirical variance: 145.44076399999994
;;; Empirical std: 12.059882420654024
;;; Empirical skewness: -0.31075889859026484
;;; Empirical kurtosis: 2.950762641996316
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 179.969485 ms
;;; Empirical mean: 193.54000000000008
;;; Empirical variance: 154.2304
;;; Empirical std: 12.418953257018082
;;; Empirical skewness: -0.2692555406366402
;;; Empirical kurtosis: 2.84544931364898
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 182.539476 ms
;;; Empirical mean: 193.02699999999996
;;; Empirical variance: 169.56827100000004
;;; Empirical std: 13.021838234289353
;;; Empirical skewness: 0.18567261296391427
;;; Empirical kurtosis: 3.106023420648873
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 170.669702 ms
;;; Empirical mean: 197.03
;;; Empirical variance: 131.4051
;;; Empirical std: 11.463206357734297
;;; Empirical skewness: 0.5101170978946636
;;; Empirical kurtosis: 2.722737393615975
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 180.589332 ms
;;; Empirical mean: 191.56100000000004
;;; Empirical variance: 111.49827900000001
;;; Empirical std: 10.559274548945112
;;; Empirical skewness: 0.3462266742098361
;;; Empirical kurtosis: 3.2082959646936127
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 259.904785 ms
;;; Empirical mean: 192.84900000000002
;;; Empirical variance: 130.94819900000002
;;; Empirical std: 11.443259981316514
;;; Empirical skewness: -0.10673257758050815
;;; Empirical kurtosis: 3.5154751315827606
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 256.437221 ms
;;; Empirical mean: 199.42200000000003
;;; Empirical variance: 175.39991600000002
;;; Empirical std: 13.243863333634941
;;; Empirical skewness: -0.34342354011014564
;;; Empirical kurtosis: 2.920016717425648
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 234.568858 ms
;;; Empirical mean: 197.86199999999997
;;; Empirical variance: 146.348956
;;; Empirical std: 12.097477257676495
;;; Empirical skewness: 0.32808385260752976
;;; Empirical kurtosis: 2.361827171986464
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 229.658454 ms
;;; Empirical mean: 179.03900000000002
;;; Empirical variance: 804.5274790000004
;;; Empirical std: 28.36419360743401
;;; Empirical skewness: -0.5332906107368361
;;; Empirical kurtosis: 1.9615231016397288
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 236.412701 ms
;;; Empirical mean: 193.58599999999996
;;; Empirical variance: 132.784604
;;; Empirical std: 11.52322020964626
;;; Empirical skewness: 0.3391224904152987
;;; Empirical kurtosis: 2.2133398731717437
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 235.952694 ms
;;; Empirical mean: 173.21999999999997
;;; Empirical variance: 699.5675999999996
;;; Empirical std: 26.449340256422268
;;; Empirical skewness: -0.17927152735612845
;;; Empirical kurtosis: 1.823548079787661
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 256.477453 ms
;;; Empirical mean: 194.052
;;; Empirical variance: 178.5432959999999
;;; Empirical std: 13.36200942972276
;;; Empirical skewness: 0.8177823662276298
;;; Empirical kurtosis: 4.0013207677751765
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 235.722577 ms
;;; Empirical mean: 190.089
;;; Empirical variance: 117.19107899999999
;;; Empirical std: 10.825482852972424
;;; Empirical skewness: 0.2719223367541197
;;; Empirical kurtosis: 2.449843118462808
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 234.208914 ms
;;; Empirical mean: 161.281
;;; Empirical variance: 858.5720390000005
;;; Empirical std: 29.301399949490477
;;; Empirical skewness: 0.32136630057098114
;;; Empirical kurtosis: 1.5016553633301357
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 232.548087 ms
;;; Empirical mean: 194.71100000000007
;;; Empirical variance: 299.4074789999999
;;; Empirical std: 17.303395013696008
;;; Empirical skewness: 0.39732454039845944
;;; Empirical kurtosis: 2.585852459407066
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 238.879116 ms
;;; Empirical mean: 194.446
;;; Empirical variance: 123.105084
;;; Empirical std: 11.095273047563994
;;; Empirical skewness: 0.5110803077294936
;;; Empirical kurtosis: 2.866648013620875
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 235.874739 ms
;;; Empirical mean: 189.37300000000008
;;; Empirical variance: 108.22387100000007
;;; Empirical std: 10.403070267954556
;;; Empirical skewness: 0.5360207367104021
;;; Empirical kurtosis: 2.8569827096166622
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 247.156941 ms
;;; Empirical mean: 190.32899999999992
;;; Empirical variance: 97.27275900000001
;;; Empirical std: 9.862695321259803
;;; Empirical skewness: -0.30376428586749715
;;; Empirical kurtosis: 4.752234784742749
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 239.003549 ms
;;; Empirical mean: 196.53900000000002
;;; Empirical variance: 134.42847900000004
;;; Empirical std: 11.594329605458007
;;; Empirical skewness: -0.382436382628881
;;; Empirical kurtosis: 2.3790870674081472
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 243.619611 ms
;;; Empirical mean: 189.99
;;; Empirical variance: 207.79390000000004
;;; Empirical std: 14.415058099085138
;;; Empirical skewness: -0.35441742480426586
;;; Empirical kurtosis: 2.617092943266831
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 236.226646 ms
;;; Empirical mean: 194.25100000000003
;;; Empirical variance: 115.55199899999995
;;; Empirical std: 10.749511570299367
;;; Empirical skewness: -0.5926375824822593
;;; Empirical kurtosis: 4.4796104659046785
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 245.073009 ms
;;; Empirical mean: 197.585
;;; Empirical variance: 85.71877500000005
;;; Empirical std: 9.25844344368966
;;; Empirical skewness: -0.2627920327868282
;;; Empirical kurtosis: 3.6262802952039226
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 237.682696 ms
;;; Empirical mean: 195.06500000000003
;;; Empirical variance: 227.31477500000005
;;; Empirical std: 15.076961729738523
;;; Empirical skewness: -0.6587047761790565
;;; Empirical kurtosis: 5.242326528032058
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 233.25206 ms
;;; Empirical mean: 196.68900000000005
;;; Empirical variance: 215.26027900000005
;;; Empirical std: 14.671751054322046
;;; Empirical skewness: 0.6272327761707274
;;; Empirical kurtosis: 2.9015191763435495
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 239.040444 ms
;;; Empirical mean: 195.46299999999997
;;; Empirical variance: 162.762631
;;; Empirical std: 12.757845860489144
;;; Empirical skewness: -0.2620319773917501
;;; Empirical kurtosis: 2.983338621922523
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 271.420388 ms
;;; Empirical mean: 192.90009246998434
;;; Empirical variance: 172.24719558937798
;;; Empirical std: 13.124297908436017
;;; Empirical skewness: -0.20440387478329775
;;; Empirical kurtosis: 4.177727974642283
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 329.087717 ms
;;; Empirical mean: 194.63596296834416
;;; Empirical variance: 177.56137025698996
;;; Empirical std: 13.325215580131902
;;; Empirical skewness: -0.2564693002983823
;;; Empirical kurtosis: 3.8361195797140493
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 257.122854 ms
;;; Empirical mean: 193.65112298249477
;;; Empirical variance: 155.32863180469766
;;; Empirical std: 12.463090780568745
;;; Empirical skewness: -0.061984052779497464
;;; Empirical kurtosis: 4.281773438156546
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 250.560037 ms
;;; Empirical mean: 194.97356045717837
;;; Empirical variance: 155.5166004116238
;;; Empirical std: 12.470629511441025
;;; Empirical skewness: -0.21373546440437768
;;; Empirical kurtosis: 5.103522474636968
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 253.200847 ms
;;; Empirical mean: 194.74417247894704
;;; Empirical variance: 160.51612573790942
;;; Empirical std: 12.669495875444667
;;; Empirical skewness: -0.30897625198492296
;;; Empirical kurtosis: 4.602293974590483
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 260.006301 ms
;;; Empirical mean: 193.6230269484354
;;; Empirical variance: 157.37624986087658
;;; Empirical std: 12.544969105616666
;;; Empirical skewness: -0.10955125093492768
;;; Empirical kurtosis: 4.696485600657638
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 265.763783 ms
;;; Empirical mean: 192.79065469684278
;;; Empirical variance: 179.23022245165026
;;; Empirical std: 13.387689212543375
;;; Empirical skewness: -0.2598128805799897
;;; Empirical kurtosis: 4.257925788256044
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 308.841968 ms
;;; Empirical mean: 193.9665329141544
;;; Empirical variance: 170.45265239270987
;;; Empirical std: 13.055751697727322
;;; Empirical skewness: -0.2927382205713607
;;; Empirical kurtosis: 4.003241893932992
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 269.173465 ms
;;; Empirical mean: 193.21746488203192
;;; Empirical variance: 162.66329741947973
;;; Empirical std: 12.753952227426593
;;; Empirical skewness: -0.17052704562638193
;;; Empirical kurtosis: 3.8307399311290053
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 304.456988 ms
;;; Empirical mean: 193.46075648799726
;;; Empirical variance: 161.57907483978803
;;; Empirical std: 12.711375804364689
;;; Empirical skewness: -0.18862156340404054
;;; Empirical kurtosis: 4.149170290636227
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 266.38455 ms
;;; Empirical mean: 195.32365220518716
;;; Empirical variance: 155.4937566505555
;;; Empirical std: 12.469713575321428
;;; Empirical skewness: -0.23846331509504223
;;; Empirical kurtosis: 4.440209720752811
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 266.063967 ms
;;; Empirical mean: 194.2832273272825
;;; Empirical variance: 164.85237241880387
;;; Empirical std: 12.839484896942084
;;; Empirical skewness: -0.10301722508451303
;;; Empirical kurtosis: 4.663663655463448
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 257.93726 ms
;;; Empirical mean: 194.1113140097736
;;; Empirical variance: 157.97329206580696
;;; Empirical std: 12.568742660497389
;;; Empirical skewness: -0.1914258319732461
;;; Empirical kurtosis: 4.196449674571811
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 286.98342 ms
;;; Empirical mean: 193.79887263780432
;;; Empirical variance: 152.40335545124205
;;; Empirical std: 12.345175391675975
;;; Empirical skewness: -0.14489636538813913
;;; Empirical kurtosis: 4.511025617409288
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 302.61603 ms
;;; Empirical mean: 194.80092922307472
;;; Empirical variance: 164.9573047959859
;;; Empirical std: 12.84357056258056
;;; Empirical skewness: -0.22714667802002825
;;; Empirical kurtosis: 4.22149862532624
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 348.400562 ms
;;; Empirical mean: 193.0836873945854
;;; Empirical variance: 155.00393492868142
;;; Empirical std: 12.450057627524517
;;; Empirical skewness: -0.21120839930549745
;;; Empirical kurtosis: 4.307929214353191
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 297.07492 ms
;;; Empirical mean: 194.17749441051748
;;; Empirical variance: 156.05748875109393
;;; Empirical std: 12.492297176704287
;;; Empirical skewness: -0.14723889105454896
;;; Empirical kurtosis: 4.240804850955116
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 303.721109 ms
;;; Empirical mean: 193.577246178983
;;; Empirical variance: 173.36194354518392
;;; Empirical std: 13.166698278049205
;;; Empirical skewness: -0.3015592785105131
;;; Empirical kurtosis: 4.167422631392098
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 295.238487 ms
;;; Empirical mean: 194.19794626117198
;;; Empirical variance: 155.23238969475048
;;; Empirical std: 12.459229097129183
;;; Empirical skewness: -0.24528312765903793
;;; Empirical kurtosis: 4.763085850760293
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 324.962859 ms
;;; Empirical mean: 193.51689913807647
;;; Empirical variance: 162.62179666439545
;;; Empirical std: 12.752325147375887
;;; Empirical skewness: -0.2760952572882693
;;; Empirical kurtosis: 4.0724907281973755
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 0
;;; Elapsed time: 216.856696 ms
;;; Empirical mean: 194.28273493338307
;;; Empirical variance: 170.99509652942575
;;; Empirical std: 13.076509340394544
;;; Empirical skewness: -0.3113790627656393
;;; Empirical kurtosis: 4.8628730395322535
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 1
;;; Elapsed time: 215.500957 ms
;;; Empirical mean: 194.90277688973921
;;; Empirical variance: 172.2199055427286
;;; Empirical std: 13.123258190812548
;;; Empirical skewness: -0.33987983883392275
;;; Empirical kurtosis: 4.492396170944698
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 2
;;; Elapsed time: 211.609857 ms
;;; Empirical mean: 194.51119108307216
;;; Empirical variance: 154.84601463105872
;;; Empirical std: 12.443713860060377
;;; Empirical skewness: -0.04041403649772128
;;; Empirical kurtosis: 3.558004464059237
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 3
;;; Elapsed time: 217.360798 ms
;;; Empirical mean: 194.50292185418888
;;; Empirical variance: 170.72270923369618
;;; Empirical std: 13.066090051491923
;;; Empirical skewness: -0.20977768131972904
;;; Empirical kurtosis: 4.330936114403662
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 4
;;; Elapsed time: 210.992127 ms
;;; Empirical mean: 195.1058222022289
;;; Empirical variance: 158.46754628394794
;;; Empirical std: 12.588389344310412
;;; Empirical skewness: 0.043798524571287994
;;; Empirical kurtosis: 3.0469316601777177
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 5
;;; Elapsed time: 248.680451 ms
;;; Empirical mean: 193.71275798301264
;;; Empirical variance: 155.6856772235329
;;; Empirical std: 12.477406670599981
;;; Empirical skewness: -0.2732910627236983
;;; Empirical kurtosis: 4.463011359954797
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 6
;;; Elapsed time: 221.686176 ms
;;; Empirical mean: 193.7045129821543
;;; Empirical variance: 167.65828740560238
;;; Empirical std: 12.948292837498014
;;; Empirical skewness: -0.04813084357681704
;;; Empirical kurtosis: 4.1265533865923985
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 7
;;; Elapsed time: 229.801166 ms
;;; Empirical mean: 194.04171830363637
;;; Empirical variance: 157.08405322026655
;;; Empirical std: 12.533317725976094
;;; Empirical skewness: -0.43276689649561123
;;; Empirical kurtosis: 4.960556144596372
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 8
;;; Elapsed time: 252.985828 ms
;;; Empirical mean: 194.1663604979207
;;; Empirical variance: 150.17772853624285
;;; Empirical std: 12.254702303044446
;;; Empirical skewness: -0.375035509138928
;;; Empirical kurtosis: 4.882418583192736
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 9
;;; Elapsed time: 225.792137 ms
;;; Empirical mean: 192.89581752341752
;;; Empirical variance: 170.17940360043937
;;; Empirical std: 13.04528281028968
;;; Empirical skewness: -0.44882230941719886
;;; Empirical kurtosis: 5.642671539600128
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 10
;;; Elapsed time: 232.635083 ms
;;; Empirical mean: 194.30059585619136
;;; Empirical variance: 152.14663946974196
;;; Empirical std: 12.334773588102133
;;; Empirical skewness: 0.026016339666311967
;;; Empirical kurtosis: 3.1519766950608217
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 11
;;; Elapsed time: 223.510453 ms
;;; Empirical mean: 194.03225553692423
;;; Empirical variance: 154.9432608588626
;;; Empirical std: 12.44762069067268
;;; Empirical skewness: -0.04997722915360567
;;; Empirical kurtosis: 4.54158940035086
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 12
;;; Elapsed time: 210.100972 ms
;;; Empirical mean: 193.71169443358437
;;; Empirical variance: 164.26123875375887
;;; Empirical std: 12.816444076020419
;;; Empirical skewness: -0.29393927759796823
;;; Empirical kurtosis: 4.3141921432364745
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 13
;;; Elapsed time: 239.701028 ms
;;; Empirical mean: 193.5771851584964
;;; Empirical variance: 166.81388594111453
;;; Empirical std: 12.915645006778195
;;; Empirical skewness: -0.4089538173933409
;;; Empirical kurtosis: 5.323378801349271
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 14
;;; Elapsed time: 209.119849 ms
;;; Empirical mean: 194.60529166401747
;;; Empirical variance: 158.565660485998
;;; Empirical std: 12.592285753031417
;;; Empirical skewness: -0.017664083101812108
;;; Empirical kurtosis: 3.0578329247626264
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 15
;;; Elapsed time: 213.158944 ms
;;; Empirical mean: 194.0394719904633
;;; Empirical variance: 163.3457827470872
;;; Empirical std: 12.780680058083263
;;; Empirical skewness: -0.4411145065776895
;;; Empirical kurtosis: 5.378969427438774
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 16
;;; Elapsed time: 213.819488 ms
;;; Empirical mean: 193.7519474785667
;;; Empirical variance: 142.55403611201146
;;; Empirical std: 11.939599495460953
;;; Empirical skewness: -0.17555052940206764
;;; Empirical kurtosis: 3.7188043499266557
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 17
;;; Elapsed time: 229.343898 ms
;;; Empirical mean: 194.5881357593664
;;; Empirical variance: 150.4570321543644
;;; Empirical std: 12.266092782722803
;;; Empirical skewness: 0.19293667699813752
;;; Empirical kurtosis: 3.0449149763055314
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 18
;;; Elapsed time: 232.02923 ms
;;; Empirical mean: 193.98861685280343
;;; Empirical variance: 177.09875421375693
;;; Empirical std: 13.307845588740385
;;; Empirical skewness: -0.7728254028954876
;;; Empirical kurtosis: 7.033104394940967
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 19
;;; Elapsed time: 205.741761 ms
;;; Empirical mean: 193.98116775352284
;;; Empirical variance: 159.4782355936537
;;; Empirical std: 12.628469249820174
;;; Empirical skewness: -0.24899699712147805
;;; Empirical kurtosis: 3.9710856825340137
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 184.339047 ms
;;; Empirical mean: 193.24899999999997
;;; Empirical variance: 149.42299899999995
;;; Empirical std: 12.223870050029163
;;; Empirical skewness: -0.17771834811649598
;;; Empirical kurtosis: 4.010096252216117
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 206.670344 ms
;;; Empirical mean: 194.19199999999992
;;; Empirical variance: 176.82713599999997
;;; Empirical std: 13.297636481721103
;;; Empirical skewness: -0.4800381394275766
;;; Empirical kurtosis: 4.9803540568912315
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 186.885344 ms
;;; Empirical mean: 193.94500000000002
;;; Empirical variance: 173.89597500000002
;;; Empirical std: 13.1869623113134
;;; Empirical skewness: -0.4546730923761092
;;; Empirical kurtosis: 5.870000959263647
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 183.182184 ms
;;; Empirical mean: 194.478
;;; Empirical variance: 168.94751599999998
;;; Empirical std: 12.997981227867657
;;; Empirical skewness: -0.24298343762916866
;;; Empirical kurtosis: 3.210947563713535
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 185.095732 ms
;;; Empirical mean: 193.95299999999992
;;; Empirical variance: 173.030791
;;; Empirical std: 13.154116884078535
;;; Empirical skewness: -0.036306288695663734
;;; Empirical kurtosis: 3.995288844266118
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 222.348166 ms
;;; Empirical mean: 193.60699999999994
;;; Empirical variance: 156.69655099999994
;;; Empirical std: 12.51784929610514
;;; Empirical skewness: -0.3734752203595768
;;; Empirical kurtosis: 5.224866253183213
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 223.204187 ms
;;; Empirical mean: 194.32199999999992
;;; Empirical variance: 126.16031600000001
;;; Empirical std: 11.232110932500623
;;; Empirical skewness: -0.04008354341693429
;;; Empirical kurtosis: 3.3964206160985015
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 189.230929 ms
;;; Empirical mean: 193.28199999999995
;;; Empirical variance: 159.80447599999997
;;; Empirical std: 12.64137951332844
;;; Empirical skewness: -0.04043514180862691
;;; Empirical kurtosis: 3.9671439043341743
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 218.861844 ms
;;; Empirical mean: 194.63799999999998
;;; Empirical variance: 166.786956
;;; Empirical std: 12.914602432905165
;;; Empirical skewness: 0.04677468410524516
;;; Empirical kurtosis: 3.379195750274689
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 236.695226 ms
;;; Empirical mean: 195.422
;;; Empirical variance: 158.62991599999995
;;; Empirical std: 12.594836878657855
;;; Empirical skewness: -0.41055171411001806
;;; Empirical kurtosis: 4.314855005417154
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 242.023106 ms
;;; Empirical mean: 195.44
;;; Empirical variance: 135.97639999999996
;;; Empirical std: 11.660891904138378
;;; Empirical skewness: -0.5149349709328659
;;; Empirical kurtosis: 5.238111364412414
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 233.877099 ms
;;; Empirical mean: 194.78700000000003
;;; Empirical variance: 159.70563100000004
;;; Empirical std: 12.637469327361394
;;; Empirical skewness: -0.3401882467522726
;;; Empirical kurtosis: 4.019618800176621
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 290.590796 ms
;;; Empirical mean: 194.1999999999999
;;; Empirical variance: 174.0519999999999
;;; Empirical std: 13.192876865945498
;;; Empirical skewness: -0.15986140134822377
;;; Empirical kurtosis: 4.032633948732376
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 222.724081 ms
;;; Empirical mean: 195.16999999999993
;;; Empirical variance: 198.7730999999999
;;; Empirical std: 14.098691428639748
;;; Empirical skewness: 0.009831004442379629
;;; Empirical kurtosis: 4.153098661726318
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 192.852326 ms
;;; Empirical mean: 195.60900000000004
;;; Empirical variance: 141.91611899999995
;;; Empirical std: 11.912855199321443
;;; Empirical skewness: -0.35865150115086397
;;; Empirical kurtosis: 5.261733012819305
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 195.537369 ms
;;; Empirical mean: 193.56400000000005
;;; Empirical variance: 158.43190399999995
;;; Empirical std: 12.586973583828637
;;; Empirical skewness: 2.747057863688963E-4
;;; Empirical kurtosis: 2.5267829964783846
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 224.348912 ms
;;; Empirical mean: 194.83000000000013
;;; Empirical variance: 146.93510000000015
;;; Empirical std: 12.121678926617392
;;; Empirical skewness: -0.19192867654790957
;;; Empirical kurtosis: 3.519976815929775
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 210.084464 ms
;;; Empirical mean: 195.026
;;; Empirical variance: 165.535324
;;; Empirical std: 12.866053163266503
;;; Empirical skewness: 0.12985914590647143
;;; Empirical kurtosis: 3.122160481145935
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 191.990316 ms
;;; Empirical mean: 195.11700000000013
;;; Empirical variance: 178.84531099999998
;;; Empirical std: 13.373305911404255
;;; Empirical skewness: -0.7450231127250578
;;; Empirical kurtosis: 5.974009354188479
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 187.099417 ms
;;; Empirical mean: 194.27700000000002
;;; Empirical variance: 178.52027100000004
;;; Empirical std: 13.361147817459399
;;; Empirical skewness: -0.6258662742768902
;;; Empirical kurtosis: 5.673269773478753
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 109.608556 ms
;;; Empirical mean: 193.54238500374365
;;; Empirical variance: 164.1135019514644
;;; Empirical std: 12.81067921507148
;;; Empirical skewness: -0.19080296968362637
;;; Empirical kurtosis: 3.05648481531349
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 168.190518 ms
;;; Empirical mean: 192.60540406628658
;;; Empirical variance: 176.17532566964712
;;; Empirical std: 13.273105351410692
;;; Empirical skewness: -0.3958122250588064
;;; Empirical kurtosis: 4.438673040636948
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 90.403241 ms
;;; Empirical mean: 193.38704095889273
;;; Empirical variance: 147.05789394960766
;;; Empirical std: 12.126742924198883
;;; Empirical skewness: -0.424630548587676
;;; Empirical kurtosis: 5.147495399560343
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 86.075319 ms
;;; Empirical mean: 194.05548417860538
;;; Empirical variance: 141.7925425379298
;;; Empirical std: 11.907667384417898
;;; Empirical skewness: 0.0585634139787991
;;; Empirical kurtosis: 3.0352533814775877
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 85.431232 ms
;;; Empirical mean: 194.42176056330064
;;; Empirical variance: 144.9837674025573
;;; Empirical std: 12.040920538005278
;;; Empirical skewness: 0.20662127281791512
;;; Empirical kurtosis: 3.7092188494010134
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 84.262985 ms
;;; Empirical mean: 192.550547150507
;;; Empirical variance: 189.100583701774
;;; Empirical std: 13.75138479214999
;;; Empirical skewness: -0.30590585852060526
;;; Empirical kurtosis: 3.0123318604386093
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 84.000288 ms
;;; Empirical mean: 193.61073955380408
;;; Empirical variance: 169.34885095391635
;;; Empirical std: 13.01341042747505
;;; Empirical skewness: -0.2122036151927779
;;; Empirical kurtosis: 4.199677274662723
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 80.847215 ms
;;; Empirical mean: 191.56189950844177
;;; Empirical variance: 166.1100386887021
;;; Empirical std: 12.888368348580906
;;; Empirical skewness: 0.13557037758273016
;;; Empirical kurtosis: 2.5983245092928633
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 103.155619 ms
;;; Empirical mean: 193.61025860989702
;;; Empirical variance: 118.50399384692832
;;; Empirical std: 10.88595397045791
;;; Empirical skewness: 0.008663013719802718
;;; Empirical kurtosis: 3.2951100051731537
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 85.295264 ms
;;; Empirical mean: 194.84315044231013
;;; Empirical variance: 159.49380381218495
;;; Empirical std: 12.629085628507907
;;; Empirical skewness: -0.09304474676010008
;;; Empirical kurtosis: 3.2672679045239135
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 85.882029 ms
;;; Empirical mean: 194.6893722545887
;;; Empirical variance: 122.59995333439952
;;; Empirical std: 11.072486321255923
;;; Empirical skewness: 0.1558311564768384
;;; Empirical kurtosis: 3.659652418154466
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 82.041586 ms
;;; Empirical mean: 193.8137909077458
;;; Empirical variance: 150.89427079999876
;;; Empirical std: 12.283902913976435
;;; Empirical skewness: -0.1167295577581592
;;; Empirical kurtosis: 3.4511224945966577
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 96.477832 ms
;;; Empirical mean: 194.4011784316345
;;; Empirical variance: 143.666216154673
;;; Empirical std: 11.986084271131794
;;; Empirical skewness: -0.3170710814659946
;;; Empirical kurtosis: 3.5099282865819803
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 100.431577 ms
;;; Empirical mean: 193.99158848153078
;;; Empirical variance: 138.86745361003648
;;; Empirical std: 11.784203562822414
;;; Empirical skewness: 0.23304055539370644
;;; Empirical kurtosis: 2.885854565281475
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 86.18972 ms
;;; Empirical mean: 192.21962152711424
;;; Empirical variance: 185.9948092539861
;;; Empirical std: 13.6379913936762
;;; Empirical skewness: -0.19794971470277076
;;; Empirical kurtosis: 3.294047820357418
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 92.212079 ms
;;; Empirical mean: 194.71858547749738
;;; Empirical variance: 135.11474991307662
;;; Empirical std: 11.623887039758973
;;; Empirical skewness: -0.07999299955063092
;;; Empirical kurtosis: 3.22915010462945
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 103.080111 ms
;;; Empirical mean: 195.1159671522586
;;; Empirical variance: 146.9575311070024
;;; Empirical std: 12.122604138839245
;;; Empirical skewness: -0.5785950468131495
;;; Empirical kurtosis: 5.330154387501204
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 108.957911 ms
;;; Empirical mean: 193.69798312442384
;;; Empirical variance: 198.0307710024631
;;; Empirical std: 14.072340636953864
;;; Empirical skewness: -0.11158156573617352
;;; Empirical kurtosis: 3.282119107944098
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 102.022551 ms
;;; Empirical mean: 195.3431326515556
;;; Empirical variance: 137.22156688389097
;;; Empirical std: 11.714160955181168
;;; Empirical skewness: -0.35996150272752664
;;; Empirical kurtosis: 5.896993965739883
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 131.945839 ms
;;; Empirical mean: 193.16250775636445
;;; Empirical variance: 121.60921874202573
;;; Empirical std: 11.02765699239987
;;; Empirical skewness: 0.37636972725891504
;;; Empirical kurtosis: 4.017027039925933
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2458.748262 ms
;;; Empirical mean: 194.13233725104587
;;; Empirical variance: 162.02519634442282
;;; Empirical std: 12.728911828763009
;;; Empirical skewness: -0.24568538891313768
;;; Empirical kurtosis: 4.5243460408780285
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 2231.312025 ms
;;; Empirical mean: 194.42723289229227
;;; Empirical variance: 160.05192858531728
;;; Empirical std: 12.651163131717071
;;; Empirical skewness: -0.26309453268124483
;;; Empirical kurtosis: 4.634607378959546
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 2445.9598 ms
;;; Empirical mean: 194.02151098963375
;;; Empirical variance: 161.56754191997143
;;; Empirical std: 12.710922150653406
;;; Empirical skewness: -0.2038739155469514
;;; Empirical kurtosis: 4.385655138153424
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 2237.403553 ms
;;; Empirical mean: 194.1155358308132
;;; Empirical variance: 161.0872307538016
;;; Empirical std: 12.69201444821907
;;; Empirical skewness: -0.20164590626513296
;;; Empirical kurtosis: 4.319871709623164
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 2051.101909 ms
;;; Empirical mean: 194.07102024203894
;;; Empirical variance: 162.40935037116742
;;; Empirical std: 12.743992717008569
;;; Empirical skewness: -0.21834008229210317
;;; Empirical kurtosis: 4.300356849439503
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 2140.408731 ms
;;; Empirical mean: 194.44251826260606
;;; Empirical variance: 162.650951497895
;;; Empirical std: 12.753468214485618
;;; Empirical skewness: -0.2841365256058523
;;; Empirical kurtosis: 4.552871476059887
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 2144.928111 ms
;;; Empirical mean: 194.1338143062481
;;; Empirical variance: 159.62540941070395
;;; Empirical std: 12.634294970860225
;;; Empirical skewness: -0.2878496749941346
;;; Empirical kurtosis: 4.595083682106533
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2104.456341 ms
;;; Empirical mean: 193.84643198821442
;;; Empirical variance: 170.1903672673488
;;; Empirical std: 13.045703019283737
;;; Empirical skewness: -0.24303348574038097
;;; Empirical kurtosis: 4.337199367066719
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 2121.358203 ms
;;; Empirical mean: 194.19654837105526
;;; Empirical variance: 160.53966829841048
;;; Empirical std: 12.670424945455084
;;; Empirical skewness: -0.21001310762172676
;;; Empirical kurtosis: 4.47549338110313
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 2038.535841 ms
;;; Empirical mean: 194.0794578338195
;;; Empirical variance: 159.8006173179949
;;; Empirical std: 12.641226891326447
;;; Empirical skewness: -0.262359331286161
;;; Empirical kurtosis: 4.584162315765218
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 2106.106392 ms
;;; Empirical mean: 193.96552730352792
;;; Empirical variance: 161.00504143974615
;;; Empirical std: 12.688776199450684
;;; Empirical skewness: -0.19887472123820998
;;; Empirical kurtosis: 4.369519054916086
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2103.557111 ms
;;; Empirical mean: 194.11643873088414
;;; Empirical variance: 166.46914045613514
;;; Empirical std: 12.902292062115752
;;; Empirical skewness: -0.23229215530414277
;;; Empirical kurtosis: 4.415702023784969
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2259.590624 ms
;;; Empirical mean: 194.1002285011435
;;; Empirical variance: 159.67474443801655
;;; Empirical std: 12.63624724504932
;;; Empirical skewness: -0.25873791828090986
;;; Empirical kurtosis: 4.632441173131861
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2212.841588 ms
;;; Empirical mean: 194.12019398498333
;;; Empirical variance: 161.75717813887084
;;; Empirical std: 12.718379540604646
;;; Empirical skewness: -0.21464736889308417
;;; Empirical kurtosis: 4.370174343570192
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2020.615978 ms
;;; Empirical mean: 194.2277751330704
;;; Empirical variance: 162.83448306446397
;;; Empirical std: 12.760661544938177
;;; Empirical skewness: -0.1990558481609918
;;; Empirical kurtosis: 4.38685498683399
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 2026.005758 ms
;;; Empirical mean: 193.9833180166045
;;; Empirical variance: 161.90778673954017
;;; Empirical std: 12.724299066728202
;;; Empirical skewness: -0.20588165470729128
;;; Empirical kurtosis: 4.421868407972612
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 2039.121734 ms
;;; Empirical mean: 194.19118176836392
;;; Empirical variance: 159.1814518513587
;;; Empirical std: 12.61671319525647
;;; Empirical skewness: -0.23272514783515116
;;; Empirical kurtosis: 4.389393200207491
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 2073.214721 ms
;;; Empirical mean: 194.20239230712264
;;; Empirical variance: 163.8572387625601
;;; Empirical std: 12.80067337145043
;;; Empirical skewness: -0.1644365161722931
;;; Empirical kurtosis: 4.2684390435291455
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 2063.600201 ms
;;; Empirical mean: 193.88043582729603
;;; Empirical variance: 160.7206628367462
;;; Empirical std: 12.677565335534508
;;; Empirical skewness: -0.2296763054819404
;;; Empirical kurtosis: 4.375677459113706
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 2459.336837 ms
;;; Empirical mean: 194.30435456124835
;;; Empirical variance: 161.86979657197355
;;; Empirical std: 12.722806159490663
;;; Empirical skewness: -0.20925349464343157
;;; Empirical kurtosis: 4.426765528398842
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1821.069011 ms
;;; Empirical mean: 194.3969999999999
;;; Empirical variance: 162.98799100000005
;;; Empirical std: 12.766675017403712
;;; Empirical skewness: 0.20425986481355649
;;; Empirical kurtosis: 2.7722562590354616
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1869.219107 ms
;;; Empirical mean: 194.55890000000005
;;; Empirical variance: 178.0159307900001
;;; Empirical std: 13.342261082365317
;;; Empirical skewness: -0.014006611814080082
;;; Empirical kurtosis: 3.0309152443140173
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1760.247319 ms
;;; Empirical mean: 194.41570000000004
;;; Empirical variance: 167.91149351000007
;;; Empirical std: 12.958066735049641
;;; Empirical skewness: -0.19728142025286524
;;; Empirical kurtosis: 4.236806681386955
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1812.844107 ms
;;; Empirical mean: 193.43730000000002
;;; Empirical variance: 162.34746871000007
;;; Empirical std: 12.741564609968435
;;; Empirical skewness: -0.27235901407337143
;;; Empirical kurtosis: 4.430564487446769
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1722.602723 ms
;;; Empirical mean: 194.89790000000005
;;; Empirical variance: 152.27727559000007
;;; Empirical std: 12.340067892438844
;;; Empirical skewness: -0.14105285770283327
;;; Empirical kurtosis: 3.1279732416661887
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1734.704437 ms
;;; Empirical mean: 194.47350000000003
;;; Empirical variance: 161.86629775000006
;;; Empirical std: 12.722668656771662
;;; Empirical skewness: 0.04826369367448476
;;; Empirical kurtosis: 3.4556332437923047
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1831.244571 ms
;;; Empirical mean: 192.91159999999996
;;; Empirical variance: 152.21498544
;;; Empirical std: 12.337543736092691
;;; Empirical skewness: -0.46045162677400897
;;; Empirical kurtosis: 5.362931148857151
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2147.223144 ms
;;; Empirical mean: 194.88790000000006
;;; Empirical variance: 158.05513359000005
;;; Empirical std: 12.57199799514779
;;; Empirical skewness: 0.04796766424796127
;;; Empirical kurtosis: 2.9093065010296058
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1874.150771 ms
;;; Empirical mean: 193.2889
;;; Empirical variance: 190.49023679000024
;;; Empirical std: 13.80182005352918
;;; Empirical skewness: -0.225564337877175
;;; Empirical kurtosis: 4.191889305497358
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1772.684417 ms
;;; Empirical mean: 194.6428999999999
;;; Empirical variance: 148.68217959000003
;;; Empirical std: 12.193530234923767
;;; Empirical skewness: -0.34762902476305735
;;; Empirical kurtosis: 5.032001096321964
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1896.341299 ms
;;; Empirical mean: 193.67370000000003
;;; Empirical variance: 156.28702831000004
;;; Empirical std: 12.50148104466027
;;; Empirical skewness: -0.06281132799568014
;;; Empirical kurtosis: 3.3231940229500205
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2371.310236 ms
;;; Empirical mean: 194.83469999999983
;;; Empirical variance: 161.35177590999993
;;; Empirical std: 12.70243188960287
;;; Empirical skewness: -0.21451289531293674
;;; Empirical kurtosis: 4.656090392480129
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1761.413264 ms
;;; Empirical mean: 194.69520000000003
;;; Empirical variance: 160.04589696000002
;;; Empirical std: 12.65092474722698
;;; Empirical skewness: 0.12743193592090324
;;; Empirical kurtosis: 3.099547499639047
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1927.268627 ms
;;; Empirical mean: 194.1531
;;; Empirical variance: 166.43386039000006
;;; Empirical std: 12.90092478816926
;;; Empirical skewness: 0.0986879206144153
;;; Empirical kurtosis: 2.8920602456974276
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2153.100657 ms
;;; Empirical mean: 193.11310000000003
;;; Empirical variance: 155.7371083900001
;;; Empirical std: 12.479467472212109
;;; Empirical skewness: -4.6932853864806287E-4
;;; Empirical kurtosis: 3.270625469301232
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1872.100133 ms
;;; Empirical mean: 194.6647
;;; Empirical variance: 163.72427391000005
;;; Empirical std: 12.795478651070464
;;; Empirical skewness: -0.24194586319011466
;;; Empirical kurtosis: 4.668735879169765
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1776.003594 ms
;;; Empirical mean: 195.10430000000005
;;; Empirical variance: 157.42402151000016
;;; Empirical std: 12.546872977359744
;;; Empirical skewness: -0.44611304650268085
;;; Empirical kurtosis: 5.032690945084791
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1826.536319 ms
;;; Empirical mean: 194.63839999999985
;;; Empirical variance: 163.42924544
;;; Empirical std: 12.783944830919758
;;; Empirical skewness: -0.39876707714953796
;;; Empirical kurtosis: 5.543736075392678
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1931.623058 ms
;;; Empirical mean: 194.55829999999986
;;; Empirical variance: 144.29320111
;;; Empirical std: 12.012210500569827
;;; Empirical skewness: -0.007355996597120147
;;; Empirical kurtosis: 3.237593342316514
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1968.235654 ms
;;; Empirical mean: 192.60379999999998
;;; Empirical variance: 135.04382555999993
;;; Empirical std: 11.620835837408595
;;; Empirical skewness: -0.07054033949633262
;;; Empirical kurtosis: 3.2738439094052763
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2547.863976 ms
;;; Empirical mean: 195.1539999999999
;;; Empirical variance: 161.08788400000003
;;; Empirical std: 12.692040182728702
;;; Empirical skewness: 0.19109262309747538
;;; Empirical kurtosis: 2.9220576865315238
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 2354.408548 ms
;;; Empirical mean: 194.3412999999999
;;; Empirical variance: 148.15961431
;;; Empirical std: 12.172083400552266
;;; Empirical skewness: 0.021716575084749855
;;; Empirical kurtosis: 3.6951449941608354
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 2331.494475 ms
;;; Empirical mean: 192.99989999999994
;;; Empirical variance: 194.64289999000005
;;; Empirical std: 13.951447953169593
;;; Empirical skewness: -0.31284307665971334
;;; Empirical kurtosis: 3.478360185883764
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 2401.17133 ms
;;; Empirical mean: 194.418
;;; Empirical variance: 168.25207600000005
;;; Empirical std: 12.97120179474516
;;; Empirical skewness: 0.037144559739212
;;; Empirical kurtosis: 3.8029277532416295
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 2362.377022 ms
;;; Empirical mean: 193.14190000000008
;;; Empirical variance: 161.39736439000015
;;; Empirical std: 12.704226241294672
;;; Empirical skewness: -0.04897922985398385
;;; Empirical kurtosis: 3.0888445719976096
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 2328.378628 ms
;;; Empirical mean: 193.99780000000013
;;; Empirical variance: 179.5239951600001
;;; Empirical std: 13.398656468467282
;;; Empirical skewness: -0.222792517646142
;;; Empirical kurtosis: 4.1217079883811545
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 2328.533203 ms
;;; Empirical mean: 193.96999999999997
;;; Empirical variance: 142.4817000000001
;;; Empirical std: 11.936569859050802
;;; Empirical skewness: -0.33700323856365005
;;; Empirical kurtosis: 3.9355484264427085
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2498.783918 ms
;;; Empirical mean: 194.25769999999994
;;; Empirical variance: 179.10949070999993
;;; Empirical std: 13.383179394672998
;;; Empirical skewness: 0.06769593380133976
;;; Empirical kurtosis: 3.4837036378613857
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 2548.629951 ms
;;; Empirical mean: 194.78800000000004
;;; Empirical variance: 143.25605600000006
;;; Empirical std: 11.968962193941463
;;; Empirical skewness: -0.08293143452713259
;;; Empirical kurtosis: 3.7661013495562075
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 2435.701225 ms
;;; Empirical mean: 194.67809999999994
;;; Empirical variance: 138.16848039000004
;;; Empirical std: 11.75450893870093
;;; Empirical skewness: -0.1298876305144238
;;; Empirical kurtosis: 4.308281587581673
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 3183.858471 ms
;;; Empirical mean: 193.1195999999999
;;; Empirical variance: 192.78449583999998
;;; Empirical std: 13.88468565866725
;;; Empirical skewness: -0.5181930650715653
;;; Empirical kurtosis: 4.960500398807623
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2514.092592 ms
;;; Empirical mean: 192.92060000000006
;;; Empirical variance: 189.05969564000011
;;; Empirical std: 13.749898022894573
;;; Empirical skewness: -0.46210112153003935
;;; Empirical kurtosis: 4.1596364640979155
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2496.525094 ms
;;; Empirical mean: 193.22150000000016
;;; Empirical variance: 155.80003775000017
;;; Empirical std: 12.481988533482964
;;; Empirical skewness: 0.06413959695735148
;;; Empirical kurtosis: 2.998109535852828
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2404.911245 ms
;;; Empirical mean: 194.6565
;;; Empirical variance: 166.90390775000012
;;; Empirical std: 12.919129527564932
;;; Empirical skewness: -0.14236013299175332
;;; Empirical kurtosis: 4.665075700398039
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2391.519485 ms
;;; Empirical mean: 195.12680000000006
;;; Empirical variance: 158.59092176000007
;;; Empirical std: 12.59328875870001
;;; Empirical skewness: -0.057175593896263095
;;; Empirical kurtosis: 2.667485368636116
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 2399.204477 ms
;;; Empirical mean: 194.77450000000005
;;; Empirical variance: 157.4038497500001
;;; Empirical std: 12.546069095537458
;;; Empirical skewness: -0.024706369969750022
;;; Empirical kurtosis: 2.892445144429649
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 2340.653912 ms
;;; Empirical mean: 192.68909999999988
;;; Empirical variance: 157.15484118999996
;;; Empirical std: 12.536141399569484
;;; Empirical skewness: 0.03078706676201548
;;; Empirical kurtosis: 2.511570402666992
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 2534.287905 ms
;;; Empirical mean: 194.57009999999994
;;; Empirical variance: 151.35048598999995
;;; Empirical std: 12.302458534374336
;;; Empirical skewness: 0.04095787669784931
;;; Empirical kurtosis: 2.687143716036481
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 2368.749845 ms
;;; Empirical mean: 195.81840000000008
;;; Empirical variance: 170.97242144000015
;;; Empirical std: 13.075642295505034
;;; Empirical skewness: -0.17184490301053804
;;; Empirical kurtosis: 4.377087143973972
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 2387.504798 ms
;;; Empirical mean: 191.7611
;;; Empirical variance: 136.64662679000006
;;; Empirical std: 11.689594808632165
;;; Empirical skewness: 0.09461656391889366
;;; Empirical kurtosis: 2.6954822914634025
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2639.091154 ms
;;; Empirical mean: 193.97237549667167
;;; Empirical variance: 163.14104772841128
;;; Empirical std: 12.772667995701262
;;; Empirical skewness: -0.1979316322754871
;;; Empirical kurtosis: 4.282298529492853
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 2613.216307 ms
;;; Empirical mean: 194.27863615551098
;;; Empirical variance: 164.22060766749195
;;; Empirical std: 12.814858862566219
;;; Empirical skewness: -0.24802026580172837
;;; Empirical kurtosis: 4.460822204860163
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 2788.098438 ms
;;; Empirical mean: 194.34506924741558
;;; Empirical variance: 164.1197667921206
;;; Empirical std: 12.810923729072803
;;; Empirical skewness: -0.26466620278668085
;;; Empirical kurtosis: 4.408052621549814
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 3146.620826 ms
;;; Empirical mean: 193.8564943874334
;;; Empirical variance: 164.80608478352482
;;; Empirical std: 12.837682220070912
;;; Empirical skewness: -0.2343374042042394
;;; Empirical kurtosis: 4.401543175087302
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 3103.184445 ms
;;; Empirical mean: 193.9525806279131
;;; Empirical variance: 160.14386939801315
;;; Empirical std: 12.654796300139056
;;; Empirical skewness: -0.26052910510841437
;;; Empirical kurtosis: 4.493944186954977
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 2570.453713 ms
;;; Empirical mean: 194.0580089135872
;;; Empirical variance: 162.58082994189925
;;; Empirical std: 12.750718800989192
;;; Empirical skewness: -0.2305055467328362
;;; Empirical kurtosis: 4.3290673175852845
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 2570.127102 ms
;;; Empirical mean: 194.13947909326737
;;; Empirical variance: 161.5382885371552
;;; Empirical std: 12.709771380208034
;;; Empirical skewness: -0.17643026933761066
;;; Empirical kurtosis: 4.364326872982255
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2637.618041 ms
;;; Empirical mean: 194.0427470980584
;;; Empirical variance: 160.4403364251857
;;; Empirical std: 12.666504506973725
;;; Empirical skewness: -0.2527135366544834
;;; Empirical kurtosis: 4.373158276981269
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 2593.315776 ms
;;; Empirical mean: 194.0778638937347
;;; Empirical variance: 161.72275592673955
;;; Empirical std: 12.717026221831091
;;; Empirical skewness: -0.23368398987202604
;;; Empirical kurtosis: 4.405615212187338
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 2569.788838 ms
;;; Empirical mean: 194.18096238944815
;;; Empirical variance: 162.53638596523137
;;; Empirical std: 12.748975879074813
;;; Empirical skewness: -0.23813200904909348
;;; Empirical kurtosis: 4.361629290209232
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 2566.0497 ms
;;; Empirical mean: 194.2225787259407
;;; Empirical variance: 161.88366888463588
;;; Empirical std: 12.723351322848705
;;; Empirical skewness: -0.2138532082145367
;;; Empirical kurtosis: 4.400672297950042
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2549.936485 ms
;;; Empirical mean: 194.27831779559213
;;; Empirical variance: 164.01696377904608
;;; Empirical std: 12.80691078203663
;;; Empirical skewness: -0.22450822059522038
;;; Empirical kurtosis: 4.415920070583806
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2533.287052 ms
;;; Empirical mean: 193.5909646288183
;;; Empirical variance: 162.49775411414421
;;; Empirical std: 12.747460692786788
;;; Empirical skewness: -0.20709480672808767
;;; Empirical kurtosis: 4.36089687420729
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2575.264333 ms
;;; Empirical mean: 194.17926410552212
;;; Empirical variance: 168.33869466023646
;;; Empirical std: 12.974540248511177
;;; Empirical skewness: -0.27087276528167736
;;; Empirical kurtosis: 4.2975018983961695
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2500.433907 ms
;;; Empirical mean: 194.0899466975716
;;; Empirical variance: 167.39113109898904
;;; Empirical std: 12.937972449305533
;;; Empirical skewness: -0.2802518683755792
;;; Empirical kurtosis: 4.265609053561664
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 2559.351137 ms
;;; Empirical mean: 193.9747831088061
;;; Empirical variance: 167.47438108957834
;;; Empirical std: 12.941189322839627
;;; Empirical skewness: -0.21739654215998452
;;; Empirical kurtosis: 4.248462998833384
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 2513.219264 ms
;;; Empirical mean: 194.04063538298843
;;; Empirical variance: 162.75161513308905
;;; Empirical std: 12.757414124072678
;;; Empirical skewness: -0.25595646304219827
;;; Empirical kurtosis: 4.490202360827975
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 2523.250366 ms
;;; Empirical mean: 193.9174197473379
;;; Empirical variance: 163.83902042105152
;;; Empirical std: 12.79996173514013
;;; Empirical skewness: -0.17803664869754263
;;; Empirical kurtosis: 4.3774403592411035
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 2534.548588 ms
;;; Empirical mean: 194.16268607891902
;;; Empirical variance: 163.51416871749456
;;; Empirical std: 12.787265881238826
;;; Empirical skewness: -0.2612355539213279
;;; Empirical kurtosis: 4.518488058621493
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 2567.397948 ms
;;; Empirical mean: 194.26755220037543
;;; Empirical variance: 164.16888609208596
;;; Empirical std: 12.812840672235254
;;; Empirical skewness: -0.2750839536545465
;;; Empirical kurtosis: 4.434625586772815
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 0
;;; Elapsed time: 2074.406475 ms
;;; Empirical mean: 194.06332811903502
;;; Empirical variance: 161.75153037998686
;;; Empirical std: 12.718157507280166
;;; Empirical skewness: -0.2275322512549018
;;; Empirical kurtosis: 4.334550210782626
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 1
;;; Elapsed time: 2036.498525 ms
;;; Empirical mean: 194.0432287408844
;;; Empirical variance: 161.17163131447015
;;; Empirical std: 12.695338960203864
;;; Empirical skewness: -0.2475069056934929
;;; Empirical kurtosis: 4.543046910143364
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 2
;;; Elapsed time: 2037.428771 ms
;;; Empirical mean: 193.97691036237083
;;; Empirical variance: 157.6345736725956
;;; Empirical std: 12.555260796677844
;;; Empirical skewness: -0.17339258149775438
;;; Empirical kurtosis: 4.309682053407114
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 3
;;; Elapsed time: 2010.347189 ms
;;; Empirical mean: 194.03027607398533
;;; Empirical variance: 163.38300170971604
;;; Empirical std: 12.782136038617178
;;; Empirical skewness: -0.2587880136765075
;;; Empirical kurtosis: 4.6433061937284075
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 4
;;; Elapsed time: 2143.197915 ms
;;; Empirical mean: 194.3274517174535
;;; Empirical variance: 160.11490731742435
;;; Empirical std: 12.653651935999518
;;; Empirical skewness: -0.18016905384927698
;;; Empirical kurtosis: 4.3208288939750314
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 5
;;; Elapsed time: 2108.695438 ms
;;; Empirical mean: 193.93422955687114
;;; Empirical variance: 162.12615452416986
;;; Empirical std: 12.732876914671321
;;; Empirical skewness: -0.19343650401177487
;;; Empirical kurtosis: 4.111994421375749
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 6
;;; Elapsed time: 2046.355798 ms
;;; Empirical mean: 194.17232974577269
;;; Empirical variance: 160.6948159652695
;;; Empirical std: 12.676545900412679
;;; Empirical skewness: -0.1418403367855255
;;; Empirical kurtosis: 4.088887730061361
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 7
;;; Elapsed time: 2074.540812 ms
;;; Empirical mean: 194.10862231421797
;;; Empirical variance: 167.08365086056978
;;; Empirical std: 12.926084127088519
;;; Empirical skewness: -0.22951029374587628
;;; Empirical kurtosis: 4.287287766299678
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 8
;;; Elapsed time: 2127.624185 ms
;;; Empirical mean: 194.1493222822006
;;; Empirical variance: 162.47995576994253
;;; Empirical std: 12.74676256035008
;;; Empirical skewness: -0.19799706788904822
;;; Empirical kurtosis: 4.3544841564085
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 9
;;; Elapsed time: 2103.506801 ms
;;; Empirical mean: 194.39777052880166
;;; Empirical variance: 163.64591313459206
;;; Empirical std: 12.792416235199356
;;; Empirical skewness: -0.29116553231344977
;;; Empirical kurtosis: 4.695395651949909
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 10
;;; Elapsed time: 2003.619145 ms
;;; Empirical mean: 193.89920009181202
;;; Empirical variance: 163.9949698560694
;;; Empirical std: 12.806052079234622
;;; Empirical skewness: -0.23731660870157592
;;; Empirical kurtosis: 4.321095109712178
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 11
;;; Elapsed time: 2087.541126 ms
;;; Empirical mean: 193.99502998738168
;;; Empirical variance: 164.4951028163643
;;; Empirical std: 12.825564424865062
;;; Empirical skewness: -0.19582812498877084
;;; Empirical kurtosis: 4.149192823876955
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 12
;;; Elapsed time: 2093.911313 ms
;;; Empirical mean: 194.2332205415146
;;; Empirical variance: 163.31141813879694
;;; Empirical std: 12.779335590663425
;;; Empirical skewness: -0.2568015474790794
;;; Empirical kurtosis: 4.409245195032393
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 13
;;; Elapsed time: 2024.331449 ms
;;; Empirical mean: 194.36435324072423
;;; Empirical variance: 166.56082597325542
;;; Empirical std: 12.905844643929951
;;; Empirical skewness: -0.20771607957802093
;;; Empirical kurtosis: 4.499226864715688
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 14
;;; Elapsed time: 2199.355287 ms
;;; Empirical mean: 193.96949827683784
;;; Empirical variance: 163.62354413343024
;;; Empirical std: 12.791541898200945
;;; Empirical skewness: -0.2812785721685496
;;; Empirical kurtosis: 4.482703882628643
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 15
;;; Elapsed time: 2047.334504 ms
;;; Empirical mean: 193.99591393590936
;;; Empirical variance: 166.47896865277863
;;; Empirical std: 12.902672926676031
;;; Empirical skewness: -0.39452996971488075
;;; Empirical kurtosis: 4.995643557188848
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 16
;;; Elapsed time: 2069.973818 ms
;;; Empirical mean: 194.1630046030536
;;; Empirical variance: 165.97077127739328
;;; Empirical std: 12.882964382369195
;;; Empirical skewness: -0.15466137146209927
;;; Empirical kurtosis: 4.31331669415067
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 17
;;; Elapsed time: 2078.21015 ms
;;; Empirical mean: 194.07502938070158
;;; Empirical variance: 171.20623185448503
;;; Empirical std: 13.084579926558018
;;; Empirical skewness: -0.35422834039940165
;;; Empirical kurtosis: 4.9571403683209505
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 18
;;; Elapsed time: 2083.878222 ms
;;; Empirical mean: 194.05098586628694
;;; Empirical variance: 161.6440303760646
;;; Empirical std: 12.71393056360088
;;; Empirical skewness: -0.25808611663092945
;;; Empirical kurtosis: 4.543926050237002
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 19
;;; Elapsed time: 2043.704579 ms
;;; Empirical mean: 193.9889764830565
;;; Empirical variance: 160.3654118823919
;;; Empirical std: 12.66354657599489
;;; Empirical skewness: -0.22316743375272685
;;; Empirical kurtosis: 4.236669008546059
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1865.976772 ms
;;; Empirical mean: 193.928
;;; Empirical variance: 160.0962160000001
;;; Empirical std: 12.652913340412955
;;; Empirical skewness: -0.28291864486679213
;;; Empirical kurtosis: 4.451752481928363
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1866.87065 ms
;;; Empirical mean: 194.29889999999997
;;; Empirical variance: 162.33575879
;;; Empirical std: 12.741105085117225
;;; Empirical skewness: -0.2573122701087674
;;; Empirical kurtosis: 4.415860501081705
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1829.249069 ms
;;; Empirical mean: 194.6211999999999
;;; Empirical variance: 163.1299105600001
;;; Empirical std: 12.772232011672827
;;; Empirical skewness: -0.1769387882083128
;;; Empirical kurtosis: 4.049864174884991
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1852.233204 ms
;;; Empirical mean: 194.23800000000003
;;; Empirical variance: 165.23475600000015
;;; Empirical std: 12.854367195626557
;;; Empirical skewness: -0.2122661939942995
;;; Empirical kurtosis: 4.527971971643338
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1816.53019 ms
;;; Empirical mean: 194.05729999999988
;;; Empirical variance: 164.87221671000003
;;; Empirical std: 12.840257657461553
;;; Empirical skewness: -0.20926767000834406
;;; Empirical kurtosis: 4.014964155682909
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1846.33556 ms
;;; Empirical mean: 194.41080000000008
;;; Empirical variance: 162.50924336000008
;;; Empirical std: 12.747911333234166
;;; Empirical skewness: -0.15874185939927
;;; Empirical kurtosis: 4.053839391905921
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1808.228505 ms
;;; Empirical mean: 194.03119999999998
;;; Empirical variance: 180.70682656000008
;;; Empirical std: 13.442723926347668
;;; Empirical skewness: -0.22179209940578956
;;; Empirical kurtosis: 4.330211429551919
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2238.808589 ms
;;; Empirical mean: 193.75799999999995
;;; Empirical variance: 162.63263600000002
;;; Empirical std: 12.752750134774853
;;; Empirical skewness: -0.04843138197495596
;;; Empirical kurtosis: 3.6272068739597074
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1786.579694 ms
;;; Empirical mean: 194.37630000000007
;;; Empirical variance: 166.65129831000013
;;; Empirical std: 12.909349259741953
;;; Empirical skewness: -0.21603584498111664
;;; Empirical kurtosis: 4.929891771292833
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1840.893967 ms
;;; Empirical mean: 194.37789999999995
;;; Empirical variance: 166.94729159000008
;;; Empirical std: 12.920808472769808
;;; Empirical skewness: -0.3668435493433351
;;; Empirical kurtosis: 5.102727637199716
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 2083.415865 ms
;;; Empirical mean: 194.06489999999997
;;; Empirical variance: 162.04368798999997
;;; Empirical std: 12.729638171998447
;;; Empirical skewness: -0.16442935088222602
;;; Empirical kurtosis: 4.239466724799969
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2070.639483 ms
;;; Empirical mean: 194.03699999999975
;;; Empirical variance: 165.10163099999988
;;; Empirical std: 12.849187950995187
;;; Empirical skewness: -0.2798213477130254
;;; Empirical kurtosis: 4.448837333468431
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2132.280379 ms
;;; Empirical mean: 193.97820000000007
;;; Empirical variance: 160.27312476000014
;;; Empirical std: 12.659902241328728
;;; Empirical skewness: -0.09154454117545083
;;; Empirical kurtosis: 4.223614014794306
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2426.553528 ms
;;; Empirical mean: 193.7501
;;; Empirical variance: 164.1188499900001
;;; Empirical std: 12.810887946976981
;;; Empirical skewness: -0.29410272584237956
;;; Empirical kurtosis: 4.449455314711511
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2234.247664 ms
;;; Empirical mean: 194.06209999999987
;;; Empirical variance: 161.82744358999994
;;; Empirical std: 12.721141599321971
;;; Empirical skewness: -0.28622270268935623
;;; Empirical kurtosis: 4.186907913097846
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 2249.704742 ms
;;; Empirical mean: 194.2641999999999
;;; Empirical variance: 157.27319836000007
;;; Empirical std: 12.540861149059904
;;; Empirical skewness: -0.3034541565063341
;;; Empirical kurtosis: 4.403448321545983
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 2316.18878 ms
;;; Empirical mean: 194.05359999999988
;;; Empirical variance: 167.25912704
;;; Empirical std: 12.932870023316557
;;; Empirical skewness: -0.2521426643140335
;;; Empirical kurtosis: 4.303462010879952
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1888.330072 ms
;;; Empirical mean: 194.28659999999996
;;; Empirical variance: 161.87706044000004
;;; Empirical std: 12.723091622714977
;;; Empirical skewness: -0.3469086864395574
;;; Empirical kurtosis: 4.701312705044212
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1874.936142 ms
;;; Empirical mean: 194.06390000000005
;;; Empirical variance: 162.2610167900001
;;; Empirical std: 12.738171642351194
;;; Empirical skewness: -0.19530974019709307
;;; Empirical kurtosis: 4.30729740599191
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1813.362701 ms
;;; Empirical mean: 194.0726000000001
;;; Empirical variance: 153.64112924000005
;;; Empirical std: 12.395205897442771
;;; Empirical skewness: -0.04872188941926289
;;; Empirical kurtosis: 3.842339032271692
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 754.320832 ms
;;; Empirical mean: 193.25433865054188
;;; Empirical variance: 157.06914436677062
;;; Empirical std: 12.532722943030802
;;; Empirical skewness: -0.12806384751161287
;;; Empirical kurtosis: 4.240797346536026
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 777.419193 ms
;;; Empirical mean: 193.6334779043236
;;; Empirical variance: 164.41529293989893
;;; Empirical std: 12.822452688152097
;;; Empirical skewness: -0.27112751795983636
;;; Empirical kurtosis: 4.5097555642687235
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 767.45874 ms
;;; Empirical mean: 192.98767104882936
;;; Empirical variance: 153.8880078447559
;;; Empirical std: 12.40516053280875
;;; Empirical skewness: -0.13812531666386912
;;; Empirical kurtosis: 3.9857233025158356
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 797.414455 ms
;;; Empirical mean: 193.61071171832808
;;; Empirical variance: 158.93476366850777
;;; Empirical std: 12.606933158722931
;;; Empirical skewness: -0.1956977883866137
;;; Empirical kurtosis: 4.5994139365616205
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 799.891612 ms
;;; Empirical mean: 193.70175275352167
;;; Empirical variance: 157.13360284177517
;;; Empirical std: 12.535294286205456
;;; Empirical skewness: -0.14295235441329399
;;; Empirical kurtosis: 3.8466786828351758
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 745.129274 ms
;;; Empirical mean: 193.5865308318118
;;; Empirical variance: 148.02606083137397
;;; Empirical std: 12.166596107020812
;;; Empirical skewness: -0.07503948446114675
;;; Empirical kurtosis: 4.046812310461349
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 750.73152 ms
;;; Empirical mean: 193.78689653763848
;;; Empirical variance: 153.1328093707347
;;; Empirical std: 12.374684212970232
;;; Empirical skewness: -0.20279891455147953
;;; Empirical kurtosis: 4.445215251014604
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 800.18339 ms
;;; Empirical mean: 193.9653331348696
;;; Empirical variance: 150.74304383206683
;;; Empirical std: 12.277745877483653
;;; Empirical skewness: -0.07906832208617208
;;; Empirical kurtosis: 3.7236683831956845
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 780.401238 ms
;;; Empirical mean: 192.70470492709873
;;; Empirical variance: 152.726088371369
;;; Empirical std: 12.358239695497454
;;; Empirical skewness: -0.3891895371523282
;;; Empirical kurtosis: 4.6964669630467775
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 748.104286 ms
;;; Empirical mean: 193.5471630483788
;;; Empirical variance: 147.44685615530705
;;; Empirical std: 12.142769706920536
;;; Empirical skewness: -0.14417947284879531
;;; Empirical kurtosis: 4.5108736647540235
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 783.367715 ms
;;; Empirical mean: 192.800793204062
;;; Empirical variance: 154.4267579249737
;;; Empirical std: 12.426856317064816
;;; Empirical skewness: -0.051727165253471416
;;; Empirical kurtosis: 4.006955750437983
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 850.111103 ms
;;; Empirical mean: 193.27748191998558
;;; Empirical variance: 149.63418186561668
;;; Empirical std: 12.232505134501954
;;; Empirical skewness: -0.004288547769660666
;;; Empirical kurtosis: 3.4599392309378003
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 745.35494 ms
;;; Empirical mean: 193.69995074746788
;;; Empirical variance: 152.92579087017745
;;; Empirical std: 12.366316786746872
;;; Empirical skewness: -0.25173264741363877
;;; Empirical kurtosis: 4.733556208561241
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 750.273339 ms
;;; Empirical mean: 193.24489614906383
;;; Empirical variance: 152.20151201442025
;;; Empirical std: 12.336997690460198
;;; Empirical skewness: -0.010620895433183992
;;; Empirical kurtosis: 3.5597485525255097
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 797.131976 ms
;;; Empirical mean: 193.7125145832012
;;; Empirical variance: 160.2283380299796
;;; Empirical std: 12.658133275881543
;;; Empirical skewness: -0.11075841125409563
;;; Empirical kurtosis: 3.7259332583471565
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 745.103078 ms
;;; Empirical mean: 193.38418467521208
;;; Empirical variance: 160.34290668863298
;;; Empirical std: 12.662657963027865
;;; Empirical skewness: -0.3573083458418756
;;; Empirical kurtosis: 4.727747858611166
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 753.145184 ms
;;; Empirical mean: 193.1297993024889
;;; Empirical variance: 161.39254800895836
;;; Empirical std: 12.704036681659824
;;; Empirical skewness: -0.0583228929517239
;;; Empirical kurtosis: 3.745145648681833
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 750.517308 ms
;;; Empirical mean: 192.90580465140044
;;; Empirical variance: 154.00598141533067
;;; Empirical std: 12.409914641742331
;;; Empirical skewness: -0.20485229152288317
;;; Empirical kurtosis: 4.405395865419377
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 785.813267 ms
;;; Empirical mean: 193.05510381883985
;;; Empirical variance: 156.81836365657657
;;; Empirical std: 12.522713909395861
;;; Empirical skewness: -0.35245007684456325
;;; Empirical kurtosis: 5.139339034934142
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 744.007225 ms
;;; Empirical mean: 193.2913627068488
;;; Empirical variance: 151.42531219299312
;;; Empirical std: 12.305499266303384
;;; Empirical skewness: -0.09078275613540172
;;; Empirical kurtosis: 3.5396781039429803
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x4369e02b]</span>","value":"#multifn[print-method 0x4369e02b]"}],"value":"[nil,#multifn[print-method 0x4369e02b]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x4369e02b]</span>","value":"#multifn[print-method 0x4369e02b]"}],"value":"[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)],nil]"}
;; <=

;; @@

;; @@
