var jsPsychModule = (function (exports) {
    'use strict';

    /*! *****************************************************************************
    Copyright (c) Microsoft Corporation.

    Permission to use, copy, modify, and/or distribute this software for any
    purpose with or without fee is hereby granted.

    THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
    REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
    AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
    INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
    LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
    OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
    PERFORMANCE OF THIS SOFTWARE.
    ***************************************************************************** */

    function __awaiter(thisArg, _arguments, P, generator) {
        function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
        return new (P || (P = Promise))(function (resolve, reject) {
            function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
            function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
            function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
            step((generator = generator.apply(thisArg, _arguments || [])).next());
        });
    }

    // Gets all non-builtin properties up the prototype chain
    const getAllProperties = object => {
    	const properties = new Set();

    	do {
    		for (const key of Reflect.ownKeys(object)) {
    			properties.add([object, key]);
    		}
    	} while ((object = Reflect.getPrototypeOf(object)) && object !== Object.prototype);

    	return properties;
    };

    var autoBind = (self, {include, exclude} = {}) => {
    	const filter = key => {
    		const match = pattern => typeof pattern === 'string' ? key === pattern : pattern.test(key);

    		if (include) {
    			return include.some(match);
    		}

    		if (exclude) {
    			return !exclude.some(match);
    		}

    		return true;
    	};

    	for (const [object, key] of getAllProperties(self.constructor.prototype)) {
    		if (key === 'constructor' || !filter(key)) {
    			continue;
    		}

    		const descriptor = Reflect.getOwnPropertyDescriptor(object, key);
    		if (descriptor && typeof descriptor.value === 'function') {
    			self[key] = self[key].bind(self);
    		}
    	}

    	return self;
    };

    var version = "7.0.0";

    /**
     * Finds all of the unique items in an array.
     * @param arr The array to extract unique values from
     * @returns An array with one copy of each unique item in `arr`
     */
    function unique(arr) {
        return [...new Set(arr)];
    }
    function deepCopy(obj) {
        if (!obj)
            return obj;
        let out;
        if (Array.isArray(obj)) {
            out = [];
            for (const x of obj) {
                out.push(deepCopy(x));
            }
            return out;
        }
        else if (typeof obj === "object" && obj !== null) {
            out = {};
            for (const key in obj) {
                if (obj.hasOwnProperty(key)) {
                    out[key] = deepCopy(obj[key]);
                }
            }
            return out;
        }
        else {
            return obj;
        }
    }

    var utils = /*#__PURE__*/Object.freeze({
        __proto__: null,
        unique: unique,
        deepCopy: deepCopy
    });

    class DataColumn {
        constructor(values = []) {
            this.values = values;
        }
        sum() {
            let s = 0;
            for (const v of this.values) {
                s += v;
            }
            return s;
        }
        mean() {
            return this.sum() / this.count();
        }
        median() {
            if (this.values.length === 0) {
                return undefined;
            }
            const numbers = this.values.slice(0).sort(function (a, b) {
                return a - b;
            });
            const middle = Math.floor(numbers.length / 2);
            const isEven = numbers.length % 2 === 0;
            return isEven ? (numbers[middle] + numbers[middle - 1]) / 2 : numbers[middle];
        }
        min() {
            return Math.min.apply(null, this.values);
        }
        max() {
            return Math.max.apply(null, this.values);
        }
        count() {
            return this.values.length;
        }
        variance() {
            const mean = this.mean();
            let sum_square_error = 0;
            for (const x of this.values) {
                sum_square_error += Math.pow(x - mean, 2);
            }
            const mse = sum_square_error / (this.values.length - 1);
            return mse;
        }
        sd() {
            const mse = this.variance();
            const rmse = Math.sqrt(mse);
            return rmse;
        }
        frequencies() {
            const unique = {};
            for (const x of this.values) {
                if (typeof unique[x] === "undefined") {
                    unique[x] = 1;
                }
                else {
                    unique[x]++;
                }
            }
            return unique;
        }
        all(eval_fn) {
            for (const x of this.values) {
                if (!eval_fn(x)) {
                    return false;
                }
            }
            return true;
        }
        subset(eval_fn) {
            const out = [];
            for (const x of this.values) {
                if (eval_fn(x)) {
                    out.push(x);
                }
            }
            return new DataColumn(out);
        }
    }

    // private function to save text file on local drive
    function saveTextToFile(textstr, filename) {
        const blobToSave = new Blob([textstr], {
            type: "text/plain",
        });
        let blobURL = "";
        if (typeof window.webkitURL !== "undefined") {
            blobURL = window.webkitURL.createObjectURL(blobToSave);
        }
        else {
            blobURL = window.URL.createObjectURL(blobToSave);
        }
        const link = document.createElement("a");
        link.id = "jspsych-download-as-text-link";
        link.style.display = "none";
        link.download = filename;
        link.href = blobURL;
        link.click();
    }
    // this function based on code suggested by StackOverflow users:
    // http://stackoverflow.com/users/64741/zachary
    // http://stackoverflow.com/users/317/joseph-sturtevant
    function JSON2CSV(objArray) {
        const array = typeof objArray != "object" ? JSON.parse(objArray) : objArray;
        let line = "";
        let result = "";
        const columns = [];
        for (const row of array) {
            for (const key in row) {
                let keyString = key + "";
                keyString = '"' + keyString.replace(/"/g, '""') + '",';
                if (!columns.includes(key)) {
                    columns.push(key);
                    line += keyString;
                }
            }
        }
        line = line.slice(0, -1); // removes last comma
        result += line + "\r\n";
        for (const row of array) {
            line = "";
            for (const col of columns) {
                let value = typeof row[col] === "undefined" ? "" : row[col];
                if (typeof value == "object") {
                    value = JSON.stringify(value);
                }
                const valueString = value + "";
                line += '"' + valueString.replace(/"/g, '""') + '",';
            }
            line = line.slice(0, -1);
            result += line + "\r\n";
        }
        return result;
    }
    // this function is modified from StackOverflow:
    // http://stackoverflow.com/posts/3855394
    function getQueryString() {
        const a = window.location.search.substr(1).split("&");
        const b = {};
        for (let i = 0; i < a.length; ++i) {
            const p = a[i].split("=", 2);
            if (p.length == 1)
                b[p[0]] = "";
            else
                b[p[0]] = decodeURIComponent(p[1].replace(/\+/g, " "));
        }
        return b;
    }

    class DataCollection {
        constructor(data = []) {
            this.trials = data;
        }
        push(new_data) {
            this.trials.push(new_data);
            return this;
        }
        join(other_data_collection) {
            this.trials = this.trials.concat(other_data_collection.values());
            return this;
        }
        top() {
            if (this.trials.length <= 1) {
                return this;
            }
            else {
                return new DataCollection([this.trials[this.trials.length - 1]]);
            }
        }
        /**
         * Queries the first n elements in a collection of trials.
         *
         * @param n A positive integer of elements to return. A value of
         *          n that is less than 1 will throw an error.
         *
         * @return First n objects of a collection of trials. If fewer than
         *         n trials are available, the trials.length elements will
         *         be returned.
         *
         */
        first(n = 1) {
            if (n < 1) {
                throw `You must query with a positive nonzero integer. Please use a
               different value for n.`;
            }
            if (this.trials.length === 0)
                return new DataCollection();
            if (n > this.trials.length)
                n = this.trials.length;
            return new DataCollection(this.trials.slice(0, n));
        }
        /**
         * Queries the last n elements in a collection of trials.
         *
         * @param n A positive integer of elements to return. A value of
         *          n that is less than 1 will throw an error.
         *
         * @return Last n objects of a collection of trials. If fewer than
         *         n trials are available, the trials.length elements will
         *         be returned.
         *
         */
        last(n = 1) {
            if (n < 1) {
                throw `You must query with a positive nonzero integer. Please use a
               different value for n.`;
            }
            if (this.trials.length === 0)
                return new DataCollection();
            if (n > this.trials.length)
                n = this.trials.length;
            return new DataCollection(this.trials.slice(this.trials.length - n, this.trials.length));
        }
        values() {
            return this.trials;
        }
        count() {
            return this.trials.length;
        }
        readOnly() {
            return new DataCollection(deepCopy(this.trials));
        }
        addToAll(properties) {
            for (const trial of this.trials) {
                Object.assign(trial, properties);
            }
            return this;
        }
        addToLast(properties) {
            if (this.trials.length != 0) {
                Object.assign(this.trials[this.trials.length - 1], properties);
            }
            return this;
        }
        filter(filters) {
            // [{p1: v1, p2:v2}, {p1:v2}]
            // {p1: v1}
            let f;
            if (!Array.isArray(filters)) {
                f = deepCopy([filters]);
            }
            else {
                f = deepCopy(filters);
            }
            const filtered_data = [];
            for (const trial of this.trials) {
                let keep = false;
                for (const filter of f) {
                    let match = true;
                    for (const key of Object.keys(filter)) {
                        if (typeof trial[key] !== "undefined" && trial[key] === filter[key]) ;
                        else {
                            match = false;
                        }
                    }
                    if (match) {
                        keep = true;
                        break;
                    } // can break because each filter is OR.
                }
                if (keep) {
                    filtered_data.push(trial);
                }
            }
            return new DataCollection(filtered_data);
        }
        filterCustom(fn) {
            return new DataCollection(this.trials.filter(fn));
        }
        select(column) {
            const values = [];
            for (const trial of this.trials) {
                if (typeof trial[column] !== "undefined") {
                    values.push(trial[column]);
                }
            }
            return new DataColumn(values);
        }
        ignore(columns) {
            if (!Array.isArray(columns)) {
                columns = [columns];
            }
            const o = deepCopy(this.trials);
            for (const trial of o) {
                for (const delete_key of columns) {
                    delete trial[delete_key];
                }
            }
            return new DataCollection(o);
        }
        uniqueNames() {
            const names = [];
            for (const trial of this.trials) {
                for (const key of Object.keys(trial)) {
                    if (!names.includes(key)) {
                        names.push(key);
                    }
                }
            }
            return names;
        }
        csv() {
            return JSON2CSV(this.trials);
        }
        json(pretty = false) {
            if (pretty) {
                return JSON.stringify(this.trials, null, "\t");
            }
            return JSON.stringify(this.trials);
        }
        localSave(format, filename) {
            format = format.toLowerCase();
            let data_string;
            if (format === "json") {
                data_string = this.json();
            }
            else if (format === "csv") {
                data_string = this.csv();
            }
            else {
                throw new Error('Invalid format specified for localSave. Must be "json" or "csv".');
            }
            saveTextToFile(data_string, filename);
        }
    }

    class JsPsychData {
        constructor(jsPsych) {
            this.jsPsych = jsPsych;
            // data properties for all trials
            this.dataProperties = {};
            this.reset();
        }
        reset() {
            this.allData = new DataCollection();
            this.interactionData = new DataCollection();
        }
        get() {
            return this.allData;
        }
        getInteractionData() {
            return this.interactionData;
        }
        write(data_object) {
            const progress = this.jsPsych.getProgress();
            const trial = this.jsPsych.getCurrentTrial();
            //var trial_opt_data = typeof trial.data == 'function' ? trial.data() : trial.data;
            const default_data = {
                trial_type: trial.type.info.name,
                trial_index: progress.current_trial_global,
                time_elapsed: this.jsPsych.getTotalTime(),
                internal_node_id: this.jsPsych.getCurrentTimelineNodeID(),
            };
            this.allData.push(Object.assign(Object.assign(Object.assign(Object.assign({}, data_object), trial.data), default_data), this.dataProperties));
        }
        addProperties(properties) {
            // first, add the properties to all data that's already stored
            this.allData.addToAll(properties);
            // now add to list so that it gets appended to all future data
            this.dataProperties = Object.assign({}, this.dataProperties, properties);
        }
        addDataToLastTrial(data) {
            this.allData.addToLast(data);
        }
        getDataByTimelineNode(node_id) {
            return this.allData.filterCustom((x) => x.internal_node_id.slice(0, node_id.length) === node_id);
        }
        getLastTrialData() {
            return this.allData.top();
        }
        getLastTimelineData() {
            const lasttrial = this.getLastTrialData();
            const node_id = lasttrial.select("internal_node_id").values[0];
            if (typeof node_id === "undefined") {
                return new DataCollection();
            }
            else {
                const parent_node_id = node_id.substr(0, node_id.lastIndexOf("-"));
                const lastnodedata = this.getDataByTimelineNode(parent_node_id);
                return lastnodedata;
            }
        }
        displayData(format = "json") {
            format = format.toLowerCase();
            if (format != "json" && format != "csv") {
                console.log("Invalid format declared for displayData function. Using json as default.");
                format = "json";
            }
            const data_string = format === "json" ? this.allData.json(true) : this.allData.csv();
            const display_element = this.jsPsych.getDisplayElement();
            display_element.innerHTML = '<pre id="jspsych-data-display"></pre>';
            document.getElementById("jspsych-data-display").textContent = data_string;
        }
        urlVariables() {
            if (typeof this.query_string == "undefined") {
                this.query_string = getQueryString();
            }
            return this.query_string;
        }
        getURLVariable(whichvar) {
            return this.urlVariables()[whichvar];
        }
        createInteractionListeners() {
            // blur event capture
            window.addEventListener("blur", () => {
                const data = {
                    event: "blur",
                    trial: this.jsPsych.getProgress().current_trial_global,
                    time: this.jsPsych.getTotalTime(),
                };
                this.interactionData.push(data);
                this.jsPsych.getInitSettings().on_interaction_data_update(data);
            });
            // focus event capture
            window.addEventListener("focus", () => {
                const data = {
                    event: "focus",
                    trial: this.jsPsych.getProgress().current_trial_global,
                    time: this.jsPsych.getTotalTime(),
                };
                this.interactionData.push(data);
                this.jsPsych.getInitSettings().on_interaction_data_update(data);
            });
            // fullscreen change capture
            const fullscreenchange = () => {
                const data = {
                    event: 
                    // @ts-expect-error
                    document.isFullScreen ||
                        // @ts-expect-error
                        document.webkitIsFullScreen ||
                        // @ts-expect-error
                        document.mozIsFullScreen ||
                        document.fullscreenElement
                        ? "fullscreenenter"
                        : "fullscreenexit",
                    trial: this.jsPsych.getProgress().current_trial_global,
                    time: this.jsPsych.getTotalTime(),
                };
                this.interactionData.push(data);
                this.jsPsych.getInitSettings().on_interaction_data_update(data);
            };
            document.addEventListener("fullscreenchange", fullscreenchange);
            document.addEventListener("mozfullscreenchange", fullscreenchange);
            document.addEventListener("webkitfullscreenchange", fullscreenchange);
        }
        // public methods for testing purposes. not recommended for use.
        _customInsert(data) {
            this.allData = new DataCollection(data);
        }
        _fullreset() {
            this.reset();
            this.dataProperties = {};
        }
    }

    class HardwareAPI {
        constructor() {
            /**
             * Indicates whether this instance of jspsych has opened a hardware connection through our browser
             * extension
             **/
            this.hardwareConnected = false;
            //it might be useful to open up a line of communication from the extension back to this page
            //script, again, this will have to pass through DOM events. For now speed is of no concern so I
            //will use jQuery
            document.addEventListener("jspsych-activate", (evt) => {
                this.hardwareConnected = true;
            });
        }
        /**
         * Allows communication with user hardware through our custom Google Chrome extension + native C++ program
         * @param		mess	The message to be passed to our extension, see its documentation for the expected members of this object.
         * @author	Daniel Rivas
         *
         */
        hardware(mess) {
            //since Chrome extension content-scripts do not share the javascript environment with the page
            //script that loaded jspsych, we will need to use hacky methods like communicating through DOM
            //events.
            const jspsychEvt = new CustomEvent("jspsych", { detail: mess });
            document.dispatchEvent(jspsychEvt);
            //And voila! it will be the job of the content script injected by the extension to listen for
            //the event and do the appropriate actions.
        }
    }

    class KeyboardListenerAPI {
        constructor(getRootElement, areResponsesCaseSensitive = false, minimumValidRt = 0) {
            this.getRootElement = getRootElement;
            this.areResponsesCaseSensitive = areResponsesCaseSensitive;
            this.minimumValidRt = minimumValidRt;
            this.listeners = new Set();
            this.heldKeys = new Set();
            this.areRootListenersRegistered = false;
            autoBind(this);
            this.registerRootListeners();
        }
        /**
         * If not previously done and `this.getRootElement()` returns an element, adds the root key
         * listeners to that element.
         */
        registerRootListeners() {
            if (!this.areRootListenersRegistered) {
                const rootElement = this.getRootElement();
                if (rootElement) {
                    rootElement.addEventListener("keydown", this.rootKeydownListener);
                    rootElement.addEventListener("keyup", this.rootKeyupListener);
                    this.areRootListenersRegistered = true;
                }
            }
        }
        rootKeydownListener(e) {
            // Iterate over a static copy of the listeners set because listeners might add other listeners
            // that we do not want to be included in the loop
            for (const listener of Array.from(this.listeners)) {
                listener(e);
            }
            this.heldKeys.add(this.toLowerCaseIfInsensitive(e.key));
        }
        toLowerCaseIfInsensitive(string) {
            return this.areResponsesCaseSensitive ? string : string.toLowerCase();
        }
        rootKeyupListener(e) {
            this.heldKeys.delete(this.toLowerCaseIfInsensitive(e.key));
        }
        isResponseValid(validResponses, allowHeldKey, key) {
            // check if key was already held down
            if (!allowHeldKey && this.heldKeys.has(key)) {
                return false;
            }
            if (validResponses === "ALL_KEYS") {
                return true;
            }
            if (validResponses === "NO_KEYS") {
                return false;
            }
            return validResponses.includes(key);
        }
        getKeyboardResponse({ callback_function, valid_responses = "ALL_KEYS", rt_method = "performance", persist, audio_context, audio_context_start_time, allow_held_key = false, minimum_valid_rt = this.minimumValidRt, }) {
            if (rt_method !== "performance" && rt_method !== "audio") {
                console.log('Invalid RT method specified in getKeyboardResponse. Defaulting to "performance" method.');
                rt_method = "performance";
            }
            const usePerformanceRt = rt_method === "performance";
            const startTime = usePerformanceRt ? performance.now() : audio_context_start_time * 1000;
            this.registerRootListeners();
            if (!this.areResponsesCaseSensitive && typeof valid_responses !== "string") {
                valid_responses = valid_responses.map((r) => r.toLowerCase());
            }
            const listener = (e) => {
                const rt = Math.round((rt_method == "performance" ? performance.now() : audio_context.currentTime * 1000) -
                    startTime);
                if (rt < minimum_valid_rt) {
                    return;
                }
                const key = this.toLowerCaseIfInsensitive(e.key);
                if (this.isResponseValid(valid_responses, allow_held_key, key)) {
                    // if this is a valid response, then we don't want the key event to trigger other actions
                    // like scrolling via the spacebar.
                    e.preventDefault();
                    if (!persist) {
                        // remove keyboard listener if it exists
                        this.cancelKeyboardResponse(listener);
                    }
                    callback_function({ key, rt });
                }
            };
            this.listeners.add(listener);
            return listener;
        }
        cancelKeyboardResponse(listener) {
            // remove the listener from the set of listeners if it is contained
            this.listeners.delete(listener);
        }
        cancelAllKeyboardResponses() {
            this.listeners.clear();
        }
        compareKeys(key1, key2) {
            if ((typeof key1 !== "string" && key1 !== null) ||
                (typeof key2 !== "string" && key2 !== null)) {
                console.error("Error in jsPsych.pluginAPI.compareKeys: arguments must be key strings or null.");
                return undefined;
            }
            if (typeof key1 === "string" && typeof key2 === "string") {
                // if both values are strings, then check whether or not letter case should be converted before comparing (case_sensitive_responses in initJsPsych)
                return this.areResponsesCaseSensitive
                    ? key1 === key2
                    : key1.toLowerCase() === key2.toLowerCase();
            }
            return key1 === null && key2 === null;
        }
    }

    /**
     * Parameter types for plugins
     */
    exports.ParameterType = void 0;
    (function (ParameterType) {
        ParameterType[ParameterType["BOOL"] = 0] = "BOOL";
        ParameterType[ParameterType["STRING"] = 1] = "STRING";
        ParameterType[ParameterType["INT"] = 2] = "INT";
        ParameterType[ParameterType["FLOAT"] = 3] = "FLOAT";
        ParameterType[ParameterType["FUNCTION"] = 4] = "FUNCTION";
        ParameterType[ParameterType["KEY"] = 5] = "KEY";
        ParameterType[ParameterType["KEYS"] = 6] = "KEYS";
        ParameterType[ParameterType["SELECT"] = 7] = "SELECT";
        ParameterType[ParameterType["HTML_STRING"] = 8] = "HTML_STRING";
        ParameterType[ParameterType["IMAGE"] = 9] = "IMAGE";
        ParameterType[ParameterType["AUDIO"] = 10] = "AUDIO";
        ParameterType[ParameterType["VIDEO"] = 11] = "VIDEO";
        ParameterType[ParameterType["OBJECT"] = 12] = "OBJECT";
        ParameterType[ParameterType["COMPLEX"] = 13] = "COMPLEX";
        ParameterType[ParameterType["TIMELINE"] = 14] = "TIMELINE";
    })(exports.ParameterType || (exports.ParameterType = {}));
    const universalPluginParameters = {
        /**
         * Data to add to this trial (key-value pairs)
         */
        data: {
            type: exports.ParameterType.OBJECT,
            pretty_name: "Data",
            default: {},
        },
        /**
         * Function to execute when trial begins
         */
        on_start: {
            type: exports.ParameterType.FUNCTION,
            pretty_name: "On start",
            default: function () {
                return;
            },
        },
        /**
         * Function to execute when trial is finished
         */
        on_finish: {
            type: exports.ParameterType.FUNCTION,
            pretty_name: "On finish",
            default: function () {
                return;
            },
        },
        /**
         * Function to execute after the trial has loaded
         */
        on_load: {
            type: exports.ParameterType.FUNCTION,
            pretty_name: "On load",
            default: function () {
                return;
            },
        },
        /**
         * Length of gap between the end of this trial and the start of the next trial
         */
        post_trial_gap: {
            type: exports.ParameterType.INT,
            pretty_name: "Post trial gap",
            default: null,
        },
        /**
         * A list of CSS classes to add to the jsPsych display element for the duration of this trial
         */
        css_classes: {
            type: exports.ParameterType.STRING,
            pretty_name: "Custom CSS classes",
            default: null,
        },
    };

    const preloadParameterTypes = [
        exports.ParameterType.AUDIO,
        exports.ParameterType.IMAGE,
        exports.ParameterType.VIDEO,
    ];
    class MediaAPI {
        constructor(useWebaudio, webaudioContext) {
            this.useWebaudio = useWebaudio;
            this.webaudioContext = webaudioContext;
            // video //
            this.video_buffers = {};
            // audio //
            this.context = null;
            this.audio_buffers = [];
            // preloading stimuli //
            this.preload_requests = [];
            this.img_cache = {};
            this.preloadMap = new Map();
        }
        getVideoBuffer(videoID) {
            return this.video_buffers[videoID];
        }
        initAudio() {
            this.context = this.useWebaudio ? this.webaudioContext : null;
        }
        audioContext() {
            if (this.context !== null) {
                if (this.context.state !== "running") {
                    this.context.resume();
                }
            }
            return this.context;
        }
        getAudioBuffer(audioID) {
            return new Promise((resolve, reject) => {
                // check whether audio file already preloaded
                if (typeof this.audio_buffers[audioID] == "undefined" ||
                    this.audio_buffers[audioID] == "tmp") {
                    // if audio is not already loaded, try to load it
                    this.preloadAudio([audioID], () => {
                        resolve(this.audio_buffers[audioID]);
                    }, () => { }, (e) => {
                        reject(e.error);
                    });
                }
                else {
                    // audio is already loaded
                    resolve(this.audio_buffers[audioID]);
                }
            });
        }
        preloadAudio(files, callback_complete = () => { }, callback_load = (filepath) => { }, callback_error = (error_msg) => { }) {
            files = unique(files.flat());
            let n_loaded = 0;
            if (files.length == 0) {
                callback_complete();
                return;
            }
            const load_audio_file_webaudio = (source, count = 1) => {
                const request = new XMLHttpRequest();
                request.open("GET", source, true);
                request.responseType = "arraybuffer";
                request.onload = () => {
                    this.context.decodeAudioData(request.response, (buffer) => {
                        this.audio_buffers[source] = buffer;
                        n_loaded++;
                        callback_load(source);
                        if (n_loaded == files.length) {
                            callback_complete();
                        }
                    }, (e) => {
                        callback_error({ source: source, error: e });
                    });
                };
                request.onerror = function (e) {
                    let err = e;
                    if (this.status == 404) {
                        err = "404";
                    }
                    callback_error({ source: source, error: err });
                };
                request.onloadend = function (e) {
                    if (this.status == 404) {
                        callback_error({ source: source, error: "404" });
                    }
                };
                request.send();
                this.preload_requests.push(request);
            };
            const load_audio_file_html5audio = (source, count = 1) => {
                const audio = new Audio();
                const handleCanPlayThrough = () => {
                    this.audio_buffers[source] = audio;
                    n_loaded++;
                    callback_load(source);
                    if (n_loaded == files.length) {
                        callback_complete();
                    }
                    audio.removeEventListener("canplaythrough", handleCanPlayThrough);
                };
                audio.addEventListener("canplaythrough", handleCanPlayThrough);
                audio.addEventListener("error", function handleError(e) {
                    callback_error({ source: audio.src, error: e });
                    audio.removeEventListener("error", handleError);
                });
                audio.addEventListener("abort", function handleAbort(e) {
                    callback_error({ source: audio.src, error: e });
                    audio.removeEventListener("abort", handleAbort);
                });
                audio.src = source;
                this.preload_requests.push(audio);
            };
            for (const file of files) {
                if (typeof this.audio_buffers[file] !== "undefined") {
                    n_loaded++;
                    callback_load(file);
                    if (n_loaded == files.length) {
                        callback_complete();
                    }
                }
                else {
                    this.audio_buffers[file] = "tmp";
                    if (this.audioContext() !== null) {
                        load_audio_file_webaudio(file);
                    }
                    else {
                        load_audio_file_html5audio(file);
                    }
                }
            }
        }
        preloadImages(images, callback_complete = () => { }, callback_load = (filepath) => { }, callback_error = (error_msg) => { }) {
            // flatten the images array
            images = unique(images.flat());
            var n_loaded = 0;
            if (images.length === 0) {
                callback_complete();
                return;
            }
            for (var i = 0; i < images.length; i++) {
                var img = new Image();
                img.onload = function () {
                    n_loaded++;
                    callback_load(img.src);
                    if (n_loaded === images.length) {
                        callback_complete();
                    }
                };
                img.onerror = function (e) {
                    callback_error({ source: img.src, error: e });
                };
                img.src = images[i];
                this.img_cache[images[i]] = img;
                this.preload_requests.push(img);
            }
        }
        preloadVideo(videos, callback_complete = () => { }, callback_load = (filepath) => { }, callback_error = (error_msg) => { }) {
            // flatten the video array
            videos = unique(videos.flat());
            let n_loaded = 0;
            if (videos.length === 0) {
                callback_complete();
                return;
            }
            for (const video of videos) {
                const video_buffers = this.video_buffers;
                //based on option 4 here: http://dinbror.dk/blog/how-to-preload-entire-html5-video-before-play-solved/
                const request = new XMLHttpRequest();
                request.open("GET", video, true);
                request.responseType = "blob";
                request.onload = function () {
                    if (this.status === 200 || this.status === 0) {
                        const videoBlob = this.response;
                        video_buffers[video] = URL.createObjectURL(videoBlob); // IE10+
                        n_loaded++;
                        callback_load(video);
                        if (n_loaded === videos.length) {
                            callback_complete();
                        }
                    }
                };
                request.onerror = function (e) {
                    let err = e;
                    if (this.status == 404) {
                        err = "404";
                    }
                    callback_error({ source: video, error: err });
                };
                request.onloadend = function (e) {
                    if (this.status == 404) {
                        callback_error({ source: video, error: "404" });
                    }
                };
                request.send();
                this.preload_requests.push(request);
            }
        }
        getAutoPreloadList(timeline_description) {
            /** Map each preload parameter type to a set of paths to be preloaded */
            const preloadPaths = Object.fromEntries(preloadParameterTypes.map((type) => [type, new Set()]));
            const traverseTimeline = (node, inheritedTrialType) => {
                var _a, _b, _c, _d;
                const isTimeline = typeof node.timeline !== "undefined";
                if (isTimeline) {
                    for (const childNode of node.timeline) {
                        traverseTimeline(childNode, (_a = node.type) !== null && _a !== void 0 ? _a : inheritedTrialType);
                    }
                }
                else if ((_c = ((_b = node.type) !== null && _b !== void 0 ? _b : inheritedTrialType)) === null || _c === void 0 ? void 0 : _c.info) {
                    // node is a trial with type.info set
                    // Get the plugin name and parameters object from the info object
                    const { name: pluginName, parameters } = ((_d = node.type) !== null && _d !== void 0 ? _d : inheritedTrialType).info;
                    // Extract parameters to be preloaded and their types from parameter info if this has not
                    // yet been done for `pluginName`
                    if (!this.preloadMap.has(pluginName)) {
                        this.preloadMap.set(pluginName, Object.fromEntries(Object.entries(parameters)
                            // Filter out parameter entries with media types and a non-false `preload` option
                            .filter(([_name, { type, preload }]) => preloadParameterTypes.includes(type) && (preload !== null && preload !== void 0 ? preload : true))
                            // Map each entry's value to its parameter type
                            .map(([name, { type }]) => [name, type])));
                    }
                    // Add preload paths from this trial
                    for (const [parameterName, parameterType] of Object.entries(this.preloadMap.get(pluginName))) {
                        const parameterValue = node[parameterName];
                        const elements = preloadPaths[parameterType];
                        if (typeof parameterValue === "string") {
                            elements.add(parameterValue);
                        }
                        else if (Array.isArray(parameterValue)) {
                            for (const element of parameterValue.flat()) {
                                if (typeof element === "string") {
                                    elements.add(element);
                                }
                            }
                        }
                    }
                }
            };
            traverseTimeline({ timeline: timeline_description });
            return {
                images: [...preloadPaths[exports.ParameterType.IMAGE]],
                audio: [...preloadPaths[exports.ParameterType.AUDIO]],
                video: [...preloadPaths[exports.ParameterType.VIDEO]],
            };
        }
        cancelPreloads() {
            for (const request of this.preload_requests) {
                request.onload = () => { };
                request.onerror = () => { };
                request.oncanplaythrough = () => { };
                request.onabort = () => { };
            }
            this.preload_requests = [];
        }
    }

    class TimeoutAPI {
        constructor() {
            this.timeout_handlers = [];
        }
        setTimeout(callback, delay) {
            const handle = window.setTimeout(callback, delay);
            this.timeout_handlers.push(handle);
            return handle;
        }
        clearAllTimeouts() {
            for (const handler of this.timeout_handlers) {
                clearTimeout(handler);
            }
            this.timeout_handlers = [];
        }
    }

    function createJointPluginAPIObject(jsPsych) {
        const settings = jsPsych.getInitSettings();
        return Object.assign({}, ...[
            new KeyboardListenerAPI(jsPsych.getDisplayContainerElement, settings.case_sensitive_responses, settings.minimum_valid_rt),
            new TimeoutAPI(),
            new MediaAPI(settings.use_webaudio, jsPsych.webaudio_context),
            new HardwareAPI(),
        ].map((object) => autoBind(object)));
    }

    function repeat(array, repetitions, unpack = false) {
        const arr_isArray = Array.isArray(array);
        const rep_isArray = Array.isArray(repetitions);
        // if array is not an array, then we just repeat the item
        if (!arr_isArray) {
            if (!rep_isArray) {
                array = [array];
                repetitions = [repetitions];
            }
            else {
                repetitions = [repetitions[0]];
                console.log("Unclear parameters given to randomization.repeat. Multiple set sizes specified, but only one item exists to sample. Proceeding using the first set size.");
            }
        }
        else {
            // if repetitions is not an array, but array is, then we
            // repeat repetitions for each entry in array
            if (!rep_isArray) {
                let reps = [];
                for (let i = 0; i < array.length; i++) {
                    reps.push(repetitions);
                }
                repetitions = reps;
            }
            else {
                if (array.length != repetitions.length) {
                    console.warn("Unclear parameters given to randomization.repeat. Items and repetitions are unequal lengths. Behavior may not be as expected.");
                    // throw warning if repetitions is too short, use first rep ONLY.
                    if (repetitions.length < array.length) {
                        let reps = [];
                        for (let i = 0; i < array.length; i++) {
                            reps.push(repetitions);
                        }
                        repetitions = reps;
                    }
                    else {
                        // throw warning if too long, and then use the first N
                        repetitions = repetitions.slice(0, array.length);
                    }
                }
            }
        }
        // should be clear at this point to assume that array and repetitions are arrays with == length
        let allsamples = [];
        for (let i = 0; i < array.length; i++) {
            for (let j = 0; j < repetitions[i]; j++) {
                if (array[i] == null || typeof array[i] != "object") {
                    allsamples.push(array[i]);
                }
                else {
                    allsamples.push(Object.assign({}, array[i]));
                }
            }
        }
        let out = shuffle(allsamples);
        if (unpack) {
            out = unpackArray(out);
        }
        return out;
    }
    function shuffle(array) {
        if (!Array.isArray(array)) {
            console.error("Argument to shuffle() must be an array.");
        }
        const copy_array = array.slice(0);
        let m = copy_array.length, t, i;
        // While there remain elements to shuffle…
        while (m) {
            // Pick a remaining element…
            i = Math.floor(Math.random() * m--);
            // And swap it with the current element.
            t = copy_array[m];
            copy_array[m] = copy_array[i];
            copy_array[i] = t;
        }
        return copy_array;
    }
    function shuffleNoRepeats(arr, equalityTest) {
        if (!Array.isArray(arr)) {
            console.error("First argument to shuffleNoRepeats() must be an array.");
        }
        if (typeof equalityTest !== "undefined" && typeof equalityTest !== "function") {
            console.error("Second argument to shuffleNoRepeats() must be a function.");
        }
        // define a default equalityTest
        if (typeof equalityTest == "undefined") {
            equalityTest = function (a, b) {
                if (a === b) {
                    return true;
                }
                else {
                    return false;
                }
            };
        }
        const random_shuffle = shuffle(arr);
        for (let i = 0; i < random_shuffle.length - 1; i++) {
            if (equalityTest(random_shuffle[i], random_shuffle[i + 1])) {
                // neighbors are equal, pick a new random neighbor to swap (not the first or last element, to avoid edge cases)
                let random_pick = Math.floor(Math.random() * (random_shuffle.length - 2)) + 1;
                // test to make sure the new neighbor isn't equal to the old one
                while (equalityTest(random_shuffle[i + 1], random_shuffle[random_pick]) ||
                    equalityTest(random_shuffle[i + 1], random_shuffle[random_pick + 1]) ||
                    equalityTest(random_shuffle[i + 1], random_shuffle[random_pick - 1])) {
                    random_pick = Math.floor(Math.random() * (random_shuffle.length - 2)) + 1;
                }
                const new_neighbor = random_shuffle[random_pick];
                random_shuffle[random_pick] = random_shuffle[i + 1];
                random_shuffle[i + 1] = new_neighbor;
            }
        }
        return random_shuffle;
    }
    function shuffleAlternateGroups(arr_groups, random_group_order = false) {
        const n_groups = arr_groups.length;
        if (n_groups == 1) {
            console.warn("shuffleAlternateGroups() was called with only one group. Defaulting to simple shuffle.");
            return shuffle(arr_groups[0]);
        }
        let group_order = [];
        for (let i = 0; i < n_groups; i++) {
            group_order.push(i);
        }
        if (random_group_order) {
            group_order = shuffle(group_order);
        }
        const randomized_groups = [];
        let min_length = null;
        for (let i = 0; i < n_groups; i++) {
            min_length =
                min_length === null ? arr_groups[i].length : Math.min(min_length, arr_groups[i].length);
            randomized_groups.push(shuffle(arr_groups[i]));
        }
        const out = [];
        for (let i = 0; i < min_length; i++) {
            for (let j = 0; j < group_order.length; j++) {
                out.push(randomized_groups[group_order[j]][i]);
            }
        }
        return out;
    }
    function sampleWithoutReplacement(arr, size) {
        if (!Array.isArray(arr)) {
            console.error("First argument to sampleWithoutReplacement() must be an array");
        }
        if (size > arr.length) {
            console.error("Cannot take a sample larger than the size of the set of items to sample.");
        }
        return shuffle(arr).slice(0, size);
    }
    function sampleWithReplacement(arr, size, weights) {
        if (!Array.isArray(arr)) {
            console.error("First argument to sampleWithReplacement() must be an array");
        }
        const normalized_weights = [];
        if (typeof weights !== "undefined") {
            if (weights.length !== arr.length) {
                console.error("The length of the weights array must equal the length of the array " +
                    "to be sampled from.");
            }
            let weight_sum = 0;
            for (const weight of weights) {
                weight_sum += weight;
            }
            for (const weight of weights) {
                normalized_weights.push(weight / weight_sum);
            }
        }
        else {
            for (let i = 0; i < arr.length; i++) {
                normalized_weights.push(1 / arr.length);
            }
        }
        const cumulative_weights = [normalized_weights[0]];
        for (let i = 1; i < normalized_weights.length; i++) {
            cumulative_weights.push(normalized_weights[i] + cumulative_weights[i - 1]);
        }
        const samp = [];
        for (let i = 0; i < size; i++) {
            const rnd = Math.random();
            let index = 0;
            while (rnd > cumulative_weights[index]) {
                index++;
            }
            samp.push(arr[index]);
        }
        return samp;
    }
    function factorial(factors, repetitions = 1, unpack = false) {
        let design = [{}];
        for (const [factorName, factor] of Object.entries(factors)) {
            const new_design = [];
            for (const level of factor) {
                for (const cell of design) {
                    new_design.push(Object.assign(Object.assign({}, cell), { [factorName]: level }));
                }
            }
            design = new_design;
        }
        return repeat(design, repetitions, unpack);
    }
    function randomID(length = 32) {
        let result = "";
        const chars = "0123456789abcdefghjklmnopqrstuvwxyz";
        for (let i = 0; i < length; i++) {
            result += chars[Math.floor(Math.random() * chars.length)];
        }
        return result;
    }
    function unpackArray(array) {
        const out = {};
        for (const x of array) {
            for (const key of Object.keys(x)) {
                if (typeof out[key] === "undefined") {
                    out[key] = [];
                }
                out[key].push(x[key]);
            }
        }
        return out;
    }

    var randomization = /*#__PURE__*/Object.freeze({
        __proto__: null,
        repeat: repeat,
        shuffle: shuffle,
        shuffleNoRepeats: shuffleNoRepeats,
        shuffleAlternateGroups: shuffleAlternateGroups,
        sampleWithoutReplacement: sampleWithoutReplacement,
        sampleWithReplacement: sampleWithReplacement,
        factorial: factorial,
        randomID: randomID
    });

    /**
     * Gets information about the Mechanical Turk Environment, HIT, Assignment, and Worker
     * by parsing the URL variables that Mechanical Turk generates.
     * @returns An object containing information about the Mechanical Turk Environment, HIT, Assignment, and Worker.
     */
    function turkInfo() {
        const turk = {
            previewMode: false,
            outsideTurk: false,
            hitId: "INVALID_URL_PARAMETER",
            assignmentId: "INVALID_URL_PARAMETER",
            workerId: "INVALID_URL_PARAMETER",
            turkSubmitTo: "INVALID_URL_PARAMETER",
        };
        const param = function (url, name) {
            name = name.replace(/[\[]/, "\\[").replace(/[\]]/, "\\]");
            const regexS = "[\\?&]" + name + "=([^&#]*)";
            const regex = new RegExp(regexS);
            const results = regex.exec(url);
            return results == null ? "" : results[1];
        };
        const src = param(window.location.href, "assignmentId")
            ? window.location.href
            : document.referrer;
        const keys = ["assignmentId", "hitId", "workerId", "turkSubmitTo"];
        keys.map(function (key) {
            turk[key] = unescape(param(src, key));
        });
        turk.previewMode = turk.assignmentId == "ASSIGNMENT_ID_NOT_AVAILABLE";
        turk.outsideTurk =
            !turk.previewMode && turk.hitId === "" && turk.assignmentId == "" && turk.workerId == "";
        return turk;
    }
    /**
     * Send data to Mechnical Turk for storage.
     * @param data An object containing `key:value` pairs to send to Mechanical Turk. Values
     * cannot contain nested objects, arrays, or functions.
     * @returns Nothing
     */
    function submitToTurk(data) {
        const turk = turkInfo();
        const assignmentId = turk.assignmentId;
        const turkSubmitTo = turk.turkSubmitTo;
        if (!assignmentId || !turkSubmitTo)
            return;
        const form = document.createElement("form");
        form.method = "POST";
        form.action = turkSubmitTo + "/mturk/externalSubmit?assignmentId=" + assignmentId;
        for (const key in data) {
            if (data.hasOwnProperty(key)) {
                const hiddenField = document.createElement("input");
                hiddenField.type = "hidden";
                hiddenField.name = key;
                hiddenField.id = key;
                hiddenField.value = data[key];
                form.appendChild(hiddenField);
            }
        }
        document.body.appendChild(form);
        form.submit();
    }

    var turk = /*#__PURE__*/Object.freeze({
        __proto__: null,
        turkInfo: turkInfo,
        submitToTurk: submitToTurk
    });

    class TimelineNode {
        // constructor
        constructor(jsPsych, parameters, parent, relativeID) {
            this.jsPsych = jsPsych;
            // track progress through the node
            this.progress = {
                current_location: -1,
                current_variable_set: 0,
                current_repetition: 0,
                current_iteration: 0,
                done: false,
            };
            // store a link to the parent of this node
            this.parent_node = parent;
            // create the ID for this node
            this.relative_id = typeof parent === "undefined" ? 0 : relativeID;
            // check if there is a timeline parameter
            // if there is, then this node has its own timeline
            if (typeof parameters.timeline !== "undefined") {
                // create timeline properties
                this.timeline_parameters = {
                    timeline: [],
                    loop_function: parameters.loop_function,
                    conditional_function: parameters.conditional_function,
                    sample: parameters.sample,
                    randomize_order: typeof parameters.randomize_order == "undefined" ? false : parameters.randomize_order,
                    repetitions: typeof parameters.repetitions == "undefined" ? 1 : parameters.repetitions,
                    timeline_variables: typeof parameters.timeline_variables == "undefined"
                        ? [{}]
                        : parameters.timeline_variables,
                    on_timeline_finish: parameters.on_timeline_finish,
                    on_timeline_start: parameters.on_timeline_start,
                };
                this.setTimelineVariablesOrder();
                // extract all of the node level data and parameters
                // but remove all of the timeline-level specific information
                // since this will be used to copy things down hierarchically
                var node_data = Object.assign({}, parameters);
                delete node_data.timeline;
                delete node_data.conditional_function;
                delete node_data.loop_function;
                delete node_data.randomize_order;
                delete node_data.repetitions;
                delete node_data.timeline_variables;
                delete node_data.sample;
                delete node_data.on_timeline_start;
                delete node_data.on_timeline_finish;
                this.node_trial_data = node_data; // store for later...
                // create a TimelineNode for each element in the timeline
                for (var i = 0; i < parameters.timeline.length; i++) {
                    // merge parameters
                    var merged_parameters = Object.assign({}, node_data, parameters.timeline[i]);
                    // merge any data from the parent node into child nodes
                    if (typeof node_data.data == "object" && typeof parameters.timeline[i].data == "object") {
                        var merged_data = Object.assign({}, node_data.data, parameters.timeline[i].data);
                        merged_parameters.data = merged_data;
                    }
                    this.timeline_parameters.timeline.push(new TimelineNode(this.jsPsych, merged_parameters, this, i));
                }
            }
            // if there is no timeline parameter, then this node is a trial node
            else {
                // check to see if a valid trial type is defined
                if (typeof parameters.type === "undefined") {
                    console.error('Trial level node is missing the "type" parameter. The parameters for the node are: ' +
                        JSON.stringify(parameters));
                }
                // create a deep copy of the parameters for the trial
                this.trial_parameters = Object.assign({}, parameters);
            }
        }
        // recursively get the next trial to run.
        // if this node is a leaf (trial), then return the trial.
        // otherwise, recursively find the next trial in the child timeline.
        trial() {
            if (typeof this.timeline_parameters == "undefined") {
                // returns a clone of the trial_parameters to
                // protect functions.
                return deepCopy(this.trial_parameters);
            }
            else {
                if (this.progress.current_location >= this.timeline_parameters.timeline.length) {
                    return null;
                }
                else {
                    return this.timeline_parameters.timeline[this.progress.current_location].trial();
                }
            }
        }
        markCurrentTrialComplete() {
            if (typeof this.timeline_parameters === "undefined") {
                this.progress.done = true;
            }
            else {
                this.timeline_parameters.timeline[this.progress.current_location].markCurrentTrialComplete();
            }
        }
        nextRepetiton() {
            this.setTimelineVariablesOrder();
            this.progress.current_location = -1;
            this.progress.current_variable_set = 0;
            this.progress.current_repetition++;
            for (var i = 0; i < this.timeline_parameters.timeline.length; i++) {
                this.timeline_parameters.timeline[i].reset();
            }
        }
        // set the order for going through the timeline variables array
        setTimelineVariablesOrder() {
            const timeline_parameters = this.timeline_parameters;
            // check to make sure this node has variables
            if (typeof timeline_parameters === "undefined" ||
                typeof timeline_parameters.timeline_variables === "undefined") {
                return;
            }
            var order = [];
            for (var i = 0; i < timeline_parameters.timeline_variables.length; i++) {
                order.push(i);
            }
            if (typeof timeline_parameters.sample !== "undefined") {
                if (timeline_parameters.sample.type == "custom") {
                    order = timeline_parameters.sample.fn(order);
                }
                else if (timeline_parameters.sample.type == "with-replacement") {
                    order = sampleWithReplacement(order, timeline_parameters.sample.size, timeline_parameters.sample.weights);
                }
                else if (timeline_parameters.sample.type == "without-replacement") {
                    order = sampleWithoutReplacement(order, timeline_parameters.sample.size);
                }
                else if (timeline_parameters.sample.type == "fixed-repetitions") {
                    order = repeat(order, timeline_parameters.sample.size, false);
                }
                else if (timeline_parameters.sample.type == "alternate-groups") {
                    order = shuffleAlternateGroups(timeline_parameters.sample.groups, timeline_parameters.sample.randomize_group_order);
                }
                else {
                    console.error('Invalid type in timeline sample parameters. Valid options for type are "custom", "with-replacement", "without-replacement", "fixed-repetitions", and "alternate-groups"');
                }
            }
            if (timeline_parameters.randomize_order) {
                order = shuffle(order);
            }
            this.progress.order = order;
        }
        // next variable set
        nextSet() {
            this.progress.current_location = -1;
            this.progress.current_variable_set++;
            for (var i = 0; i < this.timeline_parameters.timeline.length; i++) {
                this.timeline_parameters.timeline[i].reset();
            }
        }
        // update the current trial node to be completed
        // returns true if the node is complete after advance (all subnodes are also complete)
        // returns false otherwise
        advance() {
            const progress = this.progress;
            const timeline_parameters = this.timeline_parameters;
            const internal = this.jsPsych.internal;
            // first check to see if done
            if (progress.done) {
                return true;
            }
            // if node has not started yet (progress.current_location == -1),
            // then try to start the node.
            if (progress.current_location == -1) {
                // check for on_timeline_start and conditonal function on nodes with timelines
                if (typeof timeline_parameters !== "undefined") {
                    // only run the conditional function if this is the first repetition of the timeline when
                    // repetitions > 1, and only when on the first variable set
                    if (typeof timeline_parameters.conditional_function !== "undefined" &&
                        progress.current_repetition == 0 &&
                        progress.current_variable_set == 0) {
                        internal.call_immediate = true;
                        var conditional_result = timeline_parameters.conditional_function();
                        internal.call_immediate = false;
                        // if the conditional_function() returns false, then the timeline
                        // doesn't run and is marked as complete.
                        if (conditional_result == false) {
                            progress.done = true;
                            return true;
                        }
                    }
                    // if we reach this point then the node has its own timeline and will start
                    // so we need to check if there is an on_timeline_start function if we are on the first variable set
                    if (typeof timeline_parameters.on_timeline_start !== "undefined" &&
                        progress.current_variable_set == 0) {
                        timeline_parameters.on_timeline_start();
                    }
                }
                // if we reach this point, then either the node doesn't have a timeline of the
                // conditional function returned true and it can start
                progress.current_location = 0;
                // call advance again on this node now that it is pointing to a new location
                return this.advance();
            }
            // if this node has a timeline, propogate down to the current trial.
            if (typeof timeline_parameters !== "undefined") {
                var have_node_to_run = false;
                // keep incrementing the location in the timeline until one of the nodes reached is incomplete
                while (progress.current_location < timeline_parameters.timeline.length &&
                    have_node_to_run == false) {
                    // check to see if the node currently pointed at is done
                    var target_complete = timeline_parameters.timeline[progress.current_location].advance();
                    if (!target_complete) {
                        have_node_to_run = true;
                        return false;
                    }
                    else {
                        progress.current_location++;
                    }
                }
                // if we've reached the end of the timeline (which, if the code is here, we have)
                // there are a few steps to see what to do next...
                // first, check the timeline_variables to see if we need to loop through again
                // with a new set of variables
                if (progress.current_variable_set < progress.order.length - 1) {
                    // reset the progress of the node to be with the new set
                    this.nextSet();
                    // then try to advance this node again.
                    return this.advance();
                }
                // if we're all done with the timeline_variables, then check to see if there are more repetitions
                else if (progress.current_repetition < timeline_parameters.repetitions - 1) {
                    this.nextRepetiton();
                    // check to see if there is an on_timeline_finish function
                    if (typeof timeline_parameters.on_timeline_finish !== "undefined") {
                        timeline_parameters.on_timeline_finish();
                    }
                    return this.advance();
                }
                // if we're all done with the repetitions...
                else {
                    // check to see if there is an on_timeline_finish function
                    if (typeof timeline_parameters.on_timeline_finish !== "undefined") {
                        timeline_parameters.on_timeline_finish();
                    }
                    // if we're all done with the repetitions, check if there is a loop function.
                    if (typeof timeline_parameters.loop_function !== "undefined") {
                        internal.call_immediate = true;
                        if (timeline_parameters.loop_function(this.generatedData())) {
                            this.reset();
                            internal.call_immediate = false;
                            return this.parent_node.advance();
                        }
                        else {
                            progress.done = true;
                            internal.call_immediate = false;
                            return true;
                        }
                    }
                }
                // no more loops on this timeline, we're done!
                progress.done = true;
                return true;
            }
        }
        // check the status of the done flag
        isComplete() {
            return this.progress.done;
        }
        // getter method for timeline variables
        getTimelineVariableValue(variable_name) {
            if (typeof this.timeline_parameters == "undefined") {
                return undefined;
            }
            var v = this.timeline_parameters.timeline_variables[this.progress.order[this.progress.current_variable_set]][variable_name];
            return v;
        }
        // recursive upward search for timeline variables
        findTimelineVariable(variable_name) {
            var v = this.getTimelineVariableValue(variable_name);
            if (typeof v == "undefined") {
                if (typeof this.parent_node !== "undefined") {
                    return this.parent_node.findTimelineVariable(variable_name);
                }
                else {
                    return undefined;
                }
            }
            else {
                return v;
            }
        }
        // recursive downward search for active trial to extract timeline variable
        timelineVariable(variable_name) {
            if (typeof this.timeline_parameters == "undefined") {
                return this.findTimelineVariable(variable_name);
            }
            else {
                // if progress.current_location is -1, then the timeline variable is being evaluated
                // in a function that runs prior to the trial starting, so we should treat that trial
                // as being the active trial for purposes of finding the value of the timeline variable
                var loc = Math.max(0, this.progress.current_location);
                // if loc is greater than the number of elements on this timeline, then the timeline
                // variable is being evaluated in a function that runs after the trial on the timeline
                // are complete but before advancing to the next (like a loop_function).
                // treat the last active trial as the active trial for this purpose.
                if (loc == this.timeline_parameters.timeline.length) {
                    loc = loc - 1;
                }
                // now find the variable
                return this.timeline_parameters.timeline[loc].timelineVariable(variable_name);
            }
        }
        // recursively get all the timeline variables for this trial
        allTimelineVariables() {
            var all_tvs = this.allTimelineVariablesNames();
            var all_tvs_vals = {};
            for (var i = 0; i < all_tvs.length; i++) {
                all_tvs_vals[all_tvs[i]] = this.timelineVariable(all_tvs[i]);
            }
            return all_tvs_vals;
        }
        // helper to get all the names at this stage.
        allTimelineVariablesNames(so_far = []) {
            if (typeof this.timeline_parameters !== "undefined") {
                so_far = so_far.concat(Object.keys(this.timeline_parameters.timeline_variables[this.progress.order[this.progress.current_variable_set]]));
                // if progress.current_location is -1, then the timeline variable is being evaluated
                // in a function that runs prior to the trial starting, so we should treat that trial
                // as being the active trial for purposes of finding the value of the timeline variable
                var loc = Math.max(0, this.progress.current_location);
                // if loc is greater than the number of elements on this timeline, then the timeline
                // variable is being evaluated in a function that runs after the trial on the timeline
                // are complete but before advancing to the next (like a loop_function).
                // treat the last active trial as the active trial for this purpose.
                if (loc == this.timeline_parameters.timeline.length) {
                    loc = loc - 1;
                }
                // now find the variable
                return this.timeline_parameters.timeline[loc].allTimelineVariablesNames(so_far);
            }
            if (typeof this.timeline_parameters == "undefined") {
                return so_far;
            }
        }
        // recursively get the number of **trials** contained in the timeline
        // assuming that while loops execute exactly once and if conditionals
        // always run
        length() {
            var length = 0;
            if (typeof this.timeline_parameters !== "undefined") {
                for (var i = 0; i < this.timeline_parameters.timeline.length; i++) {
                    length += this.timeline_parameters.timeline[i].length();
                }
            }
            else {
                return 1;
            }
            return length;
        }
        // return the percentage of trials completed, grouped at the first child level
        // counts a set of trials as complete when the child node is done
        percentComplete() {
            var total_trials = this.length();
            var completed_trials = 0;
            for (var i = 0; i < this.timeline_parameters.timeline.length; i++) {
                if (this.timeline_parameters.timeline[i].isComplete()) {
                    completed_trials += this.timeline_parameters.timeline[i].length();
                }
            }
            return (completed_trials / total_trials) * 100;
        }
        // resets the node and all subnodes to original state
        // but increments the current_iteration counter
        reset() {
            this.progress.current_location = -1;
            this.progress.current_repetition = 0;
            this.progress.current_variable_set = 0;
            this.progress.current_iteration++;
            this.progress.done = false;
            this.setTimelineVariablesOrder();
            if (typeof this.timeline_parameters != "undefined") {
                for (var i = 0; i < this.timeline_parameters.timeline.length; i++) {
                    this.timeline_parameters.timeline[i].reset();
                }
            }
        }
        // mark this node as finished
        end() {
            this.progress.done = true;
        }
        // recursively end whatever sub-node is running the current trial
        endActiveNode() {
            if (typeof this.timeline_parameters == "undefined") {
                this.end();
                this.parent_node.end();
            }
            else {
                this.timeline_parameters.timeline[this.progress.current_location].endActiveNode();
            }
        }
        // get a unique ID associated with this node
        // the ID reflects the current iteration through this node.
        ID() {
            var id = "";
            if (typeof this.parent_node == "undefined") {
                return "0." + this.progress.current_iteration;
            }
            else {
                id += this.parent_node.ID() + "-";
                id += this.relative_id + "." + this.progress.current_iteration;
                return id;
            }
        }
        // get the ID of the active trial
        activeID() {
            if (typeof this.timeline_parameters == "undefined") {
                return this.ID();
            }
            else {
                return this.timeline_parameters.timeline[this.progress.current_location].activeID();
            }
        }
        // get all the data generated within this node
        generatedData() {
            return this.jsPsych.data.getDataByTimelineNode(this.ID());
        }
        // get all the trials of a particular type
        trialsOfType(type) {
            if (typeof this.timeline_parameters == "undefined") {
                if (this.trial_parameters.type == type) {
                    return this.trial_parameters;
                }
                else {
                    return [];
                }
            }
            else {
                var trials = [];
                for (var i = 0; i < this.timeline_parameters.timeline.length; i++) {
                    var t = this.timeline_parameters.timeline[i].trialsOfType(type);
                    trials = trials.concat(t);
                }
                return trials;
            }
        }
        // add new trials to end of this timeline
        insert(parameters) {
            if (typeof this.timeline_parameters === "undefined") {
                console.error("Cannot add new trials to a trial-level node.");
            }
            else {
                this.timeline_parameters.timeline.push(new TimelineNode(this.jsPsych, Object.assign(Object.assign({}, this.node_trial_data), parameters), this, this.timeline_parameters.timeline.length));
            }
        }
    }

    function delay(ms) {
        return new Promise((resolve) => setTimeout(resolve, ms));
    }
    class JsPsych {
        constructor(options) {
            this.extensions = {};
            this.turk = turk;
            this.randomization = randomization;
            this.utils = utils;
            //
            // private variables
            //
            /**
             * options
             */
            this.opts = {};
            // flow control
            this.global_trial_index = 0;
            this.current_trial = {};
            this.current_trial_finished = false;
            /**
             * is the experiment paused?
             */
            this.paused = false;
            this.waiting = false;
            /**
             * is the page retrieved directly via file:// protocol (true) or hosted on a server (false)?
             */
            this.file_protocol = false;
            // storing a single webaudio context to prevent problems with multiple inits
            // of jsPsych
            this.webaudio_context = null;
            this.internal = {
                /**
                 * this flag is used to determine whether we are in a scope where
                 * jsPsych.timelineVariable() should be executed immediately or
                 * whether it should return a function to access the variable later.
                 *
                 **/
                call_immediate: false,
            };
            this.progress_bar_amount = 0;
            // override default options if user specifies an option
            options = Object.assign({ display_element: undefined, on_finish: () => { }, on_trial_start: () => { }, on_trial_finish: () => { }, on_data_update: () => { }, on_interaction_data_update: () => { }, on_close: () => { }, use_webaudio: true, exclusions: {}, show_progress_bar: false, message_progress_bar: "Completion Progress", auto_update_progress_bar: true, default_iti: 0, minimum_valid_rt: 0, experiment_width: null, override_safe_mode: false, case_sensitive_responses: false, extensions: [] }, options);
            this.opts = options;
            autoBind(this); // so we can pass JsPsych methods as callbacks and `this` remains the JsPsych instance
            this.webaudio_context =
                typeof window !== "undefined" && typeof window.AudioContext !== "undefined"
                    ? new AudioContext()
                    : null;
            // detect whether page is running in browser as a local file, and if so, disable web audio and video preloading to prevent CORS issues
            if (window.location.protocol == "file:" &&
                (options.override_safe_mode === false || typeof options.override_safe_mode === "undefined")) {
                options.use_webaudio = false;
                this.file_protocol = true;
                console.warn("jsPsych detected that it is running via the file:// protocol and not on a web server. " +
                    "To prevent issues with cross-origin requests, Web Audio and video preloading have been disabled. " +
                    "If you would like to override this setting, you can set 'override_safe_mode' to 'true' in initJsPsych. " +
                    "For more information, see: https://www.jspsych.org/overview/running-experiments");
            }
            // initialize modules
            this.data = new JsPsychData(this);
            this.pluginAPI = createJointPluginAPIObject(this);
            // create instances of extensions
            for (const extension of options.extensions) {
                this.extensions[extension.type.info.name] = new extension.type(this);
            }
            // initialize audio context based on options and browser capabilities
            this.pluginAPI.initAudio();
        }
        version() {
            return version;
        }
        /**
         * Starts an experiment using the provided timeline and returns a promise that is resolved when
         * the experiment is finished.
         *
         * @param timeline The timeline to be run
         */
        run(timeline) {
            return __awaiter(this, void 0, void 0, function* () {
                if (typeof timeline === "undefined") {
                    console.error("No timeline declared in jsPsych.run. Cannot start experiment.");
                }
                if (timeline.length === 0) {
                    console.error("No trials have been added to the timeline (the timeline is an empty array). Cannot start experiment.");
                }
                // create experiment timeline
                this.timelineDescription = timeline;
                this.timeline = new TimelineNode(this, { timeline });
                yield this.prepareDom();
                yield this.checkExclusions(this.opts.exclusions);
                yield this.loadExtensions(this.opts.extensions);
                document.documentElement.setAttribute("jspsych", "present");
                this.startExperiment();
                yield this.finished;
            });
        }
        getProgress() {
            return {
                total_trials: typeof this.timeline === "undefined" ? undefined : this.timeline.length(),
                current_trial_global: this.global_trial_index,
                percent_complete: typeof this.timeline === "undefined" ? 0 : this.timeline.percentComplete(),
            };
        }
        getStartTime() {
            return this.exp_start_time;
        }
        getTotalTime() {
            if (typeof this.exp_start_time === "undefined") {
                return 0;
            }
            return new Date().getTime() - this.exp_start_time.getTime();
        }
        getDisplayElement() {
            return this.DOM_target;
        }
        getDisplayContainerElement() {
            return this.DOM_container;
        }
        finishTrial(data = {}) {
            if (this.current_trial_finished) {
                return;
            }
            this.current_trial_finished = true;
            // remove any CSS classes that were added to the DOM via css_classes parameter
            if (typeof this.current_trial.css_classes !== "undefined" &&
                Array.isArray(this.current_trial.css_classes)) {
                this.DOM_target.classList.remove(...this.current_trial.css_classes);
            }
            // write the data from the trial
            this.data.write(data);
            // get back the data with all of the defaults in
            const trial_data = this.data.get().filter({ trial_index: this.global_trial_index });
            // for trial-level callbacks, we just want to pass in a reference to the values
            // of the DataCollection, for easy access and editing.
            const trial_data_values = trial_data.values()[0];
            const current_trial = this.current_trial;
            if (typeof current_trial.save_trial_parameters === "object") {
                for (const key of Object.keys(current_trial.save_trial_parameters)) {
                    const key_val = current_trial.save_trial_parameters[key];
                    if (key_val === true) {
                        if (typeof current_trial[key] === "undefined") {
                            console.warn(`Invalid parameter specified in save_trial_parameters. Trial has no property called "${key}".`);
                        }
                        else if (typeof current_trial[key] === "function") {
                            trial_data_values[key] = current_trial[key].toString();
                        }
                        else {
                            trial_data_values[key] = current_trial[key];
                        }
                    }
                    if (key_val === false) {
                        // we don't allow internal_node_id or trial_index to be deleted because it would break other things
                        if (key !== "internal_node_id" && key !== "trial_index") {
                            delete trial_data_values[key];
                        }
                    }
                }
            }
            // handle extension callbacks
            if (Array.isArray(current_trial.extensions)) {
                for (const extension of current_trial.extensions) {
                    const ext_data_values = this.extensions[extension.type.info.name].on_finish(extension.params);
                    Object.assign(trial_data_values, ext_data_values);
                }
            }
            // about to execute lots of callbacks, so switch context.
            this.internal.call_immediate = true;
            // handle callback at plugin level
            if (typeof current_trial.on_finish === "function") {
                current_trial.on_finish(trial_data_values);
            }
            // handle callback at whole-experiment level
            this.opts.on_trial_finish(trial_data_values);
            // after the above callbacks are complete, then the data should be finalized
            // for this trial. call the on_data_update handler, passing in the same
            // data object that just went through the trial's finish handlers.
            this.opts.on_data_update(trial_data_values);
            // done with callbacks
            this.internal.call_immediate = false;
            // wait for iti
            if (typeof current_trial.post_trial_gap === null ||
                typeof current_trial.post_trial_gap === "undefined") {
                if (this.opts.default_iti > 0) {
                    setTimeout(this.nextTrial, this.opts.default_iti);
                }
                else {
                    this.nextTrial();
                }
            }
            else {
                if (current_trial.post_trial_gap > 0) {
                    setTimeout(this.nextTrial, current_trial.post_trial_gap);
                }
                else {
                    this.nextTrial();
                }
            }
        }
        endExperiment(end_message) {
            this.timeline.end_message = end_message;
            this.timeline.end();
            this.pluginAPI.cancelAllKeyboardResponses();
            this.pluginAPI.clearAllTimeouts();
            this.finishTrial();
        }
        endCurrentTimeline() {
            this.timeline.endActiveNode();
        }
        getCurrentTrial() {
            return this.current_trial;
        }
        getInitSettings() {
            return this.opts;
        }
        getCurrentTimelineNodeID() {
            return this.timeline.activeID();
        }
        timelineVariable(varname, immediate = false) {
            if (this.internal.call_immediate || immediate === true) {
                return this.timeline.timelineVariable(varname);
            }
            else {
                return {
                    timelineVariablePlaceholder: true,
                    timelineVariableFunction: () => this.timeline.timelineVariable(varname),
                };
            }
        }
        getAllTimelineVariables() {
            return this.timeline.allTimelineVariables();
        }
        addNodeToEndOfTimeline(new_timeline, preload_callback) {
            this.timeline.insert(new_timeline);
        }
        pauseExperiment() {
            this.paused = true;
        }
        resumeExperiment() {
            this.paused = false;
            if (this.waiting) {
                this.waiting = false;
                this.nextTrial();
            }
        }
        loadFail(message) {
            message = message || "<p>The experiment failed to load.</p>";
            this.DOM_target.innerHTML = message;
        }
        getSafeModeStatus() {
            return this.file_protocol;
        }
        getTimeline() {
            return this.timelineDescription;
        }
        prepareDom() {
            return __awaiter(this, void 0, void 0, function* () {
                // Wait until the document is ready
                if (document.readyState !== "complete") {
                    yield new Promise((resolve) => {
                        window.addEventListener("load", resolve);
                    });
                }
                const options = this.opts;
                // set DOM element where jsPsych will render content
                // if undefined, then jsPsych will use the <body> tag and the entire page
                if (typeof options.display_element === "undefined") {
                    // check if there is a body element on the page
                    const body = document.querySelector("body");
                    if (body === null) {
                        document.documentElement.appendChild(document.createElement("body"));
                    }
                    // using the full page, so we need the HTML element to
                    // have 100% height, and body to be full width and height with
                    // no margin
                    document.querySelector("html").style.height = "100%";
                    document.querySelector("body").style.margin = "0px";
                    document.querySelector("body").style.height = "100%";
                    document.querySelector("body").style.width = "100%";
                    options.display_element = document.querySelector("body");
                }
                else {
                    // make sure that the display element exists on the page
                    const display = options.display_element instanceof Element
                        ? options.display_element
                        : document.querySelector("#" + options.display_element);
                    if (display === null) {
                        console.error("The display_element specified in initJsPsych() does not exist in the DOM.");
                    }
                    else {
                        options.display_element = display;
                    }
                }
                options.display_element.innerHTML =
                    '<div class="jspsych-content-wrapper"><div id="jspsych-content"></div></div>';
                this.DOM_container = options.display_element;
                this.DOM_target = document.querySelector("#jspsych-content");
                // set experiment_width if not null
                if (options.experiment_width !== null) {
                    this.DOM_target.style.width = options.experiment_width + "px";
                }
                // add tabIndex attribute to scope event listeners
                options.display_element.tabIndex = 0;
                // add CSS class to DOM_target
                if (options.display_element.className.indexOf("jspsych-display-element") === -1) {
                    options.display_element.className += " jspsych-display-element";
                }
                this.DOM_target.className += "jspsych-content";
                // create listeners for user browser interaction
                this.data.createInteractionListeners();
                // add event for closing window
                window.addEventListener("beforeunload", options.on_close);
            });
        }
        loadExtensions(extensions) {
            return __awaiter(this, void 0, void 0, function* () {
                // run the .initialize method of any extensions that are in use
                // these should return a Promise to indicate when loading is complete
                try {
                    yield Promise.all(extensions.map((extension) => this.extensions[extension.type.info.name].initialize(extension.params || {})));
                }
                catch (error_message) {
                    console.error(error_message);
                    throw new Error(error_message);
                }
            });
        }
        startExperiment() {
            this.finished = new Promise((resolve) => {
                this.resolveFinishedPromise = resolve;
            });
            // show progress bar if requested
            if (this.opts.show_progress_bar === true) {
                this.drawProgressBar(this.opts.message_progress_bar);
            }
            // record the start time
            this.exp_start_time = new Date();
            // begin!
            this.timeline.advance();
            this.doTrial(this.timeline.trial());
        }
        finishExperiment() {
            const finish_result = this.opts.on_finish(this.data.get());
            const done_handler = () => {
                if (typeof this.timeline.end_message !== "undefined") {
                    this.DOM_target.innerHTML = this.timeline.end_message;
                }
                this.resolveFinishedPromise();
            };
            if (finish_result) {
                Promise.resolve(finish_result).then(done_handler);
            }
            else {
                done_handler();
            }
        }
        nextTrial() {
            // if experiment is paused, don't do anything.
            if (this.paused) {
                this.waiting = true;
                return;
            }
            this.global_trial_index++;
            // advance timeline
            this.timeline.markCurrentTrialComplete();
            const complete = this.timeline.advance();
            // update progress bar if shown
            if (this.opts.show_progress_bar === true && this.opts.auto_update_progress_bar === true) {
                this.updateProgressBar();
            }
            // check if experiment is over
            if (complete) {
                this.finishExperiment();
                return;
            }
            this.doTrial(this.timeline.trial());
        }
        doTrial(trial) {
            this.current_trial = trial;
            this.current_trial_finished = false;
            // process all timeline variables for this trial
            this.evaluateTimelineVariables(trial);
            // instantiate the plugin for this trial
            trial.type = Object.assign(Object.assign({}, autoBind(new trial.type(this))), { info: trial.type.info });
            // evaluate variables that are functions
            this.evaluateFunctionParameters(trial);
            // get default values for parameters
            this.setDefaultValues(trial);
            // about to execute callbacks
            this.internal.call_immediate = true;
            // call experiment wide callback
            this.opts.on_trial_start(trial);
            // call trial specific callback if it exists
            if (typeof trial.on_start === "function") {
                trial.on_start(trial);
            }
            // call any on_start functions for extensions
            if (Array.isArray(trial.extensions)) {
                for (const extension of trial.extensions) {
                    this.extensions[extension.type.info.name].on_start(extension.params);
                }
            }
            // apply the focus to the element containing the experiment.
            this.DOM_container.focus();
            // reset the scroll on the DOM target
            this.DOM_target.scrollTop = 0;
            // add CSS classes to the DOM_target if they exist in trial.css_classes
            if (typeof trial.css_classes !== "undefined") {
                if (!Array.isArray(trial.css_classes) && typeof trial.css_classes === "string") {
                    trial.css_classes = [trial.css_classes];
                }
                if (Array.isArray(trial.css_classes)) {
                    this.DOM_target.classList.add(...trial.css_classes);
                }
            }
            // setup on_load event callback
            const load_callback = () => {
                if (typeof trial.on_load === "function") {
                    trial.on_load();
                }
                // call any on_load functions for extensions
                if (Array.isArray(trial.extensions)) {
                    for (const extension of trial.extensions) {
                        this.extensions[extension.type.info.name].on_load(extension.params);
                    }
                }
            };
            const trial_complete = trial.type.trial(this.DOM_target, trial, load_callback);
            // see if trial_complete is a Promise by looking for .then() function
            const is_promise = trial_complete && typeof trial_complete.then == "function";
            if (!is_promise) {
                load_callback();
            }
            // done with callbacks
            this.internal.call_immediate = false;
        }
        evaluateTimelineVariables(trial) {
            for (const key of Object.keys(trial)) {
                // timeline variables on the root level
                if (typeof trial[key] === "object" &&
                    trial[key] !== null &&
                    typeof trial[key].timelineVariablePlaceholder !== "undefined") {
                    /*trial[key].toString().replace(/\s/g, "") ==
                      "function(){returntimeline.timelineVariable(varname);}"
                  )*/ trial[key] = trial[key].timelineVariableFunction();
                }
                // timeline variables that are nested in objects
                if (typeof trial[key] === "object" && trial[key] !== null) {
                    this.evaluateTimelineVariables(trial[key]);
                }
            }
        }
        evaluateFunctionParameters(trial) {
            // set a flag so that jsPsych.timelineVariable() is immediately executed in this context
            this.internal.call_immediate = true;
            // iterate over each parameter
            for (const key of Object.keys(trial)) {
                // check to make sure parameter is not "type", since that was eval'd above.
                if (key !== "type") {
                    // this if statement is checking to see if the parameter type is expected to be a function, in which case we should NOT evaluate it.
                    // the first line checks if the parameter is defined in the universalPluginParameters set
                    // the second line checks the plugin-specific parameters
                    if (typeof universalPluginParameters[key] !== "undefined" &&
                        universalPluginParameters[key].type !== exports.ParameterType.FUNCTION) {
                        trial[key] = this.replaceFunctionsWithValues(trial[key], null);
                    }
                    if (typeof trial.type.info.parameters[key] !== "undefined" &&
                        trial.type.info.parameters[key].type !== exports.ParameterType.FUNCTION) {
                        trial[key] = this.replaceFunctionsWithValues(trial[key], trial.type.info.parameters[key]);
                    }
                }
            }
            // reset so jsPsych.timelineVariable() is no longer immediately executed
            this.internal.call_immediate = false;
        }
        replaceFunctionsWithValues(obj, info) {
            // null typeof is 'object' (?!?!), so need to run this first!
            if (obj === null) {
                return obj;
            }
            // arrays
            else if (Array.isArray(obj)) {
                for (let i = 0; i < obj.length; i++) {
                    obj[i] = this.replaceFunctionsWithValues(obj[i], info);
                }
            }
            // objects
            else if (typeof obj === "object") {
                if (info === null || !info.nested) {
                    for (const key of Object.keys(obj)) {
                        if (key === "type") {
                            // Ignore the object's `type` field because it contains a plugin and we do not want to
                            // call plugin functions
                            continue;
                        }
                        obj[key] = this.replaceFunctionsWithValues(obj[key], null);
                    }
                }
                else {
                    for (const key of Object.keys(obj)) {
                        if (typeof info.nested[key] === "object" &&
                            info.nested[key].type !== exports.ParameterType.FUNCTION) {
                            obj[key] = this.replaceFunctionsWithValues(obj[key], info.nested[key]);
                        }
                    }
                }
            }
            else if (typeof obj === "function") {
                return obj();
            }
            return obj;
        }
        setDefaultValues(trial) {
            for (const param in trial.type.info.parameters) {
                // check if parameter is complex with nested defaults
                if (trial.type.info.parameters[param].type === exports.ParameterType.COMPLEX) {
                    if (trial.type.info.parameters[param].array === true) {
                        // iterate over each entry in the array
                        trial[param].forEach(function (ip, i) {
                            // check each parameter in the plugin description
                            for (const p in trial.type.info.parameters[param].nested) {
                                if (typeof trial[param][i][p] === "undefined" || trial[param][i][p] === null) {
                                    if (typeof trial.type.info.parameters[param].nested[p].default === "undefined") {
                                        console.error("You must specify a value for the " +
                                            p +
                                            " parameter (nested in the " +
                                            param +
                                            " parameter) in the " +
                                            trial.type +
                                            " plugin.");
                                    }
                                    else {
                                        trial[param][i][p] = trial.type.info.parameters[param].nested[p].default;
                                    }
                                }
                            }
                        });
                    }
                }
                // if it's not nested, checking is much easier and do that here:
                else if (typeof trial[param] === "undefined" || trial[param] === null) {
                    if (typeof trial.type.info.parameters[param].default === "undefined") {
                        console.error("You must specify a value for the " +
                            param +
                            " parameter in the " +
                            trial.type.info.name +
                            " plugin.");
                    }
                    else {
                        trial[param] = trial.type.info.parameters[param].default;
                    }
                }
            }
        }
        checkExclusions(exclusions) {
            return __awaiter(this, void 0, void 0, function* () {
                // MINIMUM SIZE
                if (exclusions.min_width || exclusions.min_height) {
                    const mw = exclusions.min_width || 0;
                    const mh = exclusions.min_height || 0;
                    if (window.innerWidth < mw || window.innerHeight < mh) {
                        this.getDisplayElement().innerHTML =
                            "<p>Your browser window is too small to complete this experiment. " +
                                "Please maximize the size of your browser window. If your browser window is already maximized, " +
                                "you will not be able to complete this experiment.</p>" +
                                "<p>The minimum width is " +
                                mw +
                                "px. Your current width is " +
                                window.innerWidth +
                                "px.</p>" +
                                "<p>The minimum height is " +
                                mh +
                                "px. Your current height is " +
                                window.innerHeight +
                                "px.</p>";
                        // Wait for window size to increase
                        while (window.innerWidth < mw || window.innerHeight < mh) {
                            yield delay(100);
                        }
                        this.getDisplayElement().innerHTML = "";
                    }
                }
                // WEB AUDIO API
                if (typeof exclusions.audio !== "undefined" && exclusions.audio) {
                    if (!window.hasOwnProperty("AudioContext") && !window.hasOwnProperty("webkitAudioContext")) {
                        this.getDisplayElement().innerHTML =
                            "<p>Your browser does not support the WebAudio API, which means that you will not " +
                                "be able to complete the experiment.</p><p>Browsers that support the WebAudio API include " +
                                "Chrome, Firefox, Safari, and Edge.</p>";
                        throw new Error();
                    }
                }
            });
        }
        drawProgressBar(msg) {
            document
                .querySelector(".jspsych-display-element")
                .insertAdjacentHTML("afterbegin", '<div id="jspsych-progressbar-container">' +
                "<span>" +
                msg +
                "</span>" +
                '<div id="jspsych-progressbar-outer">' +
                '<div id="jspsych-progressbar-inner"></div>' +
                "</div></div>");
        }
        updateProgressBar() {
            this.setProgressBar(this.getProgress().percent_complete / 100);
        }
        setProgressBar(proportion_complete) {
            proportion_complete = Math.max(Math.min(1, proportion_complete), 0);
            document.querySelector("#jspsych-progressbar-inner").style.width =
                proportion_complete * 100 + "%";
            this.progress_bar_amount = proportion_complete;
        }
        getProgressBarCompleted() {
            return this.progress_bar_amount;
        }
    }

    // temporary patch for Safari
    if (typeof window !== "undefined" &&
        window.hasOwnProperty("webkitAudioContext") &&
        !window.hasOwnProperty("AudioContext")) {
        // @ts-expect-error
        window.AudioContext = webkitAudioContext;
    }
    // end patch
    // The following function provides a uniform interface to initialize jsPsych, no matter whether a
    // browser supports ES6 classes or not (and whether the ES6 build or the Babel build is used).
    /**
     * Creates a new JsPsych instance using the provided options.
     *
     * @param options The options to pass to the JsPsych constructor
     * @returns A new JsPsych instance
     */
    function initJsPsych(options) {
        return new JsPsych(options);
    }

    exports.JsPsych = JsPsych;
    exports.initJsPsych = initJsPsych;
    exports.universalPluginParameters = universalPluginParameters;

    Object.defineProperty(exports, '__esModule', { value: true });

    return exports;

})({});
var initJsPsych = jsPsychModule.initJsPsych;
