import json
import numpy as np
import matplotlib.pyplot as plt

import bandits
import decomposition
import framework

N = 50
K = 5
C = 4
T = 10**5

MUS_MODEL = framework.generate_means_linear_model

def match_with_decomposition(decomp):
    return lambda instance: bandits.MatchAlgorithm(instance, decomp, K)

ALGS = [('etc', bandits.ExploreThenCommit),
        ('ucb_regret', bandits.IndependentUCB),
        ('lp_decomposition', match_with_decomposition(decomposition.LPDecomposition)),
        ('greedy_decomposition', match_with_decomposition(decomposition.GreedyDecomposition)),
        ('random_decomposition', match_with_decomposition(decomposition.RandomizedDecomposition))]

def generate_data(fname, verbose=True):
    mus = MUS_MODEL(N, K)

    data = dict()

    for key, alg_type in ALGS:
        instance = framework.AnonymousBandits(mus, C, T)
        alg = alg_type(instance)
        while instance.is_running():
            alg.do_round()
        data[key] = instance.regret_timeseries()

        if verbose:
            print("done {} {}".format(fname, key))

    json.dump(data, open('data/{}.json'.format(fname), 'w'))

def generate_many_runs(num_runs, prefix):
    for t in range(1, num_runs+1):
        generate_data('{}_{}'.format(prefix, t))

def plot_one_run(verbose=True):
    mus = MUS_MODEL(N, K)

    data = dict()

    for key, alg_type in ALGS:
        instance = framework.AnonymousBandits(mus, C, T)
        alg = alg_type(instance)
        while instance.is_running():
            alg.do_round()
        data[key] = instance.regret_timeseries()

        if verbose:
            print("done {}".format(key))

    for key, _ in ALGS:
        plt.plot(data[key], label=key)

    plt.title("N=50, K=5, C=4, T=10^5")
    plt.ylabel("Cumulative Regret")
    plt.xlabel("Round")
    plt.legend()
    plt.show()

if __name__ == '__main__':
    plot_one_run()
    # generate_many_runs('linear', 20)
