import torch
from torch import nn
import numpy as np
from .blocks import BasicBlock2D as BasicBlock
from deepthinking.lib.solvers import anderson, broyden

### Code borrowed from DeepThinking github repository
""" dt_net_2d.py
    DeepThinking network 2D.
    Developed for DeepThinking project
"""

class DTNet(nn.Module):
    """DeepThinking Network 2D model class"""

    def __init__(self, block, num_blocks, width, in_channels=3, recall=True, group_norm=False, **kwargs):
        super().__init__()

        self.recall = recall
        self.width = int(width)
        self.group_norm = group_norm
        proj_conv = nn.Conv2d(in_channels, width, kernel_size=3,
                              stride=1, padding=1, bias=False)

        conv_recall = nn.Conv2d(width + in_channels, width, kernel_size=3,
                                stride=1, padding=1, bias=False)

        recur_layers = []
        if recall:
            recur_layers.append(conv_recall)

        for i in range(len(num_blocks)):
            recur_layers.append(self._make_layer(block, width, num_blocks[i], stride=1))

        head_conv1 = nn.Conv2d(width, 32, kernel_size=3,
                               stride=1, padding=1, bias=False)
        head_conv2 = nn.Conv2d(32, 8, kernel_size=3,
                               stride=1, padding=1, bias=False)
        head_conv3 = nn.Conv2d(8, 2, kernel_size=3,
                               stride=1, padding=1, bias=False)

        self.projection = nn.Sequential(proj_conv, nn.ReLU())
        self.recur_block = nn.Sequential(*recur_layers)
        self.head = nn.Sequential(head_conv1, nn.ReLU(),
                                  head_conv2, nn.ReLU(),
                                  head_conv3)
        self.f_solver = eval('anderson')
        #self.f_solver = eval('broyden')
        self.total_n_step = 0
        self.count = 0

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for strd in strides:
            layers.append(block(self.width, planes, strd, group_norm=self.group_norm))
            self.width = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x, iters_to_do, interim_thought=None, plot=False, **kwargs):
        initial_thought = self.projection(x)

        if interim_thought is None:
            interim_thought = initial_thought

        all_outputs = torch.zeros((x.size(0), iters_to_do, 2, x.size(2), x.size(3))).to(x.device)
        layer_idx = np.arange(0, iters_to_do, 5)
        fp_norms = []
        prev_norms = []

        def forward_func(inp):
            if self.recall:
                temp_out = torch.cat([inp, x], 1)
            return self.recur_block(temp_out)

        prev = interim_thought.clone()
        rel_trace = []
        abs_trace = []
        for i in range(iters_to_do):
            interim_thought = forward_func(interim_thought)

            abs_diff = (interim_thought - prev).norm().item()
            if i > 1:
                abs_trace.append(abs_diff)
                rel_trace.append(abs_diff / (1e-5 + interim_thought.norm().item()))
                fp_norms.append(interim_thought.norm().item())
                prev_norms.append(prev.norm().item())

            out = self.head(interim_thought)
            all_outputs[:, i] = out
            prev = interim_thought.clone()
                                
        if self.training:
            return out, interim_thought
        
        if plot:
            return out, all_outputs, abs_trace, rel_trace

        return all_outputs

def dt_net_2d(width, **kwargs):
    return DTNet(BasicBlock, [2], width=width, in_channels=kwargs["in_channels"], recall=False)


def dt_net_recall_2d(width, **kwargs):
    return DTNet(BasicBlock, [2], width=width, in_channels=kwargs["in_channels"], recall=True)


def dt_net_gn_2d(width, **kwargs):
    return DTNet(BasicBlock, [2], width=width, in_channels=kwargs["in_channels"], recall=False, group_norm=True)


def dt_net_recall_gn_2d(width, **kwargs):
    return DTNet(BasicBlock, [2], width=width, in_channels=kwargs["in_channels"], recall=True, group_norm=True)
