%% Escape from saddle points on quantum computers
% This script runs the 4 numerical experiments presented in the paper
% "Escape from Saddle Points on Quantum Computers"
% submitted to NeurIPS 2020 with paper ID 2851. Do not distribute.
% The computing environment is MATLAB 2019a. 

% In test 0 and 1, we use the finite difference method (FDM) and the 
% leapfrog scheme (LFS) to solve time-dependent Schrodinger equation. 
% The integrator 'leapfrogsolver' (available on MathWorks) was developed 
% by Mauger Fran?ois. 


% In test 2, we use a sampler ('randsphe?re') to generate random points
% in an n-dim hypersphere. 
% It was developed by Roger Stafford, also availale in MathWorks.

% The initialization PDE parameters are the same as follows.

%% Test 2: PGD & NCGD on non-quadratic function
n = 512;
n1 = n+1;
a = 6/n1;
xaxis = -3+a:a:3-a;
yaxis = -3+a:a:3-a;
[X,Y]=meshgrid(xaxis,yaxis);
func1 = @(x,y) -x.^2./2 + 9*y.^2./8 + x.^4./16;
Z1 = func1(X,Y);
%%
eta = .05; % descent step size
r = 0.1; % radius of perturbation ball
M = 300; % # of samples

outcome_pgd = zeros(1,M);
outcome_ncgd = zeros(1,M);
outcome_pgdx = zeros(1,M);
outcome_ncgdx = zeros(1,M);
outcome_pgdy = zeros(1,M);
outcome_ncgdy = zeros(1,M);
outcome_pgdh = zeros(1,M);
outcome_ncgdh = zeros(1,M);
outcome_pgdxh = zeros(1,M);
outcome_ncgdxh = zeros(1,M);
outcome_pgdyh = zeros(1,M);
outcome_ncgdyh = zeros(1,M);
T1 = 90;
T2 = 30;
    
% sample
    seed_pgd = randsphere(M,2,r);
    seed_ncgd = randsphere(M,2,r);

% PGD & qPGD
tic
    for t = 1:M
        % pgd for type 0
        x0 = seed_pgd(t,:)';
        xT = gd(x0,eta,T1);
        outcome_pgdx(t) = xT(1);
        outcome_pgdy(t) = xT(2);
        outcome_pgd(t) = f(xT);
        xT = gd(x0,eta,T1/2);
        outcome_pgdxh(t) = xT(1);
        outcome_pgdyh(t) = xT(2);
        outcome_pgdh(t) = f(xT);
        
        %ncgd for type 0
        x0 = seed_ncgd(t,:)';
        xT = nc(x0,eta,T2,r);
        outcome_ncgdx(t) = xT(1);
        outcome_ncgdy(t) = xT(2);
        outcome_ncgd(t) = f(xT);
        xT = nc(x0,eta,T2/2,r);
        outcome_ncgdxh(t) = xT(1);
        outcome_ncgdyh(t) = xT(2);
        outcome_ncgdh(t) = f(xT);
    end
tEnd=toc;
fprintf('Test quartic_decrease, running time = %d\n',tEnd);

%% plot 
figure(1)
subplot(1,3,1)
scatter(outcome_pgdxh,outcome_pgdyh,50,'.','r');
hold on
scatter(outcome_ncgdxh,outcome_ncgdyh,50,'.','b');
hold on
contour(X,Y,Z1, [-0.5,0,1,2,3],'ShowText','on');
plot([2 -2],[0 0],'k*');
set(gca,...
    'TickDir','out','TickLength',[0.02 0.02],...
    'FontSize',12,'FontName','Times');
set(gcf, 'units','points','position',[0 0 1000 400]);
xticks([-2,-1,0,1,2]);
yticks([-2,-1,0,1,2]);
xlabel('X')
ylabel('Y')
legend('PGD-sampling','NCGD-sampling','landscape','minimizer')
title('t_{PGD} = 45,   t_{NCGD} = 15');

subplot(1,3,2)
scatter(outcome_pgdx,outcome_pgdy,70,'.','r');
hold on
scatter(outcome_ncgdx,outcome_ncgdy,70,'.','b');
hold on
contour(X,Y,Z1, [-0.5,0,1,2,3],'ShowText','on');
plot([2 -2],[0 0],'k*');
set(gca,...
    'TickDir','out','TickLength',[0.02 0.02],...
    'FontSize',12,'FontName','Times');
set(gcf, 'units','points','position',[0 0 1000 400]);
xticks([-2,-1,0,1,2]);
yticks([-2,-1,0,1,2]);
xlabel('X')
ylabel('Y')
legend('PGD-sampling','NCGD-sampling','landscape','minimizer')
title('t_{PGD} = 90,   t_{NCGD} = 30');

subplot(1,3,3)
binRange = -1.2:0.3:0;
hcx = histcounts(outcome_ncgd(:),[binRange Inf]);
hcy = histcounts(outcome_pgd(:),[binRange Inf]);
b = bar(binRange,[hcx;hcy]);
b(1).FaceColor = [0 .3 .9];
b(2).FaceColor = [.9 .2 .1];
xtips1 = b(1).XEndPoints;
ytips1 = b(1).YEndPoints;
labels1 = string(b(1).YData);
text(xtips1,ytips1,labels1,'HorizontalAlignment','center',...
    'VerticalAlignment','bottom')
xtips2 = b(2).XEndPoints;
ytips2 = b(2).YEndPoints;
labels2 = string(b(2).YData);
text(xtips2,ytips2,labels2,'HorizontalAlignment','center',...
    'VerticalAlignment','bottom')
set(gca,...
    'XDir','reverse',...
        'FontSize',12,'FontName','Times');
xticks([-1.2 -0.9 -0.6 -0.3 0]);
xticklabels({'(-1,-0.9]','(-0.9,-0.6]','(-0.6,0.3]','(-0.3,0]','>0'});
xlabel('Descent value')
ylabel('Frequency')
legend('NCGD','PGD','Location','northwest');
title('descent value sampling at t_{PGD} = 90, t_{NCGD} = 30')



%%
function x = gd(x0,eta,T)
for k = 1:T
    x = x0 - eta.*grad(x0);
    x0 = x;
end
end

function x = nc(x0,eta,T,r)
for k = 1:T
    x0 = x0 * r / norm(x0);
    x = x0 - eta.*grad(x0);
    x0 = x;
end
x = 1.8 * x / norm(x);
end

function y = f(x)
y = -x(1)^2/2 + 9*x(2)^2/8 + x(1)^4/16;
end

function v = grad(x)
v = [-x(1)+x(1)^3/4;9*x(2)/4];
end

