from __future__ import print_function
from PIL import Image
import os
import os.path
import errno
import numpy as np
import sys
import copy
if sys.version_info[0] == 2:
    import cPickle as pickle
else:
    import pickle

import torch
import torch.utils.data as data
from torchvision import datasets, transforms


class CIFAR100(data.Dataset):
    base_folder = 'cifar-100-python'
    train_list = [
        ['train', '16019d7e3df5f24257cddd939b257f8d'],
    ]

    test_list = [
        ['test', 'f0ef6b0ae62326f3e7ffdfab6717acfc'],
    ]

    def __init__(self, root, train=True,
                 transform=None, target_transform=None,
                 confusion_T=0, zero_out_ratio=0,
                 confusion=False, actual=False):
        self.root = os.path.expanduser(root)
        self.transform = transform
        self.target_transform = target_transform
        self.train = train

        if self.train:
            if confusion:
                # Use confusion training data
                confusion_path = '{}/zero_out_{}_{}.npz'.format(root, confusion_T, zero_out_ratio)
                confusion_data = np.load(confusion_path)
                self.train_data = confusion_data['training_data']
                self.train_labels = confusion_data['training_label']
                print('Load confusion training data from {}'.format(confusion_path))
            elif actual:
                # Use actual training data
                self.train_data = []
                self.train_labels = []
                for fentry in self.train_list:
                    f = fentry[0]
                    file = os.path.join(self.root, self.base_folder, f)
                    fo = open(file, 'rb')
                    if sys.version_info[0] == 2:
                        entry = pickle.load(fo)
                    else:
                        entry = pickle.load(fo, encoding='latin1')
                    self.train_data.append(entry['data'])
                    if 'labels' in entry:
                        self.train_labels += entry['labels']
                    else:
                        self.train_labels += entry['fine_labels']
                    fo.close()
                print('Load actual training data from {}'.format(root))

                self.train_data = np.concatenate(self.train_data)
                self.train_data = self.train_data.reshape((50000, 3, 32, 32))
                self.train_data = self.train_data.transpose((0, 2, 3, 1))
        else:
            f = self.test_list[0][0]
            file = os.path.join(self.root, self.base_folder, f)
            fo = open(file, 'rb')
            if sys.version_info[0] == 2:
                entry = pickle.load(fo)
            else:
                entry = pickle.load(fo, encoding='latin1')
            self.test_data = entry['data']
            if 'labels' in entry:
                self.test_labels = entry['labels']
            else:
                self.test_labels = entry['fine_labels']
            fo.close()
            self.test_data = self.test_data.reshape((10000, 3, 32, 32))
            self.test_data = self.test_data.transpose((0, 2, 3, 1))  # convert to HWC

    def __getitem__(self, index):
        if self.train:
            img, target = self.train_data[index], self.train_labels[index]
        else:
            img, target = self.test_data[index], self.test_labels[index]

        img = Image.fromarray(img)

        if self.transform is not None:
            img = self.transform(img)

        if self.target_transform is not None:
            target = self.target_transform(target)

        return img, target

    def __len__(self):
        if self.train:
            return len(self.train_data)
        else:
            return len(self.test_data)


def get_dataloader(args):
    kwargs = {'num_workers': 4, 'pin_memory': True} if args.cuda else {}

    if args.DA_for_train:
        transforms_ = transforms.Compose([
            transforms.RandomCrop(32, padding=4),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
        ])
    else:
        transforms_ = transforms.Compose([
            transforms.ToTensor(),
            transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
        ])

    if args.mode.startswith('random_init'):
        print('Apply random initialization.')
        train_dataset = CIFAR100('./cifar100_data/data', train=True, transform=transforms_, actual=True)
    elif args.mode.startswith('adversarial_init'):
        print('Apply adversarial initialization.')
        if args.Is_Init:
            print('In Adversarial Initialization / Pre Training.')
            train_dataset = CIFAR100('./cifar100_data/confusion_random_train_label', train=True, transform=transforms_,
                                     confusion_T=args.confusion_T, zero_out_ratio=args.zero_out_ratio, confusion=True)
        else:
            print('In Main Training / Fine Tuning.')
            train_dataset = CIFAR100('./cifar100_data/data', train=True, transform=transforms_, actual=True)

    test_dataset = CIFAR100('./cifar100_data/data', train=False,
                            transform=transforms.Compose([
                                transforms.ToTensor(),
                                transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
                            ]))
    train_loader = torch.utils.data.DataLoader(train_dataset, batch_size=args.batch_size, shuffle=True, **kwargs)
    test_loader = torch.utils.data.DataLoader(test_dataset, batch_size=args.batch_size, shuffle=False, **kwargs)

    return train_loader, test_loader