library(dplyr)
library(data.table)
library(Matrix)
library(parallel)

# the following line is for getting the path of your current open file
current_path <- getActiveDocumentContext()$path 
# The next line set the working directory to the relevant one:
setwd(dirname(current_path))
# you can make sure you are in the right directory
print(getwd())
source('QPSolver.R')

getResults <- function(r){
  result <- NULL
  for (i in 1:5){
    set.seed(1234 * i + r)
    n <- i * 1e4
    p <- 100
    fullData <- data.table(producerId = sample.int(p, size = n, replace = TRUE),
                           consumerId = sample.int(p, size = n, replace = TRUE),
                           pijBase = runif(n))
    
    fullData$lowerBound <- fullData$pijBase / 10
    fullData$upperBound <- fullData$pijBase * 10
    fullData <- fullData[, lj := sum(pijBase) / 5, by = consumerId]
    fullData <- fullData[, uj := sum(pijBase) * 5, by = consumerId]
    fullData <- fullData[, h := n / p * runif(1), by = producerId]
    fullData <- fullData[, htilde := first(h) - sum(pijBase), by = producerId]
    
    fullData$producerId <- factor(fullData$producerId, levels = unique(fullData$producerId))
    fullData$consumerId <- factor(fullData$consumerId, levels = unique(fullData$consumerId))
    
    groupIterativeValues <- groupIterativeQPSolver(fullData, 10, 2)
    fullValues <- groupIterativeQPSolver(fullData, p, 1)
    
    result <- rbind(result, 
                    tibble(n = n,
                           type = c("iterative", "full"), 
                           runtime = c(groupIterativeValues$runtime, fullValues$runtime),
                           objVal = c(groupIterativeValues$objVal, fullValues$objVal))
    )
  }
  return(result)
}

nCores <- detectCores() - 1

allResults <- mclapply(1:100, getResults, mc.cores = nCores)
allResults <- do.call("rbind", allResults)
save(allResults, file = "QPSolverResults.Rdata")

load("QPSolverResults.Rdata")

summary(allResults)

allResults <- allResults %>% mutate(method = factor(type), size = factor(n))

library(ggplot2)
library(gridExtra)

pdf("interative_QP.pdf", height = 6, width = 16)
p1 <- allResults %>% ggplot(aes(x = size, y = runtime, fill = method)) + geom_boxplot() + 
  theme_bw() + theme(text = element_text(size=20))  + ylab("runtime (in seconds)") + xlab("number of variables")
p2 <- allResults %>% ggplot(aes(x = size, y = objVal, fill = method)) + geom_boxplot() + 
  theme_bw() + theme(text = element_text(size=20)) + ylab("final objective value") + xlab("number of variables")
grid.arrange(p1, p2, ncol=2)
dev.off()

