from __future__ import absolute_import
from __future__ import print_function
from __future__ import division

from pytorch_pretrained_bert.tokenization import BertTokenizer
from pytorch_pretrained_bert.modeling import BertForSequenceClassification
from pytorch_pretrained_bert.optimization import BertAdam

from tqdm import tqdm
import copy
import random
import numpy as np

import torch
from torch import nn
from torch import optim

from torch.utils.data import DataLoader
from torch.utils.data import RandomSampler, SequentialSampler
from torch.utils.data import TensorDataset

from data_utils import InputFeatures

from magic_module import MagicModule


BERT_MODEL = 'bert-base-uncased'


class SequenceClassifier:
    def __init__(self, label_list, device):
        self._label_list = label_list
        self._device = device

        self._tokenizer = BertTokenizer.from_pretrained(
            BERT_MODEL, do_lower_case=True)

        self._model = BertForSequenceClassification.from_pretrained(
            BERT_MODEL, num_labels=len(label_list)).to(device)

        self._optimizer = None
        self._opt_hparams = {}

        self._dataset = {}
        self._data_loader = {}

        self._w_optimizer = None

        self._num_dev_batches = None


    def load_data(self, set_type, examples, batch_size, shuffle, has_w=False, w_lr=0.01):
        self._dataset[set_type] = examples
        ret = _make_data_loader(
            examples=examples,
            label_list=self._label_list,
            tokenizer=self._tokenizer,
            batch_size=batch_size,
            shuffle=shuffle,
            has_w=has_w,
            w_lr=w_lr)

        if isinstance(ret, (list, tuple)):
            self._data_loader[set_type], self._w_optimizer = ret
            self.w_lr = w_lr
        else:
            self._data_loader[set_type] = ret

        if set_type == 'dev':
            if len(examples) % batch_size == 0:
                self._num_dev_batches = len(examples) // batch_size
            else:
                self._num_dev_batches = len(examples) // batch_size + 1
            print('num_dev_batches: %d' % self._num_dev_batches)


    def get_optimizer(self, learning_rate, warmup_proportion, t_total):
        self._opt_hparams['learning_rate'] = learning_rate
        self._opt_hparams['warmup_proportion'] = warmup_proportion
        self._opt_hparams['t_total'] = t_total

        self._optimizer = _get_optimizer(self._model, **self._opt_hparams)

    def train_epoch(self, plain, grad_step):
        self._model.train()

        buffer = {'loss': [], 'weights': []}

        def _clear_buffer():
            buffer['loss'] = []
            buffer['weights'] = []
            self._optimizer.zero_grad()
            self.load_data(
                #'magic', random.sample(self._dataset['dev'], 50),
                'magic', self._dataset['dev'],
                self._data_loader['dev'].batch_size, shuffle=True)

        _clear_buffer()
        for step, batch in enumerate(tqdm(self._data_loader['train'])):
            batch = tuple(t.to(self._device) for t in batch)
            batch_size = batch[-1].shape[0]
            # collect weights
            if plain:
                buffer['weights'].append(
                    torch.ones(batch_size).to(self._device))
            else:
                buffer['weights'].append(self._get_weights(batch))

            # collect loss
            buffer['loss'].append(self._get_batch_loss(batch))

            if (step + 1) % grad_step == 0:
                loss = torch.cat(buffer['loss'], dim=0)
                weights = _normorlize(torch.cat(buffer['weights'], dim=0))
                loss = torch.sum(loss * weights.data)

                loss.backward()
                self._optimizer.step()

                _clear_buffer()
                # _print_weights(weights)

            #input_ids, input_mask, segment_ids, label_ids = batch
            #loss = self._model(
            #    input_ids, segment_ids, input_mask, label_ids)
            #loss.backward()
            #self._optimizer.step()

    def _get_weights(self, batch):
        input_ids, input_mask, segment_ids, label_ids, weights = batch
        batch_size = label_ids.shape[0]
        #weights = torch.zeros(batch_size, requires_grad=True).to(self._device)
        #weights = torch.ones(batch_size, requires_grad=True).to(self._device) / batch_size #TODO

        weights = _normorlize(weights)

        magic_model = MagicModule(self._model)

        model_tmp = copy.deepcopy(self._model)
        optimizer_tmp = _get_optimizer(model_tmp, **self._opt_hparams)
        criterion_tmp = nn.CrossEntropyLoss().to(self._device)

        for i in range(batch_size):
            model_tmp.load_state_dict(self._model.state_dict())
            optimizer_tmp.load_state_dict(self._optimizer.state_dict())

            model_tmp.zero_grad()
            logits = model_tmp(
                input_ids[i:i+1], segment_ids[i:i+1], input_mask[i:i+1])
            loss = criterion_tmp(logits, label_ids[i:i+1])
            loss.backward()
            optimizer_tmp.step()

            deltas = {}
            for (name, param), (name_tmp, param_tmp) in zip(
                    self._model.named_parameters(),
                    model_tmp.named_parameters()):
                assert name == name_tmp
                deltas[name] = weights[i] * (param_tmp.data - param.data)
            magic_model.update_params(deltas)

        self._w_optimizer.zero_grad()

        weight_grads = []
        for step, val_batch in enumerate(self._data_loader['magic']):
            val_batch = (t.to(self._device) for t in val_batch)
            val_input_ids, val_input_mask, val_segment_ids, val_label_ids = \
                val_batch
            val_logits = magic_model(
                val_input_ids, val_segment_ids, val_input_mask)
            val_loss = criterion_tmp(val_logits, val_label_ids)

            #val_loss = val_loss / self._num_dev_batches
            #val_loss.backward(retain_graph=True)

            weight_grad = torch.autograd.grad(
                val_loss, weights, retain_graph=True)[0]
            weight_grads.append(-weight_grad)

        #self._w_optimizer.step()
        #weights.data = _normorlize(weights).data

        ##self._w_optimizer.step()
        #print(weights.data)
        #print(sum(weight_grads).data)
        wgd = sum(weight_grads).data
        wd = weights.data
        scale = np.linalg.norm(wd.cpu()) / np.linalg.norm(wgd.cpu())
        #print('scale: {}'.format(scale))
        weights.data = weights.data / self.w_lr + wgd * scale #TODO
        #print(weights.data)
        #print(label_ids.data)
        #exit()

        #weights.data = weights.data + sum(weight_grads).data #TODO
        #weights.data = _normorlize(weights).data #TODO: bug here

        #if random.random() > 0.98:
        #    wd = _normorlize(weights.data)
        #    w = ''
        #    for i, l in enumerate(label_ids.data):
        #        if l == 0:
        #            w += ', {}'.format(wd[i])
        #    if len(w) > 0:
        #        print('weight: {}'.format(w))

        return weights.data


    def _get_batch_loss(self, batch):
        if len(batch) == 4:
            input_ids, input_mask, segment_ids, label_ids = batch
        else:
            input_ids, input_mask, segment_ids, label_ids, _ = batch

        logits = self._model(input_ids, segment_ids, input_mask)
        criterion = nn.CrossEntropyLoss(reduction='none').to(self._device)

        return criterion(logits, label_ids)

    def evaluate(self, set_type):
        self._model.eval()

        preds_all = []
        labels_all = []
        data_loader = self._data_loader[set_type]

        for batch in tqdm(data_loader, desc="Evaluating"):
            batch = tuple(t.to(self._device) for t in batch)
            input_ids, input_mask, segment_ids, label_ids = batch

            with torch.no_grad():
                logits = self._model(input_ids, segment_ids, input_mask)

            preds = torch.argmax(logits, dim=1)
            preds_all.append(preds)
            labels_all.append(label_ids)

        preds_all = torch.cat(preds_all, dim=0)
        labels_all = torch.cat(labels_all, dim=0)

        return torch.sum(preds_all == labels_all).item() / labels_all.shape[0]


def _get_optimizer(model, learning_rate, warmup_proportion, t_total):
    param_optimizer = list(model.named_parameters())
    no_decay = ['bias', 'LayerNorm.bias', 'LayerNorm.weight']
    optimizer_grouped_parameters = [
        {'params': [p for n, p in param_optimizer if
                    not any(nd in n for nd in no_decay)], 'weight_decay': 0.01},
        {'params': [p for n, p in param_optimizer if
                    any(nd in n for nd in no_decay)], 'weight_decay': 0.0}]

    return optim.Adam(optimizer_grouped_parameters, lr=learning_rate)
    # return BertAdam(
    #     optimizer_grouped_parameters,
    #     lr=learning_rate,
    #     warmup=warmup_proportion,
    #     t_total=t_total)


def _make_data_loader(examples, label_list, tokenizer, batch_size, shuffle, has_w=False, w_lr=0.01):
    all_features = _convert_examples_to_features(
        examples=examples,
        label_list=label_list,
        max_seq_length=64,
        tokenizer=tokenizer,
        output_mode='classification')

    all_input_ids = torch.tensor(
        [f.input_ids for f in all_features], dtype=torch.long)
    all_input_mask = torch.tensor(
        [f.input_mask for f in all_features], dtype=torch.long)
    all_segment_ids = torch.tensor(
        [f.segment_ids for f in all_features], dtype=torch.long)
    all_label_ids = torch.tensor(
        [f.label_id for f in all_features], dtype=torch.long)

    if has_w:
        all_weights = torch.ones(len(all_features), requires_grad=True) #TODO
        dataset = TensorDataset(
            all_input_ids, all_input_mask, all_segment_ids, all_label_ids, all_weights)
        w_optimizer = optim.Adam([all_weights], lr=w_lr)
        return DataLoader(dataset, batch_size=batch_size, shuffle=shuffle), w_optimizer
    else:
        dataset = TensorDataset(
            all_input_ids, all_input_mask, all_segment_ids, all_label_ids)
        return DataLoader(dataset, batch_size=batch_size, shuffle=shuffle)


def _normorlize(weights):
    #if torch.sum(weights) > 1e-8:
    #    return weights / torch.sum(weights)
    #return torch.zeros_like(weights)
    return nn.functional.softmax(weights, dim=0)


def _print_weights(weights):
    print('weights:', end=' ')
    for weight in weights:
        print('{:.4f}'.format(weight), end=' ')
    print()


def _convert_examples_to_features(examples, label_list, max_seq_length,
                                 tokenizer, output_mode):
    """Loads a data file into a list of `InputBatch`s."""

    label_map = {label : i for i, label in enumerate(label_list)}

    features = []
    for (ex_index, example) in enumerate(examples):
        # if ex_index % 10000 == 0:
        #     logger.info("Writing example %d of %d" % (ex_index, len(examples)))

        tokens_a = tokenizer.tokenize(example.text_a)

        tokens_b = None
        if example.text_b:
            tokens_b = tokenizer.tokenize(example.text_b)
            # Modifies `tokens_a` and `tokens_b` in place so that the total
            # length is less than the specified length.
            # Account for [CLS], [SEP], [SEP] with "- 3"
            _truncate_seq_pair(tokens_a, tokens_b, max_seq_length - 3)
        else:
            # Account for [CLS] and [SEP] with "- 2"
            if len(tokens_a) > max_seq_length - 2:
                tokens_a = tokens_a[:(max_seq_length - 2)]

        # The convention in BERT is:
        # (a) For sequence pairs:
        #  tokens:   [CLS] is this jack ##son ##ville ? [SEP] no it is not . [SEP]
        #  type_ids: 0   0  0    0    0     0       0 0    1  1  1  1   1 1
        # (b) For single sequences:
        #  tokens:   [CLS] the dog is hairy . [SEP]
        #  type_ids: 0   0   0   0  0     0 0
        #
        # Where "type_ids" are used to indicate whether this is the first
        # sequence or the second sequence. The embedding vectors for `type=0` and
        # `type=1` were learned during pre-training and are added to the wordpiece
        # embedding vector (and position vector). This is not *strictly* necessary
        # since the [SEP] token unambiguously separates the sequences, but it makes
        # it easier for the model to learn the concept of sequences.
        #
        # For classification tasks, the first vector (corresponding to [CLS]) is
        # used as as the "sentence vector". Note that this only makes sense because
        # the entire model is fine-tuned.
        tokens = ["[CLS]"] + tokens_a + ["[SEP]"]
        segment_ids = [0] * len(tokens)

        if tokens_b:
            tokens += tokens_b + ["[SEP]"]
            segment_ids += [1] * (len(tokens_b) + 1)

        input_ids = tokenizer.convert_tokens_to_ids(tokens)

        # The mask has 1 for real tokens and 0 for padding tokens. Only real
        # tokens are attended to.
        input_mask = [1] * len(input_ids)

        # Zero-pad up to the sequence length.
        padding = [0] * (max_seq_length - len(input_ids))
        input_ids += padding
        input_mask += padding
        segment_ids += padding

        assert len(input_ids) == max_seq_length
        assert len(input_mask) == max_seq_length
        assert len(segment_ids) == max_seq_length

        if output_mode == "classification":
            label_id = label_map[example.label]
        elif output_mode == "regression":
            label_id = float(example.label)
        else:
            raise KeyError(output_mode)

        # if ex_index < 5:
        #     logger.info("*** Example ***")
        #     logger.info("guid: %s" % (example.guid))
        #     logger.info("tokens: %s" % " ".join(
        #             [str(x) for x in tokens]))
        #     logger.info("input_ids: %s" % " ".join([str(x) for x in input_ids]))
        #     logger.info("input_mask: %s" % " ".join([str(x) for x in input_mask]))
        #     logger.info(
        #             "segment_ids: %s" % " ".join([str(x) for x in segment_ids]))
        #     logger.info("label: %s (id = %d)" % (example.label, label_id))

        features.append(
                InputFeatures(input_ids=input_ids,
                              input_mask=input_mask,
                              segment_ids=segment_ids,
                              label_id=label_id))

    return features


def _truncate_seq_pair(tokens_a, tokens_b, max_length):
    """Truncates a sequence pair in place to the maximum length."""

    # This is a simple heuristic which will always truncate the longer sequence
    # one token at a time. This makes more sense than truncating an equal percent
    # of tokens from each, since if one sequence is very short then each token
    # that's truncated likely contains more information than a longer sequence.
    while True:
        total_length = len(tokens_a) + len(tokens_b)
        if total_length <= max_length:
            break
        if len(tokens_a) > len(tokens_b):
            tokens_a.pop()
        else:
            tokens_b.pop()
