#include "alloc.h"

#include <vector>
#include <algorithm>
#include <random>
#include <cfloat>

#include <eigen3/Eigen/Core>
#include <eigen3/Eigen/LU>
#include <eigen3/Eigen/Eigenvalues>


using namespace std;
using namespace Eigen;




/***************************************************************
* Compute Optimal Solution for known nu
***************************************************************/
float Alloc::optimal(MatrixXf nu, MatrixXf &optimal) {
  int K = nu.rows();
  int D = nu.cols();

  vector<float> used(D);
  vector<float> score(K);
  vector<int> idx(D);

  float total_score = 0.0;
  
  for (int d=0;d!=D;++d) {
    used[d] = 0.0;
    idx[d] = K - 1;
  }

  for (int k=0;k!=K;++k) {
    score[k] = 0.0;
  }

  optimal = MatrixXf::Zero(K, D);

  MatrixXi sorted = sort_index(nu);

  int best_idx = 0;
  /* loop until no resource can be allocated */
  while (best_idx != -1) {
    best_idx = -1;
    float best = -1;
    for (int d=0;d!=D;++d) {
      /* check there are still unallocated tasks for this resource type, and still resources left */
      if (idx[d] != -1 && used[d] < 0.999) {
        /* if is the last task and there are resources left, then just allocate them */
        if (idx[d] == 0) {
          best_idx = d;
          break;
        }
        /* compute the ratio */
        float ratio;
        if (nu(sorted(idx[d],d),d) > 0 && nu(sorted(idx[d]-1,d),d) < 0) {
          ratio = FLT_MAX;
        }else {
          ratio = nu(sorted(idx[d],d),d) / nu(sorted(idx[d]-1,d),d);
        }
        /* is it the largest ratio so far? */
        if (ratio > best) {
          best = ratio;
          best_idx = d;
        }
      }
    }
    /* if some resource could be allocated */
    if (best_idx != -1) {
      /* get the index of the resource/task to set */
      int d = best_idx;
      int k = sorted(idx[d],d);
      /* do not allocate to nu(k,d) < 0 */
      if (nu(k,d) < 0) {
        used[d] = 1.0;
      }else {
        /* compute the amount of resources to assign. Either use all the resources, or complete the task */
        if (1.0 - used[d] > (1.0 - score[k]) / nu(k,d)) {
          optimal(k,d) = (1.0 - score[k]) / nu(k,d);
        }else {
          optimal(k,d) = 1.0 - used[d];
        }
        /* update the current score for the task */
        score[k] += nu(k,d) * optimal(k,d);
        total_score += nu(k,d) * optimal(k,d);
        /* update the current amount of the resource type used */
        used[d] += optimal(k,d);
        /* decrement the index into the sorted matrix to compute ratios */
        idx[d]--;
      }
    }
  }

  return total_score;
}



/***************************************************************
* Accept vector and return sorted index vector
***************************************************************/
VectorXi Alloc::vector_sort_index(VectorXf v) {
  /* a lot of ugly copying happening here ... */
  vector<pair<float,int> > x(v.size());
  for (int i=0;i!=v.size();++i) {
    x[i] = pair<float,int>(v[i],i);
  }
  sort(x.begin(), x.end());
  VectorXi sorted = VectorXi::Zero(v.size());
  for (int i=0;i!=v.size();++i) {
    sorted[i] = x[i].second;
  }
  return sorted;
}


/***************************************************************
* Accept matrix and return matrix of indexes pointing
to entries sorted on the columns
***************************************************************/
MatrixXi Alloc::sort_index(MatrixXf nu) {
  int K = nu.rows();
  int D = nu.cols();

  MatrixXi sorted = MatrixXi::Zero(K, D);
  
  for (int d=0;d!=D;++d) {
    sorted.col(d) = vector_sort_index(nu.col(d));
  }
  return sorted;
}

