library(rje)

patternRepeat = function (x, which, n, careful = TRUE)
{
  if (careful) {
    tmp = unique.default(which)
    if (!all(tmp == which)) {
      warning("repeated indices ignored")
      which = tmp
    }
    if (length(which) == 0) {
      if(length(x) != 1) stop("x not of correct length")
      else return(rep.int(x, prod(n)))
    }
    if (max(which) > length(n))
      stop("n not long enough")
    if (prod(n[which]) != length(x))
      stop("x not of correct length")
  }
  else if (length(which) == 0) return(rep.int(x, prod(n)))

  tmp = seq_along(n)[-which]
  if (length(tmp) == 0)
    return(x)

  for (add.in in tmp) {
    bl = prod(n[seq_len(add.in - 1)])
    x = x[rep.int(seq_len(bl), n[add.in]) + rep(seq.int(from = 0,
             by = bl, length.out = length(x)/bl), each = bl * n[add.in])]
  }
  return(x)
}

## Generate distributions of the form P(Y=1 | PA=x) using logistic
## link function.
## -------------------------
## np           : integer giving number of parents
## interactions : logical specifying whether to include two-way interactions
## -------------------------
## If interactions = FALSE
## logit P(Y=1 | PA1=x1, PA2=x2, ... ,PAk=xk) = sum_i (-1)^{xi} \alpha_i/2
## where \alpha_i is an independent normal r.v., and k is the number of
## parents.
## If interactions = TRUE, then add on sum_{i<j} (-1)^{xi*xj} \beta_{ij}/2
## for \beta_ij with same distribution.
##
logitDist = function(np, nu=6, interactions=FALSE) {
  if (np == 0) return(runif(1))
  effs = rnorm(np, sd=nu/np)
  out = c(-1,1)*effs[1]/2
  for (i in seq_len(np)[-1]) out = outer(out, c(-1,1)*effs[i]/2, "+")
  
  if (interactions && np > 1) {
    ints = matrix(rnorm(np^2, sd=2*nu/np), np, np)
    for (j in seq_len(np)[-1]) for (i in seq_len(j-1)) {
      out = out + patternRepeat(c(1,-1,-1,1)*ints[i,j]/2, c(i,j), rep(2,np), careful=FALSE)
    }
  }
  return(expit(c(out)))
}
