function [W validJ] = convertToCLogistic(J)
% [W validJ] = convertToCLogistic(J)
% Takes the parameters of an Ising model and coverts to those of a cascaded
% logistic model. See appendix of the reference for the derivation.
%
% Input
%    J: (m x m) Ising model parameters
%
% Output
%    W: (m x m) lower triagular matrix containing cascaded logistic parameters
%        W(i,1) = h_{i} (offset terms)
%        W(i,j) = w_{i,j-1}, 1<j<=i (interaction term between neurons j-1 and i)
%    validJ: (boolean) true if conversion exists 
%
% Reference:
% - Il Memming Park, Evan Archer, Kenneth Latimer, Jonathan Pillow. Universal models for binary spike patterns using centered Dirichlet processes. Neural Information Processing Systems (NIPS) 2013

m = size(J,1);

logex = @(x)log(1+exp(x));
W = zeros(m,m);

closeEnoughToZero = 1e-15;

% go to the end of J, pull out stuff then subtract bits from previous columns
% move back a column and repeat
for ii = m:-1:2
    W(ii,1) = J(ii,ii);
    W(ii,2:ii) = J(1:ii-1,ii)';
    
    idx = find(abs(W(ii,2:ii)) > closeEnoughToZero)+1;
    
    for jj = idx
	J(jj-1,jj-1) = J(jj-1,jj-1) ...
	    - (logex(W(ii,1)) - logex(W(ii,1) + W(ii,jj)));
    end

    if(numel(idx) == 2)
	J(idx(1)-1, idx(2)-1) = J(idx(1)-1, idx(2)-1) ...
	    - (logex(W(ii,1) + W(ii,idx(1))) ...
	    + logex(W(ii,1) + W(ii,idx(2))) - logex(W(ii,1)) ...
	    - logex(W(ii,1) + W(ii,idx(1)) + W(ii,idx(2))));
    elseif(numel(idx) > 2)
	warning('Cannot find an equivalent cascade logistic model.');
	validJ = false;
	return
    end
end
W(1,1) = J(1,1);
validJ = true;
