/*---------------------------------------------------
 * File:    sampler.h
 * purpose: header file for gibbs sampling routines
 * author:  ahollowa@uci.edu
 * date:    12/11/2009
 *-------------------------------------------------*/

#ifndef _SAMPLER_H_
#define _SAMPLER_H_

#define irand()  randomMT()
#define drand() (randomMT()/4294967296.0)

#include "graph.h"

typedef unsigned long uint32;

void seedMT(uint32 seed);

inline uint32 randomMT(void);

/*---------------------------------------------------
 *  Auxiliary Functions
 *--------------------------------------------------- */

/* Update the stick-breaking distribution */
void update_stick_break_distrib(struct node *n, int node_to_add);

/* Compute probability of transitioning from node x to child */
double compute_prob_from_existing_node(Graph *graph, int x, int child, int did, double alpha, double beta);

/* Compute the probability of the document at a given path */
double compute_prob_doc_at_node(Graph *graph, int ***dwl, int **dl, int node_id, int did, int depth, double eta, int W);

/* Compute the probability of the word at the given node */
double compute_prob_word_at_node( struct node n, int wid, double eta, double weta);


/*---------------------------------------------------
 *  Incrementing and Decrementing Functions
 *--------------------------------------------------- */
void decrement_cnts_sb(Graph *graph, int *path, int did, int length);
void increment_cnts_sb(Graph *graph, int *path, int did, int length);
void decrement_cp(struct node *n, int wid, int cnt);
void increment_cp(struct node *n, int wid, int cnt);


/*-------------------------------------------------------------
 *  Sampling from multinomials - includes log space sampling
 *------------------------------------------------------------- */
int sample_unnormalized_mn(double *probs, double totprob);


/*-------------------------------------------------------------
 *  Main Gibb's sampling and Metropolis Hastings Functions
 *------------------------------------------------------------- */

/* Sample path variables with stick-breaking prior */
void sample_nonparam_z(int L, int W, int D, int p_estimate, double alpha, double beta, double eta, int ntot, int *d, int *w, int **p, int *lengths, int ***dwl, int **dl, int *min_depth, Graph *graph, int allow_new_nodes, int *docconcept);

/* Sample truncated poisson hyperparameter */
void sample_nonparam_lambda(int D, int L, int ntot, double gamma_k, double gamma_theta, int *d, double *pi_d, int *path_lengths, int *word_levels, int *Nd, int **dl, int *docconcept, Graph *graph, gsl_rng *rgen, int *factorial);

/* Sample truncated geometric hyperparameters */
void sample_nonparam_pi(int D, int L, int ntot, double a, double b, int *d, double *pi_d, int *path_lengths, int *word_levels, int *sum_levels, int *Nd, int **dl, int *docconcept, Graph *graph, gsl_rng *rgen);

/* Sample truncated multinomial hyperparameters */
void sample_nonparam_multi(int D, int L, int ntot, double a, double b, int *d, double *pi_d, int *path_lengths, int *word_levels, int *sum_levels, int *Nd, int **dl, int *docconcept, Graph *graph, gsl_rng *rgen);

/* Sample level assignments for each word token */
void sample_level_assignments_poisson(int L, int W, int D, int ntot, double eta, double *pi_d, int **p, int *d, int *w, int **dl, int ***dwl, int *path_lengths, int *word_levels, int *Nd, int *min_depth, int *docconcept, int *factorial, Graph *graph);

/* Sample level assignments for each word token from a truncated geometric */
void sample_level_assignments_geometric(int L, int W, int D, int ntot, double eta, double *pi_d, int **p, int *d, int *w, int **dl, int ***dwl, int *path_lengths, int *word_levels, int *sum_levels, int *Nd, int *min_depth, int *docconcept, Graph *graph);

/* Sample level assignments for each word token from a truncated geometric */
//void sample_level_assignments_multinomial(int L, int W, int d, int ntot, double eta, double *pi_d, int **p, int *d, int *w, int **dl, int ***dwl, int *path_lengths, int *word_levels, int *sum_levels, int *Nd, int *min_depth, int *docconcept, Graph *graph);

/* Implements a Metropolis Hastings sampler for mixing over cluster labels */
void swap_cluster_labels(Graph *graph, double alpha, double beta);

/* MH sampler for beta parameter */
double sample_beta(double beta_prior, double beta, double alpha, Graph *graph, gsl_rng *rgen);

/* MH sampler for eta parameter */
double sample_eta(double eta_prior, double eta, int W, Graph *graph, gsl_rng *rgen);

#endif
