/*---------------------------------------------------
 * File:    myio.c
 * purpose: Input/output routines
 * author:  ahollowa@uci.edu
 * date:    12/14/09
 *-------------------------------------------------*/

#include "mylib.h"
#include "alloc.h"
#include "graph.h"
#include "node.h"
#include "sparse.h"



/* Read file in sparse matrix format */
int **read_sparse(char *fname, int *nr_, int *nc_)
{
  FILE *fp = fopen(fname,"r");
  int i, j, c, nr, nc, nnz;
  int **x;
  assert(fp);
  fscanf(fp,"%d", &nr);  assert(nr>0);
  fscanf(fp,"%d", &nc);  assert(nc>0);
  fscanf(fp,"%d", &nnz); assert(nnz>0);
  x = imat(nr,nc);
  assert(x);
  while (fscanf(fp, "%d%d%d", &i, &j, &c) != EOF) {
//	printf("%d %d %d\n", i, j, c);
    i--;
    j--;
    assert(i<nr);
    assert(j<nc);
    assert(c>0);
    x[i][j] = c;
  }
  fclose(fp);
  *nr_ = nr;
  *nc_ = nc;
  return x;
}

/* Read double file in sparse matrix format */
double **read_sparse_d(char *fname, int *nr_, int *nc_)
{
  FILE *fp = fopen(fname,"r");
  int row, col, i, j;
  double  c;
  int nr, nc, nnz;
  double **x;
  assert(fp);
  fscanf(fp,"%d", &nr);  assert(nr>0);
  fscanf(fp,"%d", &nc);  assert(nc>0);
  fscanf(fp,"%d", &nnz); assert(nnz>0);
  x = dmat(nr,nc);
  assert(x);
  while (fscanf(fp, "%d%d%le", &i, &j, &c) != EOF) {
	row = i;
	col = j;
    row--;
    col--;
    assert(row<nr);
    assert(col<nc);
    assert(c!=0);
    x[row][col] = c;
  }
  fclose(fp);
  *nr_ = nr;
  *nc_ = nc;
  return x;
}


void read_dvec (int n, double *x, char *fname){
  FILE *fp = fopen(fname,"r");
  int i;
  assert(fp);
  // note: removed x[i]--;
  // and changed %f to %lf
  for (i = 0; i < n; i++){ 
    fscanf(fp, "%lf", x+i ); 
  }
  fclose(fp);
}

void read_dmat(int nr, int nc, double **x, char *fname){
	FILE *fp = fopen(fname,"r"); assert(fp);
	int MAX_LEN = nc*nc, i, j, y;
	char buffer[MAX_LEN];
	
	for(i=0; i < nr; i++){
		//assert(fgets(buffer, MAX_LEN-1, fp) != NULL); //Grab a line of parameters. fgets stops when it hits a newline and it appends a null character
		for(j=0; j < nc; j++){
			//sscanf(buffer, "%lf", &(x[i][j]));
			assert(fscanf(fp,"%lf", &(x[i][j]))==1);
		}
	}
	fclose(fp);
}



/* Read integer vector (convert to zero base indexing)*/
void read_ivec(int n, int *x, char *fname) //
{
  FILE *fp = fopen(fname,"r");
  int i;
  assert(fp);
  for (i = 0; i < n; i++)  { fscanf(fp, "%d", x+i ); x[i]--; }
  fclose(fp);
}

/* Read integer vector (don't convert to zero base indexing)*/
int *read_count_ivec(int n, char *fname)	//
{
	FILE *fp = fopen(fname,"r");
	int i, *x;
	x = ivec(n);
	for (i=0; i < n; i++)	{	fscanf(fp, "%d", x+i); }
	fclose(fp);
	return x;
}

/* Read docword file in sparse matrix format*/
void read_dw(char *fname, int *d, int *w, int *D, int *W)
{
  int i,wt,dt,ct,count,nnz;
  FILE *fp = fopen(fname ,"r"); assert(fp);
  count = 0;
  fscanf(fp,"%d", D);    assert(*D>0);
  fscanf(fp,"%d", W);    assert(*W>0);
  fscanf(fp,"%d", &nnz); assert(nnz>0);
  while (fscanf(fp, "%d%d%d", &dt, &wt, &ct) != EOF) {
    for (i = count; i < count+ct; i++) {
      w[i] = wt-1;
      d[i] = dt-1;
    }
    count += ct;
  }
  fclose(fp);
}



/* Read a file with a variable number of integers per line
 *
 * file - File to be read
 * N    - Number of rows in file
 * M    - Maximum number of columns in file. If M = -1, we search the file
 *        to find the longest line. Else, we create an NxM matrix.
 * num_per_line - An array that stores the no. of elements per row     
 *
 */
int ** read_variable_line_i(char *file, int N, int M, int *num_per_line){
  int MAX_LEN = 2000, cnt=0,i=0,j=0,x=0, print=FALSE, longest_line=0;
  int **tmp; 
  char *str, buf[MAX_LEN+1];
  FILE *fp = fopen(file, "r"); assert(fp);
  
  if(print){
    printf("Entering read_variable_line\n");
    printf("read_variable_line: MAX_LEN == 2000\n");
  }


  //Count the number of integers per line
  while( fgets(buf, MAX_LEN+1, fp) != NULL ){

    //should check if the second to last cell in buf is either a digit or a space
    //if so, then the line was too big to fit into the buffer
    if( isdigit(buf[MAX_LEN-1]) || buf[MAX_LEN-1] == ' '){
      printf("Line overflowed buffer: ##%d##\n", buf[MAX_LEN-1]);
    }
    
    //Count how many path i.d.s on this line
    str = strtok(buf, " \n");
    while( str != NULL ){
      cnt++;
      str = strtok(NULL, " \n");
    }
    
    num_per_line[i] = cnt;
    if(print){ printf("Line %d has count %d\n", i+1, cnt);}
	if( cnt > longest_line){
		longest_line = cnt;
	}
    i++;
    cnt = 0;
  }
  
  if( M == -1 ){
	tmp = imat(N,longest_line);
  }else{
    assert(longest_line <= M);
	tmp = imat(N,M);
  }
  
  assert(i==N);
  assert(fseek(fp,0,SEEK_SET)==0); // jump to beginning of file
  
  //Read in the file
  i=0; j=0;
  while(fgets(buf,MAX_LEN+1,fp) != NULL ){
    if(num_per_line[i] > 0 ){
      if(print){ printf("Line %d has elements:\n", i+1); }
      str = strtok(buf," \n");
      tmp[i][j] = strtol(str,NULL,0); 
      tmp[i][j]--;
      if(print){printf("\t%d\n", tmp[i][j]+1); }
      j++;
      while( (str=strtok(NULL, " \n")) != NULL ){
		x = strtol(str,NULL,0); 
		tmp[i][j] = x-1; 
		if(print){printf("\t%d\n", tmp[i][j]+1); }
		j++;
      }
    }
    else if(print){
      printf("Line %d has no elements\n", i+1);
    }
    i++;
    j=0;
  }
  
  fclose(fp);
  return(tmp);
}


int ** read_line_i(char *file, int N, int num_per_line){
  int MAX_LEN = 100*num_per_line;
  int cnt=0,i=0,j=0,x=0, print=0;
  FILE *fp = fopen(file, "r"); assert(fp);
  char buf[MAX_LEN+1];
  int **tmp = imat(N,num_per_line);
  char *str;

  if(print){
    printf("Entering read_variable_line\n");
    printf("read_variable_line: MAX_LEN == 2000\n");
  }

  //Read in the file
  i=0; j=0;
  while(fgets(buf,MAX_LEN+1,fp) != NULL ){
      str = strtok(buf," \n");
      tmp[i][j] = strtol(str,NULL,0); 
      tmp[i][j]--;
      if(print){printf("\t%d\n", tmp[i][j]+1); }
      j++;
      while( (str=strtok(NULL, " \n")) != NULL ){
		x = strtol(str,NULL,0); 
		tmp[i][j] = x-1; 
		if(print){printf("\t%d\n", tmp[i][j]+1); }
		j++;
      }
	  i++;
      j=0;
  }
  
  fclose(fp);
  return(tmp);
}


//double ** read_line_d(char *file, int N, int num_per_line){
//  int MAX_LEN = 100*num_per_line;
//  int cnt=0,i=0,j=0, print=1;
//  FILE *fp = fopen(file, "r"); assert(fp);
//  char buf[MAX_LEN+1];
//  double **tmp = dmat(N,num_per_line);
//  char *str;
//
//  if(print){
//    printf("Entering read_variable_line\n");
//    printf("read_variable_line: MAX_LEN == 2000\n");
//  }
//
//  //Read in the file
//  i=0; j=0;
//  while(fgets(buf,MAX_LEN+1,fp) != NULL ){
//      str = strtok(buf," \n");
//      tmp[i][j] = strtol(str,NULL,0); 
//      tmp[i][j];
//      if(print){printf("\t%f\n", tmp[i][j]); }
//      j++;
//      while( (str=strtok(NULL, " \n")) != NULL ){
//		tmp[i][j] = strtol(str,NULL,0); 
//		if(print){printf("\t%f\n", tmp[i][j]); }
//		j++;
//      }
//	  i++;
//      j=0;
//  }
//  
//  fclose(fp);
//  return(tmp);
//}



/* Write in sparse matrix format */
void write_sparse(int nr, int nc, int **x, char *fname) //
{
  FILE *fp = fopen(fname,"w");
  int i, j;
  assert(fp);
  fprintf(fp, "%d\n", nr);
  fprintf(fp, "%d\n", nc);
  fprintf(fp, "%d\n", countnnz(nr,nc,x));
  for (i = 0; i < nr; i++)
    for (j = 0; j < nc; j++)
      if (x[i][j] > 0) fprintf(fp, "%d %d %d\n", i+1 , j+1 , x[i][j]);
  fclose(fp);
}


void write_sparse_d(int nr, int nc, double **x, char *fname) //
{
  FILE *fp = fopen(fname,"w");
  int i, j;
  assert(fp);
  fprintf(fp, "%d\n", nr);
  fprintf(fp, "%d\n", nc);
  fprintf(fp, "%d\n", nr*nc); // <-- fix this
  for (i = 0; i < nr; i++)
    for (j = 0; j < nc; j++)
      if (x[i][j] != 0) fprintf(fp, "%d %d %e\n", i+1 , j+1 , x[i][j]);
  fclose(fp);
}


/* Write double vector*/
void write_dvec(int n, double *x, char *fname){
  int i;
  FILE *fp = fopen(fname, "w");
  assert(fp);
  for( i = 0; i < n; i++){
    fprintf(fp, "%.10f\n", x[i]);
  }
  fclose(fp);
}


/* Write integer vector (convert to one base indexing) */
void write_ivec(int n, int *x, char *fname) //
{
  FILE *fp = fopen(fname,"w");
  int i;
  assert(fp);
  for (i = 0; i < n; i++)  fprintf(fp, "%d\n", x[i]+1 );
  fclose(fp);
}

/* Write integer vector (keep zero base indexing) */
void write_count_ivec( int n, int *x, char *fname)	//
{
	FILE *fp = fopen(fname,"w");
	int i;
	assert(fp);
	for (i=0; i < n; i++)	fprintf(fp, "%d\n", x[i]);
	fclose(fp);
}


void write_variable_line_i( char *file,  int **matrix, int *line_length, int num_lines){
  int i, j; 
  FILE *fp = fopen(file, "w");
  assert(fp);

  for(i=0; i< num_lines; i++){    
    for(j=0; j <= line_length[i]; j++){
      fprintf(fp, "%d ", matrix[i][j]+1);
	}
    fprintf(fp, "\n");
  }

  fclose(fp);
}


/* Print out lp from each of the nodes in the graph*/
void print_cp(Graph *graph, int W, int L, char *file){
	int i = 0, l, j = 0, cnt=0;
	int **cp;
	
//	for( i=0; i < graph->capacity; i++){
//		if( graph->nodes[i].id == NOT_IN_USE){ break; }
//		cnt++;
//	}
	cp = imat(W,graph->capacity);
	
	for( i = 0; i < graph->capacity; i++){
		if(graph->nodes[i].id == NOT_IN_USE){continue;}
		for( j = 0; j < W; j++){
			cp[j][i] = graph->nodes[i].cp[j];
		}
	}
	write_sparse(W, graph->capacity, cp, file);
	free(cp);
	return;
}


/* Print out the permutation of the nodes for each stick-breaking distribution */
void print_perm(Graph *graph, char *perm_file){
	int i,j;
	int **sb = imat(graph->capacity, graph->capacity);
	int *x = ivec(graph->capacity);
	 
	for( i = 0; i < graph->capacity; i++){
		if( graph->nodes[i].id == NOT_IN_USE){
			x[i] = -1;
			continue; 
		}
		x[i] = graph->nodes[i].num_feasible-1;
		for( j = 0; j < graph->nodes[i].num_feasible; j++){
			sb[i][j] = graph->nodes[i].feasible[j];
		}
	}

	write_variable_line_i(perm_file, sb, x, graph->capacity);
	free(sb);
	free(x);
	
	return;
}

char *create_filename(char *dir, char *file){
	char *tmp = calloc(strlen(dir)+strlen(file)+5, sizeof(char));
	assert(tmp);
	
	strncpy(tmp,dir,strlen(dir)); // don't copy the null terminating character (if it was even there)
	tmp[strlen(dir)] = '\0';      // explicitly put a null terminating character
	strncat(tmp,file,strlen(file));
	return(tmp);
}

/* Print out path assignment for each word token */
void print_paths(int num_paths, int *path_lengths, int **paths, char *file){
	int i, j;
	FILE *fp = fopen(file, "w");
	assert(fp);
	
	for(i = 0; i < num_paths; i++){
		fprintf(fp, "did=%d\n", i);
		fprintf(fp, "\t");
		for(j=0; j <= path_lengths[i]; j++){
			fprintf(fp," %d", paths[i][j]);
		}
		fprintf(fp,"\n");
	}
	
	fclose(fp);
	return;
}


/* Print out state of Gibbs sampler */
void print_state(int iter, int ntot, int D, int W, int L, char *dir, int **paths, int *path_lengths, double *pi_d, int *levels, Graph *graph){
	int j, k;
	char *filename, *outfile;
	FILE *fp;


	//Print out paths
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "path_iter%d.txt",iter);
	outfile = create_filename(dir, filename);
	write_variable_line_i(outfile, paths, path_lengths, D);
	free(outfile);
	free(filename);
	
	//Print out pi variables
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "pi_iter%d.txt",iter);
	outfile = create_filename(dir, filename);
	write_dvec(D, pi_d, outfile);
	free(outfile); 
	free(filename);
	
	//Print out levels
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "levels_iter%d.txt", iter);		
	outfile = create_filename(dir, filename);
	write_count_ivec(ntot, levels, outfile);
	free(outfile);
	free(filename);
	
	
	//Print out topics
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "cp_iter%d.txt", iter);		
	outfile = create_filename(dir, filename);
	print_cp(graph, W, L, outfile);
	free(outfile);	
	free(filename);
	
	//Print out graph structure
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "graph_iter%d.txt",iter);
	outfile = create_filename(dir, filename);
	print_graph(graph, outfile);
	free(outfile);
	free(filename);


	//Print out permutations
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "perm_iter%d.txt",iter);
	outfile = create_filename(dir, filename);
	print_perm(graph, outfile);
	free(outfile);
	free(filename);	

	//Print out etot
	filename = calloc(BUFF_SIZE+1, sizeof(char));
	sprintf(filename, "etot_iter%d.txt",iter);
	outfile = create_filename(dir,filename);
	fp = fopen(outfile, "w"); assert(fp);
	for(j=0; j < graph->capacity; j++){
		if( graph->nodes[j].id==NOT_IN_USE){continue;}
		for(k=0; k < graph->nodes[j].k_capacity; k++){
			fprintf(fp, "%d ", graph->nodes[j].etot_k[k]);
		}
		fprintf(fp,"\n");
	}
	fclose(fp);
	free(outfile);
	free(filename);

}







