	/*---------------------------------------------------
 * File:    init.c
 * purpose: Routines for initializing the gibbs chain
 * author:  ahollowa@uci.edu
 * date:    12/22/2009
 *-------------------------------------------------*/
 
#include "mylib.h"
#include "init.h"
#include "sampler.h"
#include "alloc.h"
#include "node.h"
#include "graph.h"
#include "checksum.h"


/* Random permutation of integers 0...n */
int *randperm(int n) //
{
  int *order = ivec(n);
  int k, nn, takeanumber, temp;
  for (k=0; k<n; k++) order[ k ] = k;
  nn = n;
  for (k=0; k<n; k++) {
    // take a number between 0 and nn-1
    takeanumber = (int) (nn*drand());
    temp = order[ nn-1 ];
    order[ nn-1 ] = order[ takeanumber ];
    order[ takeanumber ] = temp;
    nn--;
  }
  return order;
}


/* Initializes to a full graph */
void randomassignment_graph(int K, int L, int D, int W, int ntot, int *w, int *d, int **paths, int *path_lengths, int *levels, int *sum_levels, int ***dwl, int **dl, int *min_depth, double *pi_d, Graph *graph, int *docconcept){
	int i, j, exit, node, did, wid, level, id;

	//Create the full graph
	for(i=1; i < L; i++){
		for(j=0; j < K-1; j++){
			add_new_node(graph, (K-1)*(i-1)+(j+1), i, L, W, D); 
			update_next_avail_id(graph);
		}
	}

	for(i=0; i < D; i++){
		if(docconcept[i] != FIXED_PATH){

			//Sample path for document
			paths[i][0] = 0;
			for(j=1; j < L; j++){
				id = (int)(graph->num_equiv[j]*drand());  //integer between 0 and num_equiv-1
				node = graph->equivalence[id][j];

				if( node == NEW_NODE ){
					path_lengths[i] = j-1;
					break;
				}else if( j == L-1){
					paths[i][j] = node;
					path_lengths[i] = j;
				}else{
					paths[i][j] = node;
				}
			}
			pi_d[i] = drand();
			min_depth[i] = -1;
			increment_cnts_sb(graph,paths[i],i,path_lengths[i]);	
		}
	}


	//Assign each word to a level in the path
	for(i=0; i < ntot; i++){
		if( docconcept[d[i]] != FIXED_PATH){
			did = d[i];
			wid = w[i];
			
			level = (int)((path_lengths[did]+1)*drand()); //sample an integer between 0 and path_lengths+1
			assert(level <= path_lengths[did]);

			if( level > min_depth[did] ){
				min_depth[did] = level;
			}
			
			dwl[did][wid][level]++;
			dl[did][level]++;
			sum_levels[did] += level;
			levels[i] = level;
			
			node = paths[did][path_lengths[did]-level];
			if( graph->nodes[node].keep_cp_fixed == FALSE ){
				increment_cp( &(graph->nodes[node]), wid, 1);
			}
		}
	}

	return;
}


/* Initializes to a chain of depth L */
void randomassignment_chain(int L, int D, int W, int ntot, int *w, int *d, int **paths, int *path_lengths, int *levels, int *sum_levels, int ***dwl, int **dl, int *min_depth, double *pi_d, Graph *graph, int *docconcept){
	int i, j, exit, node, did, wid, level;

	//Create a chain of length L
	if(graph->nodes[0].id != NOT_IN_USE){
		for(i=1; i < L; i++){
			assert(graph->nodes[i].id == NOT_IN_USE);
			add_new_node(graph, i, i, L, W, D);
			update_next_avail_id(graph);
		}
	}
	else{
		for(i=0; i < L; i++){
			assert(graph->nodes[i].id == NOT_IN_USE);
			add_new_node(graph, i, i, L, W, D);
			update_next_avail_id(graph);
		}
	}


	for(i=0; i < D; i++){
		if(docconcept[i] != FIXED_PATH){

			//Sample path for document
			exit = (int)(L*drand()); // sample integer between 0 and L-1
			for(j=0; j <=exit; j++){
				paths[i][j] = j;
			}
			path_lengths[i] = exit;
			pi_d[i] = drand();
			min_depth[i] = -1;
			increment_cnts_sb(graph,paths[i],i,path_lengths[i]);	
		}
	}


	//Assign each word to a level in the path
	for(i=0; i < ntot; i++){
		if( docconcept[d[i]] != FIXED_PATH){
			did = d[i];
			wid = w[i];
			
			level = (int)((path_lengths[did]+1)*drand()); //sample an integer between 0 and path_lengths+1
			assert(level <= path_lengths[did]);

			if( level > min_depth[did] ){
				min_depth[did] = level;
			}
			
			dwl[did][wid][level]++;
			dl[did][level]++;
			sum_levels[did] += level;
			levels[i] = level;
			
			node = paths[did][path_lengths[did]-level];
			if( graph->nodes[node].keep_cp_fixed == FALSE ){
				increment_cp( &(graph->nodes[node]), wid, 1);
			}
		}
	}
	

	return;
}

/* Initialize gibbs sampling chain. Every path is initialized to the root node. */
void randomassignment(int L, int D, int W, int ntot, int *w, int *d, int **paths, int *path_lengths, int *levels, int *sum_levels, int ***dwl, int **dl, int *min_depth, double *pi_d, Graph *graph, int *docconcept){
	int i, wid, did, pid, j, from, to, level, node, exit_node;
	int curr;
	
	//Assign each document to the root node
	for(i = 0; i < D; i++){
		if( docconcept[i] != FIXED_PATH){
			path_lengths[i] = 0;
			for(j=0; j < L; j++){
				paths[i][j] = 0;
			}
			pi_d[i] = drand();
			min_depth[i] = -1;
			increment_cnts_sb(graph,paths[i],i,path_lengths[i]);
		}
	}
	
	//Assign each word to the root node
	for(i=0; i<ntot; i++){
		if( docconcept[d[i]] != FIXED_PATH){
			did = d[i];
			wid = w[i];			
			assert(path_lengths[did] == 0);
			level = (int)((path_lengths[did]+1)*drand());

			if( level > min_depth[did] ){
				min_depth[did] = level;
			}
			
			dwl[did][wid][level]++;
			dl[did][level]++;
			sum_levels[did] += level;
			levels[i] = level;
			
			node = paths[did][path_lengths[did]-level];
			assert(node==0);
			if( graph->nodes[node].keep_cp_fixed == FALSE ){
				increment_cp( &(graph->nodes[node]), wid, 1);
			}
		}
	}
	return;
}


/* Initialize gibbs sampling chain. Every path ends at docconcept unless FREE_PATH. Then it ends at root node */
void randomassignment_no_edges(int L, int D, int W, int ntot, int *w, int *d, int **paths, int *path_lengths, int *levels, int *sum_levels, int ***dwl, int **dl, int *min_depth, int *docconcept, double *pi_d, Graph *graph){
	int i, exit_level, j, wid, level, did, node;

	for(i=0; i < D; i++){
		if( docconcept[i] != FIXED_PATH){
			pi_d[i] = drand();
			min_depth[i] = -1;
			
			if( docconcept[i] == FREE_PATH){
				exit_level = 0;
			}
			else{
				exit_level = graph->nodes[docconcept[i]].equiv_class;
			}

			if( exit_level == 0 ){
				increment_cnts_sb(graph,paths[i],i,path_lengths[i]);
			}
			else{
				for(j = 0; j < exit_level; j++){
					node = (int)((graph->num_equiv[j]-1)*drand());
					paths[i][j] = graph->equivalence[node][j];
					assert(node != NEW_NODE);
				}
				paths[i][j] = docconcept[i];
				path_lengths[i] = j;
				increment_cnts_sb(graph,paths[i],i,path_lengths[i]);
			}
		}
	}
	
	for(i=0; i < ntot; i++){
		if( docconcept[d[i]] != FIXED_PATH ){
			wid = w[i];	
			did = d[i];	
			level = (int)((path_lengths[did]+1)*drand());

			if( level > min_depth[did] ){
				min_depth[did] = level;
			}
			
			dwl[did][wid][level]++;
			dl[did][level]++;
			sum_levels[did] += level;
			levels[i] = level;
			
			node = paths[did][path_lengths[did]-level];
			if( graph->nodes[node].keep_cp_fixed == FALSE ){
				increment_cp( &(graph->nodes[node]), wid, 1);
			}
		}
	}

	return;

}

