/*---------------------------------------------------
 * File:    graph.c
 * purpose: Routines for graph structure
 * author:  ahollowa@uci.edu
 * date:    12/11/2009
 *-------------------------------------------------*/

#include "mylib.h"
#include "graph.h"
#include "alloc.h"
#include "node.h"
#include "sampler.h"


/* Allocate memory for graph data structure */
Graph *allocate_graph(){
	Graph *tmp = malloc(sizeof(Graph));
	assert(tmp);
	return(tmp);	
}


/* Increase capacity of nodes */
struct node *increase_nodes_capacity(struct node *nodes_array, int capacity){
	nodes_array = (struct node *)realloc(nodes_array, capacity*sizeof(struct node));
	assert(nodes_array);
	return(nodes_array);
}


/* Print the graph */
void print_graph(Graph *graph, char *outfile){
	int i, j, node, sum;
	FILE *fp = fopen(outfile, "w");
	assert(fp);
	
	fprintf(fp, "Graph Information:\n");
	fprintf(fp, "next_avail_id = %d\n", graph->next_avail_id);
	fprintf(fp, "capacity      = %d\n", graph->capacity);
	fprintf(fp, "max_depth     = %d\n", graph->max_depth);
	
	fprintf(fp,"Equivalence Classes:\n");
	for(i=0; i < graph->max_depth; i++){
		fprintf(fp, "\tClass %d: ", i);
		for(j=0; j < graph->num_equiv[i]; j++){
			if( graph->equivalence[j][i] == NEW_NODE)
				fprintf(fp, " %d", graph->equivalence[j][i]);
			else
				fprintf(fp, " %d", graph->equivalence[j][i]+1);
		}
		fprintf(fp, "\n");
	}
	fprintf(fp, "\n");

	
	fprintf(fp, "\nNode Information:\n");
	for( i = 0; i < graph->capacity; i++){
		fprintf(fp, "\tPosition %d in Node Array:\n", i);
		if( graph->nodes[i].id == NOT_IN_USE ){
			fprintf(fp, "\tNode Id       = NOT_IN_USE\n");
			fprintf(fp, "\n");
			continue;
		}else{
			fprintf(fp, "\tNode Id       = %d\n", graph->nodes[i].id);
		}
		fprintf(fp, "\tEquiv Class   = %d\n", graph->nodes[i].equiv_class);
		fprintf(fp, "\tK Capacity    = %d\n", graph->nodes[i].k_capacity);
		fprintf(fp, "\tNum. Feasible = %d\n", graph->nodes[i].num_feasible);
		fprintf(fp, "\tFeasible:\n");
		fprintf(fp, "\t\t");
		for( j = 0; j < graph->nodes[i].num_feasible; j++){
			fprintf(fp, " %d", graph->nodes[i].feasible[j]+1);
		}
		fprintf(fp,"\n");
		
		fprintf(fp, "\tPerm:\n");
		fprintf(fp, "\t\tNode ----------> Pos\n");
		for( j = 0; j < graph->nodes[i].num_feasible; j++){
			node = graph->nodes[i].feasible[j];
			fprintf(fp, "\t\t%d ---- %d ----> %d\n", node+1, graph->nodes[i].etot_k[node], graph->nodes[i].perm[node]);
		}
		fprintf(fp, "\tZtot          = %d\n", graph->nodes[i].ztot);


		fprintf(fp,"\n\n");
	}
	
	fclose(fp);
	return;
}


/* Allocate space for new node (if needed). Allocate memory for node's data structures. Construct stick-breaking distribution */
void add_new_node(Graph *graph, int add_id, int equiv_class, int L, int W, int D){
	int i, k, node, pos_of_new_node, num_nodes;

	//COW: Should have some mechanism in here so that if this function is called but there is no more room in the equivalence
	//matrix then we return an error or some signal saying the node could not be created

	// Ensure graph capacity is large enough
	if( add_id >= graph->capacity){
		graph->nodes = increase_nodes_capacity(graph->nodes, 2*add_id);
		for(k= graph->capacity; k < 2*add_id; k++){
			graph->nodes[k].id = NOT_IN_USE;
		}
		graph->capacity = 2*add_id;
		
	}
	
	// Initialize the new node to the correct values
	graph->nodes[add_id].id = add_id;
	graph->nodes[add_id].equiv_class = equiv_class;
	graph->nodes[add_id].k_capacity = graph->capacity; //As many nodes as are currently in the graph...
	graph->nodes[add_id].D = D;
	graph->nodes[add_id].W = W;

	//Initialize a topic
	graph->nodes[add_id].cp = ivec(W);
	graph->nodes[add_id].ztot = 0;
	graph->nodes[add_id].keep_cp_fixed = FALSE;
	
	graph->nodes[add_id].etot_k = ivec(graph->nodes[add_id].k_capacity);
	graph->nodes[add_id].etot_k_agg = ivec(graph->nodes[add_id].k_capacity);

	//Allocate memory for stick-breaking distribution
	graph->nodes[add_id].feasible = ivec(graph->nodes[add_id].k_capacity);
	graph->nodes[add_id].perm     = ivec(graph->nodes[add_id].k_capacity);

	
	//Create stick-breaking distribution
	if( equiv_class+1 == L ){
		graph->nodes[add_id].num_feasible = 1;
		graph->nodes[add_id].feasible[0] = add_id;
		graph->nodes[add_id].perm[add_id] = 0;
	}
	else{
		graph->nodes[add_id].num_feasible = graph->num_equiv[equiv_class+1];
		for( i = 0; i < graph->nodes[add_id].num_feasible; i++){
			node = graph->equivalence[i][equiv_class+1];			
			if( node == NEW_NODE ){
				graph->nodes[add_id].feasible[i] = add_id;
				graph->nodes[add_id].perm[add_id] = i;
			}else{
				graph->nodes[add_id].feasible[i] = node;
				graph->nodes[add_id].perm[node] = i;
			}
		}
	}
	
	//Update other stick breaking distributions to include this node
	if( equiv_class-1 >= 0 ){
		for( k = 0; k < graph->num_equiv[equiv_class-1]; k++){
			node = graph->equivalence[k][equiv_class-1];
			
			if( node == NEW_NODE ){
				continue;
			}
			update_stick_break_distrib( &graph->nodes[node], add_id);
		}
	}
	
	//Add node to equivalence class
	num_nodes = graph->num_equiv[equiv_class];
	pos_of_new_node = graph->num_equiv[equiv_class] - 1;
	if( num_nodes+1 >= graph->equiv_capacity){
		graph->equivalence = resize_two_dim_i(graph->equiv_capacity, 2*graph->equiv_capacity, graph->max_depth, graph->equivalence);
		graph->equiv_capacity *= 2;
	}
	graph->equivalence[pos_of_new_node][equiv_class] = add_id;
	graph->equivalence[pos_of_new_node+1][equiv_class] = NEW_NODE;
	graph->num_equiv[equiv_class]++;
	
	return;
}




///* Delete node from graph */
//void delete_node(Graph *graph, int delete_id){
//	int i, equiv_class, x, stick_break_pos, k;
//	
//	graph->nodes[delete_id].id = NOT_IN_USE;
//	deallocate_node(graph->nodes[delete_id]);
//	
//	// Remove from feasible node
//	equiv_class = graph->nodes[delete_id].equiv_class-1;
//	assert(equiv_class >= 0); //should always be the case
//	for( i = 0; i < graph->num_equiv[equiv_class]; i++){
//		x = graph->equivalence[i][equiv_class];
//		stick_break_pos = graph->nodes[x].perm[delete_id];
//
//		//Decrement perm matrix
//		for( k = 0; k < graph->nodes[x].k_capacity; k++){
//			if( graph->nodes[x].perm[k] > stick_break_pos ){
//				graph->nodes[x].perm[k]--;
//			}
//		}
//		graph->nodes[x].perm[delete_id] = 0;
//				
//		//Decrement feasible matrix
//		for( k = 0; k < graph->nodes[x].num_feasible; k++){
//			if( k > stick_break_pos){
//				graph->nodes[x].feasible[k-1] = graph->nodes[x].feasible[k];
//			}				
//		}
//	}
//	return;
//}


/* Update next_avail_id. Increase graph size if needed */
void update_next_avail_id(Graph *graph){
	int i;
	
	for( i = 0; i < graph->capacity; i++){
		if( graph->nodes[i].id == NOT_IN_USE ){
			graph->next_avail_id = i;
			break;
		}
	}

	if( i == graph->capacity){
		graph->nodes = increase_nodes_capacity(graph->nodes, graph->capacity);
		graph->next_avail_id = i;
	}
	
	return;
}



