GetExperiment <- function(networkWithTreatments, treatmentSet, experimentParameters, experimentId, outputPath, seed = sample.int(10000, 1)){
  # networkWithTreatments is data.frame containing 
  # consumer, producer, baseLevelExposure, treatmentLevelExposure, treatmentlevel exposures for all treatments
  networkWithTreatments <- networkWithTreatments %>% select(producer, consumer, edgeStrength, baseLevelExposure, eval(treatmentSet))
  set.seed(seed * experimentId)
  consumerSet <- unique(networkWithTreatments$consumer)
  producerSet <- unique(networkWithTreatments$producer)
  
  nTreatments <- length(treatmentSet)
  nConsumers <- length(consumerSet)
  nProducers <- length(producerSet)
  
  cat(paste0("Getting design for experiment ", experimentId, " with ", nTreatments, " treatments, ", 
             nProducers, " producers, ", nConsumers, " consumers and ", nrow(networkWithTreatments), " connections \n"))

  # assign randomly selected producers to one of the treatment conditions
  producerInTreatment <- data.frame(producer = sample(producerSet,  size = experimentParameters$producerSampleSize),
                                treatment = sample(nTreatments, experimentParameters$producerSampleSize, replace = TRUE))
  # assign randomly selected producers to one of the treatment conditions
  consumerInTreatment <- data.frame(consumer = sample(setdiff(consumerSet, producerInTreatment$producer),  
                                                      size = experimentParameters$consumerSampleSize),
                                    treatment = sample(nTreatments, experimentParameters$consumerSampleSize, replace = TRUE))
  # update consumerInTreatment by appending producerInTreatment
  consumerInTreatment <- rbind(consumerInTreatment, producerInTreatment %>% rename(consumer = producer))
  
  treatmentAssignment <- networkWithTreatments %>% group_by(producer) %>% 
    left_join(producerInTreatment, by = "producer") %>% rename(producerInTreatment = treatment) %>% ungroup() %>% 
    group_by(consumer) %>% left_join(consumerInTreatment, by = "consumer") %>% rename(consumerInTreatment = treatment)
  treatmentAssignment$consumerInTreatment[is.na(treatmentAssignment$consumerInTreatment)] <- 0
  
  # split the network based on if producers are assigned to treatments or not
  fullDesign <- split(treatmentAssignment,  !is.na(treatmentAssignment$producerInTreatment))
  names(fullDesign) <- c("nonExperimentalSet", "experimentalSet")
  
  cat("Preparing data for optimization \n")
  # perpare data for optimization
  dataWithSelectedConsumers <- SelectConsumerForOptimation(fullDesign$experimentalSet, experimentParameters, treatmentSet)
  dataWithSelectedConsumers <- split(dataWithSelectedConsumers, dataWithSelectedConsumers$selectedForOptimization)
  names(dataWithSelectedConsumers) <- c("notSelected", "selected")
  
  finalData <- dataWithSelectedConsumers$selected %>% 
    select(producer, consumer, exposureDiff, baseLevelExposure, Aj, q, lj, uj, lowerBound, upperBound)
  cat(paste0("Solving QP with ", length(unique(finalData$consumer)), " consumers, ", 
             length(unique(finalData$producer)), " producers and ", nrow(finalData), " connections \n"))
  optimalPij <- QPSolver(finalData[sample(nrow(finalData)), ], experimentParameters$groupSizeQP, experimentParameters$maxIterQP)
  optimalPij <- as_tibble(optimalPij) %>% select(consumer = consumerId, producer = producerId, optimalPij = pijCurrent) %>% 
    mutate(optimalPij = abs(optimalPij))
  
  dataWithSelectedConsumers$selected <- dataWithSelectedConsumers$selected %>% 
    left_join(optimalPij, by = c("consumer" = "consumer", "producer" = "producer")) %>% 
    mutate(experimentLevelExposure = optimalPij) %>% select(-optimalPij)
  fullDesign$experimentalSet <- do.call("rbind", dataWithSelectedConsumers) %>% 
    select(consumer, producer, edgeStrength, producerInTreatment, consumerInTreatment, baseLevelExposure, experimentLevelExposure) 
  
  normalizationFactor <- dataWithSelectedConsumers$selected %>% group_by(consumer) %>%
    summarise(normalizationFactor = (1 - sum(baseLevelExposure)) / (1 - sum(experimentLevelExposure)))
  
  fullDesign$nonExperimentalSet <- fullDesign$nonExperimentalSet %>% group_by(consumer) %>% 
    left_join(normalizationFactor, by = "consumer") %>% ungroup()
  fullDesign$nonExperimentalSet$normalizationFactor[is.na(fullDesign$nonExperimentalSet$normalizationFactor)] <- 1 
  fullDesign$nonExperimentalSet <- split(fullDesign$nonExperimentalSet, fullDesign$nonExperimentalSet$consumerInTreatment)
  fullDesign$nonExperimentalSet <- lapply(1:length(fullDesign$nonExperimentalSet), 
                                          function(i) fullDesign$nonExperimentalSet[[i]] %>% 
                                            mutate(experimentLevelExposure = unlist(fullDesign$nonExperimentalSet[[i]][, c("baseLevelExposure", treatmentSet)[i]])))
  fullDesign$nonExperimentalSet <- do.call("rbind", fullDesign$nonExperimentalSet)
  fullDesign$nonExperimentalSet <- fullDesign$nonExperimentalSet %>% 
    mutate(experimentLevelExposure = experimentLevelExposure / normalizationFactor) %>% 
    select(consumer, producer, edgeStrength, producerInTreatment, consumerInTreatment, baseLevelExposure, experimentLevelExposure)
  producerInTreatmentSet <- producerInTreatment$producer
  fullDesign <- do.call("rbind", fullDesign) %>% filter(!is.na(producerInTreatment) | (consumer %in% producerInTreatmentSet))
  fileName <- paste0(outputPath, "design_ID=", experimentId, ".Rdata")
  save(fullDesign, file = fileName)
  return(fileName)
}
