index <- function(from, to) {
    if (from > to) {
      R<-c()
    } else {
      R<-from:to
    }
    return(R)
}

nvectors<-function(arities) {
  prod(arities)
}

next_vector <- function(x, arities) {
    i <- which(x != arities - 1)
    if (length(i) == 0) return(NA)
    i <- max(i)
    x[i] <- x[i] + 1
    x[index(i+1,length(x))] <- 0
    return(x)
}

removeDuplicateLabels<-function(labels) {
    duplicate<-rep(FALSE,nrow(labels))
    for ( i in 1:nrow(labels)) {
        for ( j in index(1,i-1)) { #check lower indexed lables
            if ( all(labels[i,] == labels[j,]) ) {
                duplicate[i]<-TRUE
                break
            }
            if ( duplicate[i]) break
        }
    }
    return(labels[!duplicate,,drop=FALSE])
}

merge_CPT<-function(CPT, i, j) {
    #merges rows i and j
    npa <- ncol(CPT) - 1
    k <- min( c(CPT[which(CPT[,npa+1] == CPT[i,npa+1]),npa+1],CPT[which(CPT[,npa+1] == CPT[j,npa+1]),npa+1]) )
    #mark the rows to be merged with -1
    CPT[which(CPT[,npa+1] == CPT[i,npa+1]),npa+1] <- k
    CPT[which(CPT[,npa+1] == CPT[j,npa+1]),npa+1] <- k
    return(reindex_CPT(CPT))
}

reindex_CPT<-function(CPT) {
    npa <- ncol(CPT) - 1
    i <- 0
    while ( TRUE ) {
        i <- i + 1
        if ( all(CPT[,npa+1] < i) ) break #if all are less 
        if ( any(CPT[,npa+1] == i) ) next #if the index is used
        #otherwise must replace wit
        #cat(i)
        k <- min( CPT[CPT[,npa+1] > i,npa+1] )
        CPT[CPT[,npa+1]== k,npa+1] <- i
    }
    return(CPT)
}


ldag_to_pdag <- function(M, verbose = FALSE) {
    #this should be ldag to pdag?
    stopifnot(class(M) == 'ldag')
    #adding labels to edges such that no additional CSIs are formed
    #how can this be done?
    M$CPT<-list()
    #create the CPT
    #recalculate labels from the CPT?
    for ( to in 1:nrow(M$G) ) {
        pa <- which(M$G[to,] == 1)
        #create a CPT
        npa <- length(pa)
        M$CPT[[to]] <- array(0, c(nvectors(M$arities[pa]), npa + 1))
        colnames(M$CPT[[to]]) <- c(pa, 'P')
        rownames(M$CPT[[to]]) <- c()
        conf<-rep(0,npa)
        i<-1
        while (!any(is.na(conf))) {
          M$CPT[[to]][i,] <- c(conf,i)
          conf <- next_vector(conf,M$arities[pa])
          i <- i + 1
        }
        #now enforce the restrictions due to the labels
        for (e in M$edges ) {
            if ( e$to == to ) {
                for ( i in index(1,nrow(e$labels)) ) {
                    fromindex <- which(pa == e$from) #find the index of from
                    conf <- rep(NA, npa)
                    conf[-fromindex] <- e$labels[i,] #this should be the order in the parentset!
                    #NOTE NOW THERE MAY BE MANY ROWS TO MERGE!!!!
                    conf[fromindex] <- 0
                    row0<-which(colSums(t(M$CPT[[to]][,1:npa]) == conf) == npa)
                    for ( confi in index(1,M$arities[e$from]-1) ) {
                        conf[fromindex] <- confi
                        row<-which(colSums(t(M$CPT[[to]][,1:npa]) == conf) == npa)
                        M$CPT[[to]]<-merge_CPT(M$CPT[[to]],row0,row)
                    }#for confi
                }#for i
            }#if e$to == to
        }#for e 
    } #for to
    M$edges <- NULL
    class(M) <- 'pdag'
    return(M)
}

pdag_to_ldag <- function(M) {
    n <- nrow(M$G)
    M$edges <- list()
    nedges <- 0
    for ( from in 1:n ) {
        for ( to in 1:n ) {
            if ( M$G[to,from] == 0 ) next
            nedges <- nedges + 1
            pa <- which(M$G[to,] == 1)
            fromindex<-which(pa == from)
            opa <- pa[-fromindex]
            labels <- array(0,c(0,length(opa)))
            colnames(labels) <- opa
            if ( length(opa) != 0 ) {
                label<-rep(0,length(opa))
                while (!any(is.na(label)) ) {
                    #CHECK IF LABEL IS IMPLIED BY THE PDAG
                    I <- which(colSums(t(M$CPT[[to]][,-c(fromindex,length(pa)+1),drop=FALSE]) == label) == length(opa))
                    indexes<-M$CPT[[to]][I,length(pa)+1]
                    if ( all( indexes == indexes[1]  ) ) {
                      labels<-rbind(labels,label)
                    }
                    #if ( from== 3 && to == 2 ) browser()
                    label<-next_vector(label,M$arities[opa])
                }
            }
            rownames(labels) <- NULL
            M$edges[[nedges]] <- list(from=from,to=to,labels=labels)
        }
    }
    M$CPT <- NULL
    class(M) <- 'ldag'
    return(M)
}

maximize_ldag <- function(M, verbose = FALSE) {
    #this should be ldag to pdag?
    P <- ldag_to_pdag(M)
    M <- pdag_to_ldag(P)
    return(M)
}

regularize_ldag <- function(M, verbose = FALSE) {
    #regularizing means deleting edges for which the label set includes all labels
    didsomething <- TRUE
    while ( didsomething ) {
        #print.ldag(M)
        didsomething<-FALSE
        for ( i in index(1,length(M$edges)) ) {
            e <- M$edges[[i]]
            pa <- which(M$G[e$to,]==1)
            opa <- setdiff(pa,e$from)
            if (verbose ) cat(nrow(e$labels)," vs.",nvectors(M$arities[opa]),'\n')
            if ( nrow(e$labels) == nvectors(M$arities[opa]) ) {
                if ( verbose ) cat('Delete edge:',e$from,'->',e$to,'\n')
                for ( j in index(1,length(M$edges)) ) { #rermove it from all labels as well
                    if ( j == i ) next
                    e2 <- M$edges[[j]]
                    if ( e2$to == e$to  ) {
                        if ( verbose ) cat('\t Mod label:',e2$from,'->',e2$to,'\n')
                        #delete the column?
                        r <- colnames(e2$labels)
                        #print(e2$labels)
                        e2$labels <- e2$labels[,-which(r==e$from),drop=FALSE]
                        e2$labels <- removeDuplicateLabels(e2$labels)
                        #print(e2$labels)
                        M$edges[[j]] <- e2
                    }
                }
                M$G[e$to,e$from] <- 0
                didsomething <- TRUE
                M$edges[[i]] <- NULL
                break
            }
        }
    }
    return(M)
}

removeDuplicateLabels <- function(labels) {
    duplicate <- rep(FALSE, nrow(labels))
    for ( i in 1:nrow(labels)) {
        for ( j in index(1,i-1)) { #check lower indexed lables
            if ( all(labels[i,] == labels[j,]) ) {
                duplicate[i]<-TRUE
                break
            }
            if ( duplicate[i]) break
        }
    }
    labels[!duplicate,,drop=FALSE]
}



random_ldag <- function(n, pedge = 0.5, plabel = 0.5, pnolab = 0.5, type = "binary") {
    M <- list()
    class(M) <- "ldag"
    M$X <- 0
    M$Y <- n
    while ( TRUE ) {
        M$G <- array(0, c(n, n))
        M$G[lower.tri(M$G)] <- sample(c(0, 1), sum(lower.tri(M$G)), replace = TRUE, prob = c(1 - pedge, pedge))
        if ( any(rowSums(M$G) == 0 & colSums(M$G) == 0) ) next #all nodes must be connected by at least an arc
        G <- M$G
        diag(G) <- 1
        for ( i in 1:n ) G <- G %*% G
        linked <- G[n,1:(n-1)] > 0
        if ( !all(linked) ) next
        else M$X <- sample(which(linked), 1)
        break
        # if (length(linked) == 1) M$X <- linked
    }

    M$names <- as.character(1:n)
    M$names[c(M$X,M$Y)] <- c("X", "Y")

    if ( type == "binary" || type == "bin") {
        M$arities<-rep(2,n)
    } else if ( type == "trinary" || type == "tri") {
        M$arities<-rep(3,n)
    } else if ( type=="ternary" || type == "ter" ) {
        M$arities<-rep(4,n)
    } else {
        M$arities<-sample(c(2,3,4),n,replace=TRUE)
    }
    
    #now we need to save the labels
    M$edges <- list()
    nedges <- 0
    for ( from in 1:n ) {
        for ( to in 1:n ) {
            if ( M$G[to,from] == 0 ) next
            nedges <- nedges + 1
            opa <- setdiff(which(M$G[to, ] == 1), from)
            labels <- matrix(0, 0, length(opa))
            colnames(labels) <- opa
            n_opa <- length(opa)
            if ( n_opa != 0 ) {
                label <- rep(0, n_opa)
                if ( sample(c(FALSE, TRUE), 1, prob = c(pnolab, 1 - pnolab)) ) {
                    while ( !any(is.na(label)) ) {
                        if ( sample(c(FALSE, TRUE), 1, prob = c(1 - plabel, plabel)) ) {
                            labels <- rbind(labels, label)
                        }
                        label <- next_vector(label, M$arities[opa])
                    }
                }
                # if (nrow(labels) == 2^n_opa) {
                #     labels <- labels[-sample.int(nrow(labels),1), ,drop = FALSE]
                # }
            }
            rownames(labels) <- NULL
            M$edges[[nedges]]<- list(from = from, to = to, labels = labels)
        }
    }
    return(M)
}

latent_projection <- function(G, l) {
    for (i in 1:length(l)) {
        e <- igraph::E(G)
        v <- igraph::get.vertex.attribute(G, "name")
        inc.edges <- e[to(l[i]) & (is.na(description) | description != "U")]
        out.edges <- e[from(l[i]) & (is.na(description) | description != "U")]
        unobs.edges <- e[to(l[i]) & description == "U" & !is.na(description)]
        inc.ind <- igraph::get.edges(G, inc.edges)[ ,1]
        out.ind <- igraph::get.edges(G, out.edges)[ ,2]
        unobs.ind <- setdiff(igraph::get.edges(G, unobs.edges)[ ,1], out.ind)
        inc.len <- length(inc.ind)
        out.len <- length(out.ind)
        unobs.len <- length(unobs.ind)
        if (inc.len > 0 & out.len > 0) {
            obs.new <- t(as.matrix(expand.grid(inc.ind, out.ind)))
            G <- G + igraph::edges(v[c(obs.new)], description = rep(NA, ncol(obs.new))) # replace path v_1 -> L -> v_2 with v_1 -> v_2
        }
        if (out.len > 1) {
            unobs.new <- combn(out.ind, 2)
            G <- G + igraph::edges(v[c(unobs.new, unobs.new[2:1, ])], description = rep("U", 2 * ncol(unobs.new))) # replace path v_1 <- L -> v_2 with v_1 <-> v_2
        }
        if (unobs.len > 0 & out.len > 0) {
            unobs.old <- t(as.matrix(expand.grid(unobs.ind, out.ind)))
            G <- G + igraph::edges(v[c(unobs.old, unobs.old[2:1, ])], description = rep("U", 2 * ncol(unobs.old))) # replace path v_1 <-> L -> v_2 with v_1 <-> v_2
        }
        G <- igraph::induced.subgraph(G, setdiff(v, l[i]))
        e.dat <- as.data.frame(igraph::get.edges(G, igraph::E(G)))
        e.dat[ ,3] <- igraph::edge.attributes(G)
        G <- igraph::subgraph.edges(G, which(!duplicated(e.dat)), delete.vertices = FALSE)
    }
    return(G)
}

M_to_str <- function(M, add_ivars = TRUE, latent = c()) {
    di <- ""
    nr <- nrow(M$G)
    intv <- list()
    for ( x in 1:nr ) {
        if ( add_ivars ) {
            if (!latent[x] && x != M$Y) {
                ivar <- paste("I_", M$names[x], sep = "")
                di <- paste(di, ivar, "->", M$names[x], "\n", sep = " ")
                intv[[M$names[x]]] <- paste(ivar, "=", "1")
            }
        }
        for ( y in 1:nr ) {
            if ( M$G[x,y] ) {
                di <- paste(di, M$names[y], "->", M$names[x],  sep = " ")
                labs <- character()
                for (edge in M$edges) {
                    if (x == edge$to && y == edge$from) {
                        v <- M$names[as.numeric(colnames(edge$labels))]
                        if (!is.null(intv[[M$names[x]]])) labs <- c(labs, intv[[M$names[x]]])
                        if ((nrl <- nrow(edge$labels)) > 0) {
                            for (i in 1:nrl) {
                                a <- edge$labels[i, ]
                                labs <- c(labs, paste(v, a, sep = " = ", collapse = ", "))
                            }
                        }
                        break
                    }
                }
                if (length(labs) > 0) {
                    di <- paste(di, ":", paste(labs, collapse = "; "), "\n")
                } else {
                    di <- paste(di, "\n")
                }
            }
        }
    }
    return(di)
}

M_to_igraph <- function(M, latent) {
    n <- nrow(M$G)
    edges <- c()
    ei <-0
    for ( i in 1:n ) {
        for ( j in 1:n ) {
            if( M$G[j,i] == 1 ) {
                edges <- c(edges, i, j)
                ei <- ei + 1
            }
        }
    }
    g <- igraph::make_graph(edges, n = n, directed = TRUE)
    v <- as.character(1:n)
    v[c(M$X,M$Y)] <- c("X", "Y")
    g <- igraph::set.vertex.attribute(g, "name", value = v)
    g <- igraph::set.edge.attribute(g, "description", 1:length(igraph::E(g)), NA)
    if (any(latent == 1)) {
        g <- latent_projection(g, v[latent])
    }
    return(g)
}

sim <- function(confs = NULL, nsim = 10, infer_labels = TRUE, control = list(cache = TRUE)) {
    if (is.null(confs)) stop("No configurations for simulation")
    res_con <- vector(mode = "list", length = nrow(confs))
    res_full <- vector(mode = "list", length = nrow(confs))
    control$benchmark <- TRUE
    control$formula <- FALSE
    for (i in 1:nrow(confs)) {
        res_con[[i]] <- vector(mode = "list", length = nsim)
        res_full[[i]] <- vector(mode = "list", length = nsim)
        m <- confs[i,"n"]
        deg <- confs[i,"degree"]
        total <- (m * m - m)/2
        pedge <- m * deg / (2 * total)
        for (j in 1:nsim) {
            M <- random_ldag(m, pedge, confs[i,"plabel"], pnolab = 0, "binary")
            if (infer_labels) {
                M <- maximize_ldag(M)
                M <- regularize_ldag(M)
            }
            if (confs[i,"latent"] > m - 2) stop("Too many latent variables. ")
            lat <- sample((1:m)[-c(M$X,M$Y)], confs[i,"latent"])
            lat_ind <- rep(FALSE, m)
            lat_ind[lat] <- TRUE
            vars <- paste(M$names[!lat_ind], collapse = ",")
            data <- paste("P(", vars, ")", collapse = "", sep = "")
            query <- "P(Y|X,I_X=1)"
            graph <- M_to_str(M, add_ivars = TRUE, latent = lat_ind)
            g <- M_to_igraph(M, latent = lat)
            cat("Index: i = ", i, " j = ", j, "\n")
            r_con <- list(
                data = data, query = query, graph = graph,
                res = get_derivation(data, query, graph, control = control, context_only = FALSE)
            )
            r_full <- list(
                data = data, query = query, graph = graph,
                res = get_derivation(data, query, graph, control = control, context_only = TRUE)
            )
            res_con[[i]][[j]] <- r_con
            res_full[[i]][[j]] <- r_full
        }
    }
    return(list(res_con = res_con, res_full = res_full))
}

run <- function(nsim = 10, n = 5, seed = NULL) {
    if (!is.null(seed)) set.seed(seed)
    confs <- expand.grid(degree = 3, plabel = 0.5, latent = 2, n = n)
    colnames(confs) <- c("degree", "plabel", "latent", "n")
    return(sim(confs = confs, nsim = nsim, control = list(cache = TRUE)))
}