import numpy as np
import torch
from scipy import linalg
from torch.nn.functional import adaptive_avg_pool2d
from core.inception import InceptionV3


def get_activations(dataloader, model, device, verbose=False):
    """
        :param dataloader: (n_images, 3, height, width) from -1 to 1
        :param model: inception model
        :param device: cpu or indexed cuda
        :return:
    """
    dims = 2048
    model.eval()
    n_imgs = len(dataloader.dataset)
    n_batches = len(dataloader)

    pred_arr = np.empty((n_imgs, dims))
    start_pointer = 0

    for i, batch in enumerate(dataloader, 0):
        if verbose:
            print('\rPropagating batch %d/%d' % (i + 1, n_batches), flush=True)
        sample = batch[0].to(device)
        batch_size_i = sample.size(0)
        end_pointer = start_pointer + batch_size_i

        pred = model(sample)[0]
        pred_arr[start_pointer:end_pointer] = pred.cpu().data.numpy().reshape(batch_size_i, -1)
        start_pointer += batch_size_i

    return pred_arr


def calculate_frechet_distance(mu1, sigma1, mu2, sigma2, silent, eps=1e-6, full=False):
    """Numpy implementation of the Frechet Distance.
    The Frechet distance between two multivariate Gaussians X_1 ~ N(mu_1, C_1)
    and X_2 ~ N(mu_2, C_2) is
            d^2 = ||mu_1 - mu_2||^2 + Tr(C_1 + C_2 - 2*sqrt(C_1*C_2)).

    Stable version by Dougal J. Sutherland.

    Params:
    -- mu1   : Numpy array containing the activations of a layer of the
               inception net (like returned by the function 'get_predictions')
               for generated samples.
    -- mu2   : The sample mean over activations, precalculated on an
               representative data set.
    -- sigma1: The covariance matrix over activations for generated samples.
    -- sigma2: The covariance matrix over activations, precalculated on an
               representative data set.

    Returns:
    --   : The Frechet Distance.
    """

    mu1 = np.atleast_1d(mu1)
    mu2 = np.atleast_1d(mu2)

    sigma1 = np.atleast_2d(sigma1)
    sigma2 = np.atleast_2d(sigma2)

    assert mu1.shape == mu2.shape, \
        'Training and test mean vectors have different lengths'
    assert sigma1.shape == sigma2.shape, \
        'Training and test covariances have different dimensions'

    diff = mu1 - mu2
    if full:
        if not silent:
            print('full')
        # Product might be almost singular
        covmean, _ = linalg.sqrtm(sigma1.dot(sigma2), disp=False)
        if not np.isfinite(covmean).all():
            msg = ('fid calculation produces singular product; '
                   'adding %s to diagonal of cov estimates') % eps
            print(msg)
            offset = np.eye(sigma1.shape[0]) * eps
            covmean = linalg.sqrtm((sigma1 + offset).dot(sigma2 + offset))

        # Numerical error might give slight imaginary component
        if np.iscomplexobj(covmean):
            if not np.allclose(np.diagonal(covmean).imag, 0, atol=1e-3):
                m = np.max(np.abs(covmean.imag))
                raise ValueError('Imaginary component {}'.format(m))
            covmean = covmean.real
        tr_covmean = np.trace(covmean)
    else:
        if not silent:
            print('diag approx')
        sigma1_diag = np.diag(sigma1)
        sigma2_diag = np.diag(sigma2)
        covmean = np.sqrt(sigma1_diag * sigma2_diag)
        tr_covmean = np.sum(covmean)

    fid_mu = diff.dot(diff)
    fid_cov = np.trace(sigma1) + np.trace(sigma2) - 2 * tr_covmean
    fid = fid_mu + fid_cov

    return fid, fid_mu, fid_cov


def calculate_activation_statistics(dataloader, model, device):
    act = get_activations(dataloader, model, device)
    mu = np.mean(act, axis=0)
    sigma = np.cov(act, rowvar=False)

    return mu, sigma


def load_statistics_of_path(path):
    m, s = None, None
    if path.endswith('.npz'):
        f = np.load(path)
        m, s = f['mu'][:], f['sigma'][:]
        f.close()

    return m, s


def calculate_fid(paths, dataloaders, device, full=False, silent=False):
    dims = 2048
    block_idx = InceptionV3.BLOCK_INDEX_BY_DIM[dims]
    model = InceptionV3([block_idx], normalize_input=False)
    model.to(device)

    if paths[0].endswith('.npz'):
        m1, s1 = load_statistics_of_path(paths[0])
    else:
        m1, s1 = calculate_activation_statistics(dataloaders[0], model, device)
    if not silent:
        print('stat 1 done')

    if paths[1].endswith('.npz'):
        m2, s2 = load_statistics_of_path(paths[1])
    else:
        m2, s2 = calculate_activation_statistics(dataloaders[1], model, device)
    if not silent:
        print('stat 2 done')

    if not silent:
        print(full)
    fid_value = calculate_frechet_distance(m1, s1, m2, s2, silent=silent, full=full)

    np.savez('path0_act.npz', mu=m1, sigma=s1)
    np.savez('path1_act.npz', mu=m2, sigma=s2)

    return fid_value
