#ifndef DISTRO_H_INCLUDED
#define DISTRO_H_INCLUDED

#include <random>
#include <vector>

// create a well-seeded random number generator
std::mt19937 well_seeded_mt19937(unsigned int seed);

// custom random distribution (abstract class)
class distro {
public:
    virtual ~distro(){};
    virtual double extract_num(std::mt19937& ran_gen) = 0;
    virtual std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n) = 0;
};

// uniform distribution on a given interval
class distro_uniform: public distro {
public:
    double min_val;
    double max_val;

    distro_uniform(double min_val, double max_val);
    double extract_num(std::mt19937& ran_gen);
    std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n);
};

// beta distribution with given prior
class distro_beta: public distro {
public:
    double alpha;
    double beta;

    distro_beta(double alpha, double beta);
    double extract_num(std::mt19937& ran_gen);
    std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n);
};

// dirac's delta distribution with given bernoulli parameter
class distro_dirac: public distro {
public:
    double p;

    distro_dirac(double p);
    double extract_num(std::mt19937& ran_gen);
    std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n);
};

// kronecker's delta distribution on the integers
class distro_kronecker: public distro {
public:
    int val;

    distro_kronecker(int val);
    double extract_num(std::mt19937& ran_gen);
    std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n);
};

// power law distribution
class distro_power: public distro {
private:
    double p_min;
    double p_max;
    double d_pow;
public:
    double v_min;
    double v_max;
    double c_pow;

    distro_power(double v_min, double v_max, double c_pow);
    double extract_num(std::mt19937& ran_gen);
    std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n);
};

// exponential distribution
class distro_exponential: public distro {
public:
    double v_min;
    double lambda;

    distro_exponential(double v_min, double lambda);
    double extract_num(std::mt19937& ran_gen);
    std::vector<double> extract_vec(std::mt19937& ran_gen, unsigned int n);
};

// distro parser
distro* distro_parse(int *argc, char **argv[]);

// home-made unit tests
void distro_uniform_test();
void distro_beta_test();
void distro_dirac_test();
void distro_kronecker_test();
void distro_power_test();
void distro_exponential_test();
void distro_parse_test();

#endif // DISTRO_H_INCLUDED
