% Copyright (C) 2009  Arno Onken
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

% -------------------------------------------------------------------------
% Cumulative distribution function of the Frank copula.
%
% Arguments:
%  x     - n-by-d matrix of random samples
%  theta - Scalar parameter of the Frank copula
%
% Returns:
%  c     - n-by-1 vector of cumulative probabilties
% -------------------------------------------------------------------------
function c = frankcdf (x, theta)

    % Check arguments
    if (nargin ~= 2)
        error ('frankcdf: usage c = frankcdf (x, theta)');
    end

    if (size (x, 2) ~= 2)
        error('frankcdf: x must have two columns');
    end

    if (~isscalar (theta))
        error('frankcdf: theta must be a scalar');
    end

    if (theta == 0)
        % Independence
        c = prod (x, 2);
    else
        % CDF of the Frank copula family
        c = -log (1 + (prod (expm1 (-theta .* x), 2)) ./ (expm1 (-theta))) ./ theta;
    end
end