% Copyright (C) 2009  Arno Onken
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

% -------------------------------------------------------------------------
% Calculates the probability mass function for samples <X> of a Frank
% copula family based distribution with discrete margins. This function has
% exponential complexity in the number of elements.
%
% Arguments:
%  X         - Discrete sample
%  margincdf - CDF of the distribution margins
%  theta     - Copula parameter
%
% Returns:
%  p         - Probabilitiy of X
% -------------------------------------------------------------------------
function p = dmfrankpdf (X, margincdf, theta)

    % Check arguments
    if (nargin ~= 3)
        error ('dmfrankpdf: usage p = dmfrankpdf (X, margincdf, theta)');
    end

    if (~isscalar (theta))
        error ('dmfrankpdf: theta must be a scalar');
    end

    X = X';
    [n, trials] = size (X);

    % All binary combinations in a (2^n)-by-n matrix
    bcomb = floor (mod (((0:(2 ^ n - 1))' * 2 .^ ((1 - n):0)), 2));

    p = zeros (trials, 1);

    for itrials = 1:trials
        x = X(:, itrials)';

        % Apply the inclusion-exclusion principle
        % - but only to the subset of elements ~= 0
        nz = find (x);
        nnz = length (nz);
        if (nnz > 0)
            % Signs of the terms
            signs = -ones (2 ^ nnz, 1);
            signs(mod (sum (bcomb(1:(2^nnz), (n-nnz+1):n), 2), 2) == 0) = 1;

            % Compute cdf of all sub-parts
            x_sub = zeros (2 ^ nnz, n);
            x_sub(:, nz) = bcomb(1:(2^nnz), (n-nnz+1):n);

            csample = margincdf (repmat (x, 2 ^ nnz, 1) - x_sub);

            subpart = frankcdf (csample, theta);
            p(itrials) = sum (signs .* subpart);
        else
            csample = margincdf (x);
            p(itrials) = frankcdf (csample, theta);
        end
    end
end