function E = initE(d,nose,k1,k2,M,L)
%initE:      initialize swimmer
%   E = initE(d,nose,k1,k2) creates a structure E with various parameters
%   related to a swimmer. Some parameters are passed to the function while
%   some are set internally. 
%   d      - The number of links the swimmer is composed of.
%   nose   - The location of the nose point on the swimmer's body. Valid
%            values are nose = 1..d+1 where 1 and d+1 correspond to the
%            two endpoints.
%   k1     - The drag coefficient for velocity in the direction
%            perpendicular to a given stick
%   k2     - The drag coefficient for velocity in the direction
%            parallel to a given stick. 
%            k2 should always be smaller than k1.
%   M      - An optional size d vector with the masses of the links.
%   L      - An optional size d vector with the lengths of the links.
%   I      - A matrix with the moments of intertia on the diagonal.
%   P      - Related to the transformation between cartesian and angular
%            variables, see the derivation of the swimmer equations for
%            details.  
%   U      - Transforms the internal angle torques to opposing torques on
%            each link, preserving linear and angular momentum modulo the
%            drag forces. 
%   Angles - a logical vector indicating (surprise!) the angular variables.
%            It is used for distance functions etc. 
%
%   Author: Yuval Tassa

   E.d      = d;
   E.k1     = k1;
   E.k2     = k2;
   E.nose   = nose;
   if nargin > 4
      E.M      = diag(M);
   else
      E.M      = eye(E.d);       
   end
   if nargin > 5
      E.L      = diag(L);
   else
      E.L      = eye(E.d);       
   end
   E.I      = 1/12*E.M*E.L^2;
   Q        = full(gallery('tridiag',E.d,0,-1,1));
   Q(end,:) = diag(E.M)';
   A        = full(gallery('tridiag',E.d,0,1,1));
   A(d,d)   = 0;
   E.P      = 0.5*Q^-1*A*E.L; 
   E.U      = full(gallery('tridiag',E.d,-1,1,0));
   E.U      = E.U(:,1:end-1);
   E.angles = logical([0 0 ones(1,E.d-1) zeros(1,E.d+2)]);   

   function internalize()
   end
   % Internalize has been put in its own .m file and is kept here in
   % this degenerate form for compatibillity with legacy code. 
end